
//   vim:tw=110:ts=4:
/**************************************************************************************************************
*
* FILE   :	HCF.C
*
* DATE    :	$Date: 2003/06/05 15:21:48 $   $Revision: 1.2 $
*
* AUTHOR :	Nico Valster
*
* DESC   :	HCF Routines (callable via the Wireless Connection I/F or WCI)
*			Local Support Routines for above procedures
*
*			Customizable via HCFCFG.H, which is included by HCF.H
*
***************************************************************************************************************
* COPYRIGHT (c) 1994 - 1995	by AT&T.	 			All Rights Reserved
* COPYRIGHT (c) 1996 - 2000 by Lucent Technologies.	All Rights Reserved
* COPYRIGHT (c) 2001 - 2003	by Agere Systems.		All Rights Reserved
*
* SOFTWARE LICENSE
*
* This software is provided subject to the following terms and conditions,
* which you should read carefully before using the software.  Using this
* software indicates your acceptance of these terms and conditions.  If you do
* not agree with these terms and conditions, do not use the software.
*
* Copyright  2003 Agere Systems Inc.
* All rights reserved.
*
* Redistribution and use in source or binary forms, with or without
* modifications, are permitted provided that the following conditions are met:
*
* . Redistributions of source code must retain the above copyright notice, this
*    list of conditions and the following Disclaimer as comments in the code as
*    well as in the documentation and/or other materials provided with the
*    distribution.
* 
* . Redistributions in binary form must reproduce the above copyright notice,
*    this list of conditions and the following Disclaimer in the documentation
*    and/or other materials provided with the distribution.
* 
* . Neither the name of Agere Systems Inc. nor the names of the contributors
*    may be used to endorse or promote products derived from this software
*    without specific prior written permission.
*
* Disclaimer
*
* THIS SOFTWARE IS PROVIDED AS IS AND ANY EXPRESS OR IMPLIED WARRANTIES,
* INCLUDING, BUT NOT LIMITED TO, INFRINGEMENT AND THE IMPLIED WARRANTIES OF
* MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  ANY
* USE, MODIFICATION OR DISTRIBUTION OF THIS SOFTWARE IS SOLELY AT THE USERS OWN
* RISK. IN NO EVENT SHALL AGERE SYSTEMS INC. OR CONTRIBUTORS BE LIABLE FOR ANY
* DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
* (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
* LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
* ON ANY THEORY OF LIABILITY, INCLUDING, BUT NOT LIMITED TO, CONTRACT, STRICT 
* LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
* OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
* DAMAGE.
*
**************************************************************************************************************/

/**************************************************************************************************************

*** Questions ***
 - 	Is IFB_LinkStat needed these days (HM?)
*
* ToDo
*
 1:	For all/most functions, update "MSF-accessible fields of Result Block:" entry
 2: Use the "numbered comments" in the NARRATIVE consistently, i.e. hcf_put_info
 5:	review/walktrough potential problems with re-entrancy when HCF_ACT_INT_OFF is called from task level
	and being interrupted

*
* Implementation Notes
*
 -	a leading marker of //! is used. The purpose of such a sequence is to help to understand the flow
 	An example is:	//!rc = HCF_SUCCESS;
	if this is superfluous because rc is already guaranteed to be 0 but it shows to the (maintenance)
	programmer it is an intentional omission at the place where someone could consider it most appropriate at
	first glance
 -	using near pointers in a model where ss!=ds is an invitation for disaster, so be aware of how you specify
 	your model and how you define variables which are used at interrupt time
 -	remember that sign extension on 32 bit platforms may cause problems unless code is carefully constructed,
 	e.g. use "(hcf_16)~foo" rather than "~foo"

*************************************************************************************************************/

#include "hcf.h"				// HCF and MSF common include file
#include "hcfdef.h"				// HCF specific include file
#include "mmd.h"				// MoreModularDriver common include file
#if (HCF_TYPE) & HCF_TYPE_USB
#include "..\wl_usb\unuhcf.h"	// USB specific include file
wci_recordp usb_simp;			// "simulation" pointer for USB
#endif // HCF_TYPE_USB


/*************************************************************************************************************/
/***************************************  PROTOTYPES  ********************************************************/
/*************************************************************************************************************/
STATIC void			assert(IFBP ifbp, unsigned int line_number, int q );
STATIC hcf_16		check_comp( IFBP ifbp, CFG_RANGES_STRCT *p, hcf_16 type );
STATIC int			cmd_exe( IFBP ifbp, hcf_16 cmd_code, int par_0 );
STATIC int			debug_trigger(IFBP ifbp, hcf_16 where, hcf_16 what );
STATIC int			init( IFBP ifbp );
STATIC int			put_info( IFBP ifbp, LTVP ltvp	);
STATIC void 		send_msg_frag( IFBP ifbp, wci_bufp bufp, int len );
STATIC void 		send_mic_dcp( IFBP ifbp );
STATIC int 			strio( IFBP, hcf_16 type, hcf_16 fid, hcf_16 offset, wci_recordp pc_addr, \
								  int len BE_PAR(int wlen) ); 
STATIC void 		update_mic( hcf_32* p, hcf_32 M );
STATIC void 		update_mic_dummy( hcf_32* p, hcf_32 M );

#if (HCF_TYPE) & HCF_TYPE_USB
int					get_info_usb(IFBP ifbp, LTVP ltvp );
STATIC hcf_16		put_usb_nic_cmd( IFBP ifbp, hcf_16 cmd, hcf_16 par0, hcf_16 par1, hcf_16 par2);
#else
#if (HCF_TYPE) & HCF_TYPE_HII
STATIC void			ack_the_bastard( IFBP ifbp, hcf_16 mask );
#else
STATIC int			aux_cntl( IFBP ifbp, hcf_16 cmd );
#endif // HCF_TYPE_HII
STATIC void			calibrate( IFBP ifbp );
STATIC int			check_mic( IFBP ifbp );
STATIC int			cmd_cmpl( IFBP ifbp );
STATIC hcf_16		get_fid( IFBP ifbp );
STATIC void	 		isr_info( IFBP ifbp );
STATIC void 		rcv_msg_frag( IFBP ifbp, wci_bufp bufp, int len );
#if defined HCF_DLV //;?currently not supported || defined HCF_DLNV
STATIC int			download(IFBP ifbp, LTVP ltvp );
#endif // HCF_DLV / HCF_DLNV
#if defined HCF_MB_ON
STATIC void			get_info_mb( IFBP ifbp, LTVP ltvp );
STATIC void			put_info_mb( IFBP ifbp, hcf_16 type, wci_recordp bufp, hcf_16 len );
#endif // HCF_MB_ON
#endif // HCF_TYPE_USB

#if HCF_ENCAPSULATION == 0x0001
STATIC hcf_8	 	hcf_encap( wci_bufp type );
#endif // HCF_ENCAPSULATION


/**************************************************************************************************************
******************************* D A T A    D E F I N I T I O N S **********************************************
**************************************************************************************************************/

#if defined HCF_ASSERT       
#define THIS__FILE__ "hcf"

static struct {
	hcf_16	len;					//length of assert_strct
	hcf_16	trace;					//trace log copy from IFB
	hcf_16	qualifier;				//qualifier from entry parameter
	hcf_16	line_number;			//line number from entry parameter
	TCHAR val[sizeof(THIS__FILE__)];
	TCHAR align;					//padding which MAY be needed depending on length file-name
} BASED assert_strct = { (sizeof(assert_strct)+1)/sizeof(hcf_16) - 1, 0, 0, 0, TEXT(THIS__FILE__), TEXT('\0') };

#endif // HCF_ASSERT

#if HCF_ENCAPSULATION == 0x0001
/* SNAP header to be inserted in Ethernet-II frames */
STATIC  hcf_8 BASED snap_header[] = { 0xAA, 0xAA, 0x03, 0x00, 0x00,	//5 bytes signature +
									  0 };							//1 byte protocol identifier
#endif // HCF_ENCAPSULATION

STATIC hcf_8 BASED mic_pad[] = { 0x5A, 0, 0, 0, 0, 0, 0, 0 };		//MIC padding of message

#if defined MSF_COMPONENT_ID
CFG_IDENTITY_STRCT BASED cfg_drv_identity = {
	sizeof(cfg_drv_identity)/sizeof(hcf_16) - 1,	//length of RID
	CFG_DRV_IDENTITY,			// (0x0826)
	MSF_COMPONENT_ID,
	MSF_COMPONENT_VAR,
	MSF_COMPONENT_MAJOR_VER,
	MSF_COMPONENT_MINOR_VER
} ;

CFG_RANGES_STRCT BASED cfg_drv_sup_range = {
	sizeof(cfg_drv_sup_range)/sizeof(hcf_16) - 1,	//length of RID
	CFG_DRV_SUP_RANGE,			// (0x0827)

	COMP_ROLE_SUPL,
	COMP_ID_DUI,
	{	DUI_COMPAT_VAR,
		DUI_COMPAT_BOT,
		DUI_COMPAT_TOP
	}
} ;

struct CFG_RANGE3_STRCT BASED cfg_drv_act_ranges_pri = {
	sizeof(cfg_drv_act_ranges_pri)/sizeof(hcf_16) - 1,	//length of RID
	CFG_DRV_ACT_RANGES_PRI,		// (0x0828)

	COMP_ROLE_ACT,
	COMP_ID_PRI,
	{
#if defined HCF_PRI_VAR_1
	 {	1, 									//variant[1] - Variant number
		CFG_DRV_ACT_RANGES_PRI_1_BOTTOM,    	//           - Bottom Compatibility
		CFG_DRV_ACT_RANGES_PRI_1_TOP 	        //           - Top Compatibility
	 },
#else // HCF_PRI_VAR_1
	 { 0, 0, 0 },
#endif // HCF_PRI_VAR_1
#if defined HCF_PRI_VAR_2
	 {	2, 									//variant[2] - Variant number
		CFG_DRV_ACT_RANGES_PRI_2_BOTTOM,		//           - Bottom Compatibility
		CFG_DRV_ACT_RANGES_PRI_2_TOP  	 		//         	 - Top Compatibility
	 },
#else // HCF_PRI_VAR_2
	 { 0, 0, 0 },
#endif // HCF_PRI_VAR_2
#if defined HCF_PRI_VAR_3
	 {	3, 									//variant[2] - Variant number
		CFG_DRV_ACT_RANGES_PRI_3_BOTTOM,		//           - Bottom Compatibility
		CFG_DRV_ACT_RANGES_PRI_3_TOP  	 		//         	 - Top Compatibility
	 }
#else // HCF_PRI_VAR_3
	 { 0, 0, 0 }
#endif // HCF_PRI_VAR_3
	}
} ;


struct CFG_RANGE2_STRCT BASED cfg_drv_act_ranges_sta = {
	sizeof(cfg_drv_act_ranges_sta)/sizeof(hcf_16) - 1,	//length of RID
	CFG_DRV_ACT_RANGES_STA,		// (0x0829)

	COMP_ROLE_ACT,
	COMP_ID_STA,
	{
#if defined HCF_STA_VAR_1
	 {	1, 									//variant[1] - Variant number
		CFG_DRV_ACT_RANGES_STA_1_BOTTOM,    	//           - Bottom Compatibility
		CFG_DRV_ACT_RANGES_STA_1_TOP 	        //           - Top Compatibility
	 },
#else // HCF_STA_VAR_1
	 { 0, 0, 0 },
#endif // HCF_STA_VAR_1
#if defined HCF_STA_VAR_2
	 {	2,                      			//variant[1] - Variant number
		CFG_DRV_ACT_RANGES_STA_2_BOTTOM,		//           - Bottom Compatibility
		CFG_DRV_ACT_RANGES_STA_2_TOP			//           - Top Compatibility
	 }
#else // HCF_STA_VAR_2
	 { 0, 0, 0 }
#endif // HCF_STA_VAR_2
	}
} ;


// !!!!!see note below!!!!!!!!!!
struct CFG_RANGE5_STRCT BASED cfg_drv_act_ranges_hsi = {
	sizeof(cfg_drv_act_ranges_hsi)/sizeof(hcf_16) - 1,	//length of RID
	CFG_DRV_ACT_RANGES_HSI,		// (0x082A)
	COMP_ROLE_ACT,
	COMP_ID_HSI,
	{
#if defined HCF_HSI_VAR_0					// Controlled deployment
	 {	0,     				       			// variant[1] - Variant number
		CFG_DRV_ACT_RANGES_HSI_0_BOTTOM,		//           - Bottom Compatibility	!!!!!see note below!!!!!!!
		CFG_DRV_ACT_RANGES_HSI_0_TOP			//           - Top Compatibility
	 },
#else // HCF_HSI_VAR_0
	 { 0, 0, 0 },				 
#endif // HCF_HSI_VAR_0
#if defined HCF_HSI_VAR_1					// WaveLAN
	 {	1,            						// variant[1] - Variant number
		CFG_DRV_ACT_RANGES_HSI_1_BOTTOM,		//           - Bottom Compatibility	!!!!!see note below!!!!!!!
		CFG_DRV_ACT_RANGES_HSI_1_TOP			//           - Top Compatibility
	 },
#else // HCF_HSI_VAR_1
	 { 0, 0, 0 },
#endif // HCF_HSI_VAR_1
#if defined HCF_HSI_VAR_2					// Mini PCI
	 {	2,            						// variant[1] - Variant number
		CFG_DRV_ACT_RANGES_HSI_2_BOTTOM,		//           - Bottom Compatibility	!!!!!see note below!!!!!!!
		CFG_DRV_ACT_RANGES_HSI_2_TOP			//           - Top Compatibility
	 },
#else // HCF_HSI_VAR_2
	 { 0, 0, 0 },
#endif // HCF_HSI_VAR_2
#if defined HCF_HSI_VAR_3					// Chico Lama
	 {	3,            						// variant[1] - Variant number
		CFG_DRV_ACT_RANGES_HSI_3_BOTTOM,		//           - Bottom Compatibility	!!!!!see note below!!!!!!!
		CFG_DRV_ACT_RANGES_HSI_3_TOP			//           - Top Compatibility
	 },
#else // HCF_HSI_VAR_3
	 { 0, 0, 0 },
#endif // HCF_HSI_VAR_3
#if defined HCF_HSI_VAR_4					// Hermes-II all types
	 {	4,            						// variant[1] - Variant number
		CFG_DRV_ACT_RANGES_HSI_4_BOTTOM,		//           - Bottom Compatibility	!!!!!see note below!!!!!!!
		CFG_DRV_ACT_RANGES_HSI_4_TOP			//           - Top Compatibility
	 }
#else // HCF_HSI_VAR_4
	 { 0, 0, 0 }
#endif // HCF_HSI_VAR_4
	}
} ;

//--#endif // HCF_HII

CFG_RANGE2_STRCT BASED cfg_drv_act_ranges_apf = {
	sizeof(cfg_drv_act_ranges_apf)/sizeof(hcf_16) - 1,	//length of RID
	CFG_DRV_ACT_RANGES_APF,		// (0x082B)

	COMP_ROLE_ACT,
	COMP_ID_APF,
	{
#if defined HCF_APF_VAR_1				//(Fake) Hermes-I
	 {	1,                      			//variant[1] - Variant number
		CFG_DRV_ACT_RANGES_APF_1_BOTTOM,		//           - Bottom Compatibility
		CFG_DRV_ACT_RANGES_APF_1_TOP			//           - Top Compatibility
	 },
#else // HCF_APF_VAR_1
	 { 0, 0, 0 },
#endif // HCF_APF_VAR_1
#if defined HCF_APF_VAR_2				//Hermes-II
	 {	2,            						// variant[1] - Variant number
		CFG_DRV_ACT_RANGES_APF_2_BOTTOM,		//           - Bottom Compatibility	!!!!!see note below!!!!!!!
		CFG_DRV_ACT_RANGES_APF_2_TOP			//           - Top Compatibility
	 }
#else // HCF_APF_VAR_2
	 { 0, 0, 0 }
#endif // HCF_APF_VAR_2
	}
} ;
#define HCF_VERSION  "HCF$Revision: 1.2 $"

static struct /*CFG_HCF_OPT_STRCT*/ {
	hcf_16	len;					//length of cfg_hcf_opt struct
	hcf_16	typ;					//type 0x082C
	hcf_16	v0;						//number of following fields up to Revison number
	hcf_16	v1;						//align
	hcf_16	v2;						//endian
	hcf_16	v3;						//HCF_TYPE
		hcf_16		v4;						//HCF_MAX_LTV
		hcf_16		v5;						//HCF_MAX_MSG
		hcf_16		v6;						//HCF_MAX_NOTIFY
		hcf_16		v7;						//HCF_PORT_IO/HCF_MEM_IO
	hcf_16			v8;						//HCF_PROT_TIME
	TCHAR 	val[sizeof(HCF_VERSION)];
	TCHAR 	align;					//padding which MAY be needed depending on length file-name
} BASED cfg_hcf_opt = {
	sizeof(cfg_hcf_opt)/sizeof(hcf_16) -1,
	CFG_HCF_OPT,				// (0x082C)
	( sizeof(cfg_hcf_opt) - sizeof(HCF_VERSION) +1 )/sizeof(hcf_16) - 4,
	HCF_ALIGN,
#if defined HCF_BIG_ENDIAN		//;?replace ala HCF_TYPE
	1,
#else // HCF_LITTLE_ENDIAN
	0,
#endif // HCF_BIG_ENDIAN
	HCF_TYPE,
	HCF_MAX_LTV,
	HCF_MAX_MSG,
	0,	//HCF_MAX_NOTIFY,
#if defined HCF_MEM_IO
	1,
#else // HCF_PORT_IO
	0,
#endif // HCF_MEM_IO
	HCF_PROT_TIME,
	HCF_VERSION
}; // cfg_hcf_opt


#if (HCF_TYPE) & HCF_TYPE_USB
//						Actor USB Converter board I/F
CFG_RANGE2_STRCT BASED cfg_drv_act_ranges_ubi = {
	sizeof(cfg_drv_act_ranges_ubi)/sizeof(hcf_16) - 1,	//length of RID
	CFG_DRV_ACT_RANGES_UBI,					// (0x0886)

	COMP_ROLE_ACT,
	COMP_ID_UBI,							// (0x19)
	{
	 {	1,                      			//variant[1] - Variant number
		CFG_DRV_ACT_RANGES_UBI_1_BOTTOM,		//           - Bottom Compatibility
		CFG_DRV_ACT_RANGES_UBI_1_TOP			//           - Top Compatibility
	 },
	 {	2,                      			//variant[1] - Variant number
		CFG_DRV_ACT_RANGES_UBI_2_BOTTOM,		//           - Bottom Compatibility
		CFG_DRV_ACT_RANGES_UBI_2_TOP			//           - Top Compatibility
	 }
	}
} ;


//						Actor USB Bootloader I/F
CFG_RANGES_STRCT BASED cfg_drv_act_ranges_udi = {
	sizeof(cfg_drv_act_ranges_udi)/sizeof(hcf_16) - 1,	//length of RID
	CFG_DRV_ACT_RANGES_UDI,					// (0x0885)

	COMP_ROLE_ACT,
	COMP_ID_UDI,							// (0x1A)
	{
	 {	1,                      			//variant[1] - Variant number
		CFG_DRV_ACT_RANGES_UDI_1_BOTTOM,		//           - Bottom Compatibility
		CFG_DRV_ACT_RANGES_UDI_1_TOP			//           - Top Compatibility
	 }
	}
} ;
#endif // HCF_TYPE_USB

/*
	The below table accessed via a computed index was the original implementation for hcf_get_info with
	CFG_DRV_IDENTITY, CFG_DRV_SUP_RANGE, CFG_DRV_ACT_RANGES_PRI, CFG_DRV_ACT_RANGES_STA, CFG_DRV_ACT_RANGES_HSI
	as type. However it was reported that the 68K compiler for MAC OS is unable to initialize pointers.
	Accepting this story at face value, the HCF is coded around this problem by implementing a direct access..
	To save part of the invested effort, the original table is kept as comment.
*/
STATIC LTV_STRCT*   BASED xxxx[ ] = {
	(LTV_STRCT*)&cfg_drv_identity,	      	//CFG_DRV_IDENTITY              0x0826
	(LTV_STRCT*)&cfg_drv_sup_range, 	    //CFG_DRV_SUP_RANGE             0x0827
	(LTV_STRCT*)&cfg_drv_act_ranges_pri,	//CFG_DRV_ACT_RANGES_PRI        0x0828
	(LTV_STRCT*)&cfg_drv_act_ranges_sta,  	//CFG_DRV_ACT_RANGES_STA      	0x0829
	(LTV_STRCT*)&cfg_drv_act_ranges_hsi,	//CFG_DRV_ACT_RANGES_HSI		0x082A
	(LTV_STRCT*)&cfg_drv_act_ranges_apf,	//CFG_DRV_ACT_RANGES_APF		0x082B
	(LTV_STRCT*)&cfg_hcf_opt,				//CFG_HCF_OPT					0x082C
  };

#if (HCF_TYPE) & HCF_TYPE_USB
STATIC LTV_STRCT*   BASED xuuu[ ] = {
	(LTV_STRCT*)&cfg_drv_act_ranges_udi,	//CFG_DRV_ACT_RANGES_UDI		0x0885
	(LTV_STRCT*)&cfg_drv_act_ranges_ubi		//CFG_DRV_ACT_RANGES_UBI		0x0886
  };
#endif // HCF_TYPE_USB
#endif // MSF_COMPONENT_ID


/**************************************************************************************************************
************************** T O P   L E V E L   H C F   R O U T I N E S ****************************************
**************************************************************************************************************/


/*******************************************************************************************************************

.MODULE			hcf_action
.DESCRIPTION	Changes the run-time Card behavior

.ARGUMENTS
  int hcf_action( IFBP ifbp, hcf_16 action )

.RETURNS
	HCF_SUCCESS 			all (including invalid)
	IFB_IntOffCnt     		HCF_ACT_INT_FORCE_ON, HCF_ACT_INT_ON
	HCF_INT_PENDING			HCF_ACT_INT_OFF, interrupt pending
	HCF_ERR_NO_NIC			HCF_ACT_INT_OFF

.NARRATIVE

 Parameters:
  ifbp		address of the Interface Block
  action	number identifying the type of change

  o HCF_ACT_INT_ON			enable interrupt generation by WaveLAN NIC
  o HCF_ACT_INT_OFF			disable interrupt generation by WaveLAN NIC
  o HCF_ACT_INT_FORCE_ON	MSF reported Card insertion (if HCF_CARD_CHECK_ON) / Card Initialization
  o HCF_ACT_RX_ACK			MSF reported Card removal (if HCF_CARD_CHECK_ON)
  o HCF_ACT_TALLIES			Hermes Inquire Tallies (F100) command
  o HCF_ACT_SCAN			Hermes Inquire Scan (F101) command
  o HCF_ACT_ACS_SCAN		Hermes Automatic Channel Select Scan (F102) command

 Returns:
  o	HCF_ACT_INT_OFF
		0: no interrupt pending
		1: interrupt pending
		   HCF_ERR_NO_NIC
  o HCF_ACT_INT_FORCE_ON, HCF_ACT_INT_ON
		as an aid during debugging, IFB_IntOffCnt is reported
  o	all other
		0 

 Remarks:
  o	HCF_ACT_INT_OFF/HCF_ACT_INT_ON codes may be nested but must be balanced. The INT_OFF/INT_ON 
  	housekeeping is initialized at 0x0000 by hcf_connect, causing the interrupt generation mechanism to be 
	disabled at first. This suits MSF implementation based on a polling strategy. An MSF
	based on a interrupt strategy must call hcf_action with INT_ON in its initialization logic.

.DIAGRAM
 0:	The assert embedded in HCFLOGENTRY checks against re-entrancy. Re-entrancy could be caused by a MSF
 	logic at task-level calling hcf_functions without shielding with HCF_ACT_ON/_OFF. However the
	HCF_ACT_INT_OFF action itself can per definition not be protected this way. Based on code inspection,
	it can be concluded, that there is no re-entrancy PROBLEM in this particular flow. It does not seem
	worth the trouble to explicitly check for this condition (although there was a report of an MSF which
	ran into this assert.
 2: IFB_IntOffCnt is used to balance the INT_OFF and INT_ON calls.
 	Disabling of the interrupts is achieved by writing a zero to the Hermes IntEn register.
 	In a shared interrupt environment (e.g. the mini-PCI NDIS driver) it is considered more correct
	to return the status HCF_INT_PENDING if and only if, the current invocation of hcf_service_nic is
 	(apparently) called in the ISR when the ISR was activated as result of a change in HREG_EV_STAT matching
	a bit in HREG_INT_EN, i.e. not if invoked as result of another device generating an interrupt on the shared
 	interrupt line.
 	Note 1: it has been observed that under certain adverse conditions on certain platforms the writing of
 	HREG_INT_EN can apparently fail, therefor it is paramount that HREG_INT_EN is written again with 0 for
	each and every call to HCF_ACT_INT_OFF.
	Note 2: it has been observed that under certain H/W & S/W architectures this logic is called when there
	is no NIC at all. To cater for this, the value of HREG_INT_EN is validated. If the unused bit 0x0100
	is set, it is assumed there is no NIC. since this kludge is created to protect against some error 
	condtions reported in the Windows OS and associated system software, the card presence check should not 
	depend on HCF_CARD_CHECK (a macro facility which may become outmoded soon anyway)
	Note 3: During the download process, some versions of the F/W reset HREG_SW_0, hence checking this register 
	for HCF_MAGIC (the classical NIC presence test) when HCF_ACT_INT_OFF is called due to another card 
	interrupting via a shared IRQ during a download, fails.
 4:	The construction "if ( ifbp->IFB_IntOffCnt-- == 0 )". is optimal (in the sense of shortest/quickest
 	path in error free flows) but NOT fail safe in case of too many INT_ON invocations compared to INT_OFF).
	When a real life MSF programmer ran to a MSF sequence problem, exactly causing that problem, he
	was annoyed. As a side-effect of this unhappy MSF programmer adventures to find his problem, the
	return status is defined to reflect the IFBIntOffCnt.
	Note that this is solely intended to aid debugging, no MSF logic should depend on this feature, no
	guarantees for the future are given.
 	Enabling of the interrupts is achieved by writing the Hermes IntEn register.
	 - If the HCF is in Defunct mode, the interrupts stay disabled.
	 - Under "normal" conditions, the HCF is only interested in Info Events, Rx Events and Notify Events.
	 - When the HCF is out of Tx/Notify resources, the HCF is also interested in Alloc Events.
 6:	ack the "old" Rx-event. See "Rx Buffer free strategy" in hcf_service_nic above for more explanation.
	IFB_RxFID, IFB_RxLen and IFB_RxStat must be cleared to bring both the internal HCF house keeping as the
	information supplied to the MSF in the state "no frame received"
 8:	The HCF_ACT_SCAN, HCF_ACT_ACS_SCAN and HCF_ACT_TALLIES activity are merged by "clever" algebraic 
 	manipulations of the RID-values and action codes, so foregoing robustness against migration problems 
	for ease of implementation. The assumptions about numerical relationships between CFG_TALLIES etc and 
	HCF_ACT_TALLIES etc are checked by the "#if" statements just prior to the body of this routine,
	resulting in: err "maintenance" during compilation if the assumptions are no longer met
	The writing of HREG_PARAM_1 with 0x3FFF in case of an ACS scan, is a kludge to get around lack of 
	specification, hence different implementation in F/W and Host	
	When there is no NIC RAM available, some versions of the Hermes F/W do report 0x7F00 as error in the Result
	field of the Status register and some F/W versions don't. To mask this difference to the MSF all return
	codes of the Hermes are ignored ("best" and "most simple" solution to these types of analomies with a
	acceptable loss due to ignoring all error situations as well)
 	The tallying of "No inquire space" is done by cmd_exe.
30: do not HCFASSERT( rc, rc ) since rc == HCF_INT_PENDING is no error
.ENDOC				END DOCUMENTATION

**************************************************************************************************************/
#if CFG_SCAN != CFG_TALLIES - HCF_ACT_TALLIES + HCF_ACT_SCAN
err "maintenance" apparently inviolated the underlying assumption about the numerical values of these macros
#endif
#if CFG_ACS_SCAN != CFG_TALLIES - HCF_ACT_TALLIES + HCF_ACT_ACS_SCAN
err "maintenance" apparently inviolated the underlying assumption about the numerical values of these macros
#endif
int
hcf_action( IFBP ifbp, hcf_16 action )
{

int		rc = HCF_SUCCESS;
hcf_16	i;

	HCFLOGENTRY( HCF_TRACE_ACTION, action )														/* 0 */
	HCFASSERT( ifbp->IFB_Magic == HCF_MAGIC, ifbp->IFB_Magic )
#if defined HCF_INT_ON
	HCFASSERT( ifbp->IFB_IntOffCnt != 0xFFFF || 
			   action == HCF_ACT_INT_OFF || action == HCF_ACT_INT_FORCE_ON ||
			   action == HCF_ACT_SCAN || action == HCF_ACT_TALLIES,
	 		   action );
#endif // HCF_INT_ON
	
	switch (action) {
#if defined HCF_INT_ON
	  case HCF_ACT_INT_OFF:						// Disable Interrupt generation
/*2*/	ifbp->IFB_IntOffCnt++;																		
//!     rc = 0;
		i = IPW( HREG_INT_EN );	
		OPW( HREG_INT_EN, 0 );
		if ( i & 0x1000 ) rc = HCF_ERR_NO_NIC;
		else if ( i & IPW( HREG_EV_STAT ) ) rc = HCF_INT_PENDING;
		break;

	  case HCF_ACT_INT_FORCE_ON:				// Enforce Enable Interrupt generation
		ifbp->IFB_IntOffCnt = 0;
		//Fall through in HCF_ACT_INT_ON

	  case HCF_ACT_INT_ON:						// Enable Interrupt generation
		HCFASSERT( ifbp->IFB_IntOffCnt != 0xFFFF, ifbp->IFB_IntOffCnt )
/*4*/	if ( ( rc = ifbp->IFB_IntOffCnt-- ) == 0 && !(ifbp->IFB_CardStat & CARD_STAT_DEFUNCT) ) {
			OPW( HREG_INT_EN, ifbp->IFB_RscInd ? HREG_EV_INFO | HREG_EV_RX | HCF_EX_INT 
											   : HREG_EV_INFO | HREG_EV_RX | HCF_EX_INT | HREG_EV_ALLOC );
		}
		break;
#endif // HCF_INT_ON

#if ! ((HCF_TYPE) & HCF_TYPE_USB)
	  case 	HCF_ACT_RX_ACK:						//Receiver ACK 
/*6*/	if ( ifbp->IFB_RxFID ) DAWA_ACK( ifbp, HREG_EV_RX );															
		ifbp->IFB_RxFID = ifbp->IFB_RxLen = 0;
		break;
#endif // HCF_TYPE_USB
		break;

/*8*/ case	HCF_ACT_ACS_SCAN:					// Hermes ACS Scan (F102)
		IF_NOT_USB( OPW( HREG_PARAM_1, 0x3FFF ); )	/*;? kludge to get around lack of specification, 
													 *  hence different implementation in F/W and Host	*/
			//Fall through in HCF_ACT_TALLIES
	  case 	HCF_ACT_TALLIES:					// Hermes Inquire Tallies (F100) 					
	  case 	HCF_ACT_SCAN:						// Hermes Inquire Scan (F101)
		/*!! the assumptions about numerical relationships between CFG_TALLIES etc and HCF_ACT_TALLIES etc
		 *   are checked by #if statements just prior to this routine resulting in: err "maintenance" 	*/
		cmd_exe( ifbp, HCMD_INQUIRE, action - HCF_ACT_TALLIES + CFG_TALLIES );
		break;

	  default:
		HCFASSERT( DO_ASSERT, action )
		break;
	}
	//! do not HCFASSERT( rc == HCF_SUCCESS, rc )														/* 30*/
	HCFLOGEXIT( HCF_TRACE_ACTION )
	return rc;
}/* hcf_action */


/*******************************************************************************************************************

.MODULE			hcf_cntl_port
.DESCRIPTION    Enables or disables data transmission and reception on a specific port
.ARGUMENTS
  int hcf_cntl_port( IFBP ifbp, hcf_16 port_cntl )
.RETURNS
	HCF_SUCCESS
	HCF_ERR_NO_NIC
	>>cmd_wait


.NARRATIVE
  Parameters:
  	ifbp		address of the Interface Block
	port_cntl	0x0001:	0 = disable, 1 = enable
				0x0700: port to be enabled (range HCF_PORT_0 through HCF_PORT_6)

  Description:

.DIAGRAM
	hcf_cntl_port takes successively the following actions:
 2:	If the HCF is incompatible with the Primary or Station Supplier in the Hermes, hcf_cntl_port() 
 	returns immediately with HCF_ERR_NO_NIC as status.
 4: Masking out all non-port bits prevents screwing up the Hermes via cmd_exe
 6: To assure that all static configurations are activated, the Hermes enable command is preceded by
 	an Hermes disable command. 

.NOTICE
  When the Hermes enable cmd is given, the static configuration of the Hermes is done.
  In case of the first port to be enabled, this is the NIC wide as well as the particular port
  configuration. If already one or more ports are enabled, only the static configuration of that particular
  port is done.

.NOTICE
  In former days, hcf_enable/disable (the predecessors of hcf_cntl_port) checked whether HCF_ACT_CARD_IN
  had happend. These days, the HCF relies on the appropriate sequencing by the MSF. Once HCF_ACT_CARD_IN
  is integrated in the hcf_connect this is no longer even a theoretical issue
.ENDOC				END DOCUMENTATION

***************************************************************************************************************/
#if HCMD_ENABLE != 0x0001 || HCMD_ENABLE != HCF_PORT_ENABLE || HCMD_ENABLE & (HCF_PORT_DISABLE|HFS_TX_CNTL_PORT)
potentially disaster, the coding is inherently based on HCMD_ENABLE being a single bit, 
fully disjunct with HCMD_DISABLE and HFS_TX_CNTL_PORT, and the same numerical value as HCMD_ENABLE
#endif // HCMD_ENABLE
#if HCMD_DISABLE != 0x0002 || HCMD_DISABLE != HCF_PORT_DISABLE || HCMD_DISABLE & (HCMD_ENABLE|HFS_TX_CNTL_PORT)
potentially disaster, the coding is inherently based on HCMD_DISABLE being a single bit, 
fully disjunct with HCMD_ENABLE and HFS_TX_CNTL_PORT, not necissarily the same numerical value as HCF_PORT_DISABLE
#endif // HCMD_ENABLE


int
hcf_cntl_port( IFBP ifbp, hcf_16 port_cntl )
{

int	rc = HCF_ERR_NO_NIC;	/*;?this preset is not really needed in case the NIC is really absent 
							  because cmd_exe will fail. In case of CARD_STAT_INCOMP_... it may not be the
							  most optimal value */

	HCFASSERT( (port_cntl & HCF_PORT_ENABLE) ^ (port_cntl & HCF_PORT_DISABLE), port_cntl )
	HCFASSERT( (port_cntl & ~(HCMD_ENABLE|HCMD_DISABLE|HFS_TX_CNTL_PORT) ) == 0, port_cntl ) //;!  HFS_TX_CNTL_PORT is 0 for pure Station
	if ( ( ifbp->IFB_CardStat & ( CARD_STAT_INCOMP_PRI|CARD_STAT_INCOMP_STA ) ) == 0 ) {				 /*2*/
/*4*/	port_cntl &= HFS_TX_CNTL_PORT | HCMD_ENABLE;																	
/*6*/	rc = cmd_exe( ifbp, (hcf_16)(HCMD_DISABLE | (port_cntl & HFS_TX_CNTL_PORT) ), 0 );			
/* ;?!!!!! moved from other places , but this place is plain wrong. It must be somewhere in the initialization
   ;?!!!!! flow. This place will do to get going for the time being with simple driver
*/
		if ( rc == HCF_SUCCESS && port_cntl & HCMD_ENABLE ) {
			rc = cmd_exe( ifbp, (hcf_16)( port_cntl & ( HFS_TX_CNTL_PORT | HCMD_ENABLE ) ), 0 );
		}
	}
	return rc;
}/* hcf_cntl_port */


/******************************************************************************************************************

.MODULE			hcf_connect
.DESCRIPTION	Initializes Card and HCF housekeeping

.ARGUMENTS
  void hcf_connect( IFBP ifbp, hcf_io io_base )

.RETURNS
	HCF_SUCCESS
	HCF_ERR_DEFUNCT_CMD_SEQ
	HCF_ERR_NO_NIC 
	HCF_ERR_TIME_OUT 

.NARRATIVE

 Parameters:
	ifbp		address of the Interface Block
	io_base		I/O Base address of the NIC (connect)
				0 (disconnect)

  MSF-accessible fields of Result Block:
	IFB_IOBase				entry parameter io_base
	IFB_IORange				HREG_IO_RANGE (0x40)
	IFB_Version				version of the IFB layout (0x01 for this release)

.DIAGRAM
0:	Throughout hcf_connect you need to distinguish the connect from the disconnect case, which requires
	some attention about what to use as "I/O" address when for which purpose.
2:
2a: Reset H-II by toggling reset bit in IO-register on and off. 
	The HCF_TYPE_PRELOADED caters for the DOS environment where H-II is loaded by a seperate program to
	overcome the 64k size limit posed on DOS drivers.
	The macro OPW is not yet useable because the IFB_IOBase field is not set.
	Note: hopefully the clearing and initializing of the IFB (see below) acts as a delay which meets the 
	specification for S/W reset
2b:	Experimentally it is determined over a wide range of F/W versions that waiting for the for Cmd bit in 
	Ev register gives a workable strategy. The available documentation does not give much clues.
 4:	clear and initialize the IFB
 	The HCF house keeping info is designed such that zero is the appropriate initial value for as much as
	feasible IFB-items.
	The readable fields mentioned in the description section and some HCF specific fields are given their
	actual value.
	IFB_TickIni is initialized at best guess before calibration
	Hcf_connect defaults to "no interrupt generation" and "no card present" (implicitly achieved by the 
	zero-filling).
 6:	Register compile-time linked MSF Routine and set default filter level
 	cast needed to get around the "near" problem in DOS COM model
	error C2446: no conversion from 'void (__near __cdecl *)(unsigned char __far *,unsigned int,unsigned short ,int )'
								to  'void (__far  __cdecl *)(unsigned char __far *,unsigned int,unsigned short ,int )'
 8:									
10:	Ack everything to unblock a (possibly) blocked cmd pipe line
	Note 1: it is very likely that an Alloc event is pending and very well possible that a (Send) Cmd event is 
	pending on non-initial calls  
	Note 2: it is assumed that this strategy takes away the need to ack every conceivable event after an 
	Hermes Initialize
12:	Only H-II NEEDS the Hermes Initilialize command. Due to the different semantics for H-I and H-II 
	Initialize command, init() does not (and can not, since it is called e.g. after a download) execute
	the Hermes Initialize command. Executing the Hermes Initialize command for H-I would not harm but not
	do anything usefull either, so it is skipped.
	The return status of cmd_exe is ignored. It is assumed that if cmd_exe fails, init fails too
14: use io_base as a flag to merge hcf_connect and hcf_disconnect into 1 routine 
	the call to init and its subsequent call of cmd_exe will return HCF_ERR_NO_NIC if appropriate. This status
	is (badly) needed by some legacy combination of NT4 and card services which do not yield an I/O address
	in time.

.NOTICE
  On platforms where the NULL-pointer is not a bit-pattern of all zeros, the zero-filling of the IFB results
  in an incorrect initialization of pointers.
  The implementation of the MailBox manipulation in put_mb_info protects against the absence of a MailBox
  based on IFB_MBSize, IFB_MBWp and ifbp->IFB_MBRp. This has ramifications on the initialization of the MailBox
  via hcf_put_info with the CFG_REG_MB type, but it prevents dependency on the "NULL-"ness of IFB_MBp.

.NOTICE
  There are a number of problems when asserting and logging hcf_connect, e.g.
   - Asserting on re-entrancy of hcf_connect by means of
	 "HCFASSERT( (ifbp->IFB_AssertTrace & HCF_ASSERT_CONNECT) == 0, 0 )" is not useful because IFB contents
	 are undefined
   - Asserting before the IFB is cleared will cause assert() to interpret the garbage in IFB_AssertRtn
   	 as a routine address
  - The same applies to calling debug_trigger (indirectly via HCFTRACE or HCFLOGENTRY). In addition
	hcf_connect should not perform I/O while debug_trigger (depending on the options) may do I/O.
	Therefor HCFTRACE nor HCFLOGENTRY is called by hcf_connect.
.ENDOC				END DOCUMENTATION

**************************************************************************************************************/
int
hcf_connect( IFBP ifbp, hcf_io io_base )
{

hcf_io	io_addr;
int		rc = HCF_SUCCESS;
hcf_32	prot_cnt;															
hcf_8 	*q;

	if ( io_base ) io_addr = io_base;		//connect													/* 0 */
	else {									//disconnect
		io_addr = (hcf_io)/*;?NDIS*/ifbp->IFB_IOBase;
		IF_NOT_USB(OPW( HREG_INT_EN, 0 );)				//;?workaround against dying F/W on subsequent hcf_connect calls
		hcf_cntl_port( ifbp, HCF_PORT_DISABLE );
	}
#if ( (HCF_TYPE) & (HCF_TYPE_HII | HCF_TYPE_PRELOADED) ) == HCF_TYPE_HII
	OUT_PORT_WORD( io_addr + HREG_IO, 0x0001 );	//OPW not yet useable									/* 2a*/
#endif // HCF_TYPE_HII
	for ( q = (hcf_8*)(ifbp+1); q > (hcf_8*)ifbp; *--q = 0) /*NOP*/;									/* 4 */

	ifbp->IFB_IOBase 	= io_addr;
	ifbp->IFB_IORange	= HREG_IO_RANGE;
	ifbp->IFB_Magic		= HCF_MAGIC;
	ifbp->IFB_Version	= IFB_VERSION;
	prot_cnt = ifbp->IFB_TickIni = INI_TICK_INI;
#if ( (HCF_TYPE) & (HCF_TYPE_HII | HCF_TYPE_PRELOADED) ) == HCF_TYPE_HII
	//rv: for ( prot_cnt = 0; prot_cnt <100; prot_cnt++)
	OPW( HREG_IO, 0x0000 );						//OPW useable											/* 2b*/
	HCF_WAIT_WHILE( (IPW( HREG_EV_STAT) & HREG_EV_CMD) == 0 );
	if ( prot_cnt ) prot_cnt = ifbp->IFB_TickIni;
#endif // HCF_TYPE_HII
#if defined HCF_ASSERT
	ifbp->IFB_AssertLvl = 1;
#if HCF_ASSERT & 0x8000
	ifbp->IFB_AssertRtn = (MSF_ASSERT_RTNP)msf_assert;   												/* 6 */
#endif // HCF_ASSERT
#endif // HCF_ASSERT
	HCFASSERT( ((long)(void FAR *)ifbp & (HCF_ALIGN-1) ) == 0, (hcf_16)(long)(void FAR *)ifbp )
#if (HCF_TYPE) & HCF_TYPE_USB
//	HCFASSERT( io_base == 0, io_base )  consider what to do when hcf_connect/disconnect for USB is used again
#else	
	HCFASSERT( (io_base & 0x003F) == 0, io_base )
#endif // HCF_TYPE_USB
#if ! ((HCF_TYPE) & HCF_TYPE_USB)
												//if Busy bit in Cmd register
	if (IPW( HREG_CMD ) & HCMD_BUSY ) {																	/* 8 */
												//.  Ack everything to unblock a (possibly) blocked cmd pipe line
		OPW( HREG_EV_ACK, 0xFFFF );
												//.  Wait for Busy bit drop  in Cmd register
												//.  Wait for Cmd  bit raise in Ev  register
		HCF_WAIT_WHILE( ( IPW( HREG_CMD ) & HCMD_BUSY ) && (IPW( HREG_EV_STAT) & HREG_EV_CMD) == 0 );
	}
												//if prot_cnt == 0, then PANICK
		/* assume if prot_cnt == 0, then the next cmd_exe will fail, causing the HCF to go into DEFUNCT mode */
	HCFASSERT( prot_cnt, IPW( HREG_EV_STAT) )
	OPW( HREG_EV_ACK, 0xFFFF );																			/*10*/
#endif // HCF_TYPE_USB
#if ( (HCF_TYPE) & (HCF_TYPE_HII | HCF_TYPE_PRELOADED) ) == HCF_TYPE_HII								/*12*/
	(void)cmd_exe( ifbp, HCMD_INI, 0 );
#endif
	if ( io_base ) rc = init( ifbp );																	/*14*/
	ifbp->IFB_IOBase = io_base;																			/* 0*/
	return rc;
}/* hcf_connect	*/


/**************************************************************************************************************

.MODULE			hcf_encap
.DESCRIPTION	test whether len/type is E-II and if so, whether RFC1042 or Bridge-Tunnel
				encapsulation is to be used. 
				The return value is ENC_NONE or the appropriate type

.ARGUMENTS
 hcf_8 hcf_encap( wci_bufp type )

.RETURNS
	ENC_NONE		len/type is "len" ( (BIG_ENDIAN)type <= 1500 )
	ENC_TUNNEL 		len/type is "type" and 0x80F3 or 0x8137
	ENC_1042		len/type is "type" but not 0x80F3 or 0x8137

.NARRATIVE

 Parameters:
	type	len/type (in "network" Endian format)

 1:	presume 802.3, hence preset return value at ENC_NONE
 2:	convert type from "network" Endian format to native Endian
 4:	the litmus test to distinguish type and len.
 	The hard code "magic" value of 1500 is intentional and should NOT be replaced by a mnemonic and definitely
 	NOT be HCF_MAX_MSG because it is not related at all to the maximum frame size supported  by the Hermes.
6:	check type against:
		0x80F3	//AppleTalk Address Resolution Protocol (AARP)
		0x8137	//IPX
	to determine the type of encapsulation

**************************************************************************************************************/
#if HCF_ENCAPSULATION	//i.e 0x0001 or 0x0002
hcf_8 hcf_encap( wci_bufp type )
{

hcf_8	rc = ENC_NONE;																					/* 1 */
hcf_16	t = (hcf_16)(*type<<8) + *(type+1);																/* 2 */

	if ( t > 1500 ) { 																					/* 4 */
	  	if ( t == 0x8137 || t == 0x80F3 ) rc = ENC_TUNNEL;												/* 6 */
		else rc = ENC_1042;
	}
	return rc;
} /* hcf_encap */
#endif // HCF_ENCAPSULATION


/**************************************************************************************************************

.MODULE			hcf_get_info
.DESCRIPTION
	Obtains transient and persistent configuration information from the
	Card and from the HCF

.ARGUMENTS
  int hcf_get_info( IFBP ifbp, LTVP ltvp )
  Card Interrupts disabled

.RETURNS
	HCF_ERR_LEN
	HCF_SUCCESS
	HCF_ERR_NO_NIC		all cases except the "HCF embedded" pseudo RIDs
	>>strio		>= CFG_RID_CFG_MIN

.NARRATIVE
	parameters:
		ifbp	address of the Interface Block
		ltvp	address of LengthTypeValue structure specifying the "what" and the "how much" of the
				information to be collected from the HCF or from the Hermes

.NOTICE

  "HCF embedded" pseudo RIDs:
	CFG_MB_INFO, CFG_TALLIES_VACATE, CFG_TALLIES, CFG_RESTRAINT_INFO, CFG_IFB,
	CFG_DRV_IDENTITY, CFG_DRV_SUP_RANGE, CFG_DRV_ACT_RANGES_PRI, CFG_DRV_ACT_RANGES_STA,
	CFG_DRV_ACT_RANGES_HSI
	Note the HCF_ERR_LEN is NOT adequately set, when L >= 2 but less than needed


 Remarks: Transfers operation information and transient and persistent
 	configuration information from the Card and from the HCF to the MSF.
	The exact layout of the provided data structure
	depends on the action code. Copying stops if either the complete
	Configuration Information is copied or if the number of bytes indicated
	by len is copied.  Len acts as a safe guard against Configuration
	Information blocks which have different sizes for different Hermes
	versions, e.g. when later versions support more tallies than earlier
	versions. It is a conscious decision that unused parts of the PC RAM buffer are not cleared.

 Remarks: The only error against which is protected is the "Read error"
	as result of Card removal. Only the last hcf_io_string need
	to be protected because if the first fails the second will fail
	as well. Checking for cmd_exe errors is supposed superfluous because
	problems in cmd_exe are already caught or will be caught by
	hcf_enable.

	CFG_MB_INFO: copy the oldest MailBox Info Block or the "null" block if none available.

 3:	tallying of "No inquire space" is done by cmd_wait

 Note:
	the codes for type are "cleverly" chosen to be identical to the RID


	The mechanism to HCF_ASSERT on invalid typ-codes in the LTV record is based on the following strategy:
	  - during the pseudo-asynchronous Hermes commands (diagnose, download)	only CFG_MB_INFO is acceptable
	  -	some codes (e.g. CFG_TALLIES) are explicitly handled by the HCF which implies that these codes
		are valid
	  - all other codes in the range 0xFC00 through 0xFFFF are passed to the Hermes.  The Hermes returns an
	  	LTV record with a zero value in the L-field for all Typ-codes it does not recognize. This is
	  	defined and intended behavior, so HCF_ASSERT does not catch on this phenomena.
	  -	all remaining codes are invalid and cause an ASSERT.

.DIAGRAM
	Hcf_get_mb_info copies the contents of the oldest MailBox Info block in the MailBox
	to PC RAM. If len is less than the size of the MailBox Info block, only as much as
	fits in the PC RAM buffer is copied. After the copying the MailBox Read pointer is
	updated to point to the next MailBox Info block, hence the remainder of an "oversized"
	MailBox Info block is lost. The truncation of the MailBox Info block is NOT reflected in
	the return status. Note that hcf_get_info guarantees the length of the PC RAM buffer meets the
	minimum requirements of at least 2, so no PC RAM buffer overrun.

	Calling hcf_get_mb_info when their is no MailBox Info block available or
	when there is no MailBox at all, results in a "NULL" MailBox Info block.

12:	see NOTICE
17:	The return status of cmd_wait and the first hcfio_in_string can be ignored, because when one fails, the
 	other fails via the IFB_DefunctStat mechanism
20:	"HCFASSERT( rc == HCF_SUCCESS, rc )" is not suitable because this will always trigger as side effect of
	the HCFASSERT in hcf_put_info which calls hcf_get_info to figure out whether the RID exists at all.


.NOTICE	The protection in hcf_get_info() is less restrictive than the protection in hcf_put_info(). This is
	one of those cases that are impossible to decide at forehand what is the best strategy. The current
	choice implies that an Information RID (e.g. MaxTransmitLifetime, 0xFD4a) can not be changed into
	a Configuration RID in a transparent fashion for the HCF


**************************************************************************************************************/
int
hcf_get_info(IFBP ifbp, LTVP ltvp )
{

int				rc = HCF_ERR_LEN;
hcf_16			len = ltvp->len;
hcf_16			type = ltvp->typ;
wci_bufp		cp = (wci_bufp)ltvp->val;	//destination char pointer (in LTV record)
wci_recordp		p = ltvp->val;				//destination word pointer (in LTV record)
wci_recordp		q;							//source word pointer
hcf_io			i;

	HCFLOGENTRY( HCF_TRACE_GET_INFO, ltvp->typ )
	HCFASSERT( ifbp->IFB_Magic == HCF_MAGIC, ifbp->IFB_Magic )
	HCFASSERT_INT
	HCFASSERT( ltvp, 0 )
	HCFASSERT( ltvp->len > 1, ltvp->len )
	HCFASSERT( ltvp->len > 1, ltvp->typ )

	IF_USB( if ( ltvp->len > HCF_MAX_MSG ) ltvp->len = HCF_MAX_MSG; )//;?nv just as a try to get around PDA problems
	if ( len >= 2 ) {
		rc = HCF_SUCCESS;
		switch ( type ) {

#if defined MSF_COMPONENT_ID
		  case CFG_DRV_IDENTITY:				//0x0826
		  case CFG_DRV_SUP_RANGE:				//0x0827
		  case CFG_DRV_ACT_RANGES_PRI:			//0x0828
		  case CFG_DRV_ACT_RANGES_STA:			//0x0829
		  case CFG_DRV_ACT_RANGES_HSI:			//0x082A
		  case CFG_DRV_ACT_RANGES_APF:			//0x082B
		  case CFG_HCF_OPT:						//0x082C
		  	q = (wci_recordp)xxxx[ type - CFG_DRV_IDENTITY ];
			ltvp->len = *q;
			q += 2;								//skip L and T
			//FALL THROUGH
#endif // MSF_COMPONENT_ID
#if defined HCF_MB_ON
		  case CFG_MB_INFO:											//Get Mail Box Info Block
		  	if ( type == CFG_MB_INFO ) {		//share the code without gotos, without the overhead of function
//		  	get_info_mb( ifbp, ltvp );
				ltvp->len = 1;
				ltvp->typ = CFG_NULL;
				if ( ifbp->IFB_MBp ) {
					if ( ifbp->IFB_MBp[ifbp->IFB_MBRp] == 0xFFFF ) ifbp->IFB_MBRp = 0;
					q = &ifbp->IFB_MBp[ifbp->IFB_MBRp];
					if ( *q ) {
						ltvp->len = q[0];
						ltvp->typ = q[1];
						ifbp->IFB_MBRp += *q + 1;				//update read pointer
						if ( ifbp->IFB_MBp[ifbp->IFB_MBRp] == 0xFFFF ) ifbp->IFB_MBRp = 0;
						q += 2;
					}
					ifbp->IFB_MBInfoLen = ifbp->IFB_MBp[ifbp->IFB_MBRp];
				}
			}
			//FALL THROUGH
#endif // HCF_MB_ON
		  case CFG_IFB:							//0x0824		
		  	if ( type == CFG_IFB ) {			//share the code without gotos, without the overhead of function
				ltvp->len = sizeof(IFB_STRCT)/ sizeof(hcf_16) + 1;
				q = (wci_recordp)ifbp;
			}
			//FALL THROUGH
#if HCF_TALLIES
/*3*/	  case CFG_TALLIES:																			
		  	if ( type == CFG_TALLIES ) {		//share the code without gotos, without the overhead of function
				(void)hcf_action( ifbp, HCF_ACT_TALLIES );
//This or That	ltvp->len = (HCF_TOT_TAL_CNT + HCF_TOT_TAL_CNT) + 1;
				ltvp->len = (HCF_TOT_TAL_CNT + HCF_TOT_TAL_CNT);
				q = (wci_recordp)&ifbp->IFB_NIC_Tallies;
			}
#endif //HCF_TALLIES
			if ( ltvp->len > len ) {
				rc = HCF_ERR_LEN;
				ltvp->len = len;
			}
			len = ltvp->len;
			while ( --len ) {
				*p++ = *q;
#if HCF_TALLIES & 0x0002
				if ( type == CFG_TALLIES ) *q = 0;
#endif //HCF_TALLIES
				q++;
			}
			break;

#if (HCF_TYPE) & HCF_TYPE_USB
		  default:
			get_info_usb( ifbp, ltvp );
#else			
		  case CFG_PROD_DATA: 
			HI_AUX_CNTL( ifbp, HREG_CNTL_AUX_ENA_CNTL );	//ignore return status, PDA check will fail supposedly
			OPW( HREG_AUX_PAGE, PLUG_DATA_OFFSET >> 7 );
			OPW( HREG_AUX_OFFSET, PLUG_DATA_OFFSET & 0x7E );
			--len;								//pre-decrement compensates for space occupied by T
			i = (hcf_io)ifbp->IFB_IOBase + HREG_AUX_DATA;	//to prevent side effects of the MSF-defined macro
			IN_PORT_STRING( i, cp, len );	
			HI_AUX_CNTL( ifbp, HREG_CNTL_AUX_DIS_CNTL );	//ignore return status, previous call apparently succeeded
			break;
		  default:											//all "in the right range" are passed to Hermes
/*12*/		if ( type < CFG_RID_CFG_MIN ) {														
				HCFASSERT( DO_ASSERT, type )
				ltvp->len = 0;
			} else {
//;?prepared to merge the 2 Busy waits	rc = strio( ifbp, IO_IN, type, 0, NULL, 0, BE_PAR(0) ) |
/*17*/			rc = cmd_exe( ifbp, HCMD_ACCESS /*| HCMD_BUSY*/, type );	//;?prepared to merge the 2 Busy waits
				if ( rc == HCF_SUCCESS ) {
//;?prepared to merge the 2 Busy waits: wait for Busy low in offset, call cmd_cmpl, IN_PORT_STRING
					rc = strio( ifbp, IO_IN, type, 0, (wci_recordp)ltvp, len+1 BE_PAR(2) );
//update				HCFASSERT( i <= HCF_MAX_LTV, i )
//update				HCFASSERT( i <= HCF_MAX_LTV, type )
				}
//update ltvp->len					
				if ( rc == HCF_SUCCESS ) {
					if ( len < ltvp->len ) { 
						ltvp->len = len;
						rc = HCF_ERR_LEN;
					}
//					len--;	//len was at least 2 so it can stand another decrement 
//					i = (hcf_16/*;?NDIS*/)ifbp->IFB_IOBase + HREG_DATA_0;	//to prevent side effects of the MSF-defined macro
//					IN_PORT_STRING( i, cp, len );	
				}
			}
#endif //HCF_TYPE_USB
		}
	}
	HCFASSERT( rc == HCF_SUCCESS ||
/*20*/		   ( rc == HCF_ERR_LEN && ifbp->IFB_AssertTrace & 1<<HCF_TRACE_PUT_INFO ), rc )				
	HCFASSERT( rc == HCF_SUCCESS ||
			   ( rc == HCF_ERR_LEN && ifbp->IFB_AssertTrace & 1<<HCF_TRACE_PUT_INFO ), type )
	HCFLOGEXIT( HCF_TRACE_GET_INFO )
	return rc;

}/* hcf_get_info */

#if (HCF_TYPE) & HCF_TYPE_USB
int
get_info_usb(IFBP ifbp, LTVP ltvp )
{

int				rc = HCF_SUCCESS;
wci_recordp		p = ltvp->val;				//destination word pointer (in LTV record)
hcf_16			type = ltvp->typ;
hcf_16			len;
hcf_16			i;
hcf_16 			*q;							//source pointer

	if ( ltvp->len > HCF_MAX_MSG ) ltvp->len = HCF_MAX_MSG; //;? just as a try to get around PDA problems
	len = ltvp->len;
	switch ( type ) {

#if defined MSF_COMPONENT_ID
	  case CFG_DRV_ACT_RANGES_UDI:			// (0x0885)
	  case CFG_DRV_ACT_RANGES_UBI:			// (0x0886)
		q = (hcf_16*)xuuu[ type - CFG_DRV_ACT_RANGES_UDI ];
		ltvp->len = *q;
		q += 2;								//skip L and T
		if ( ltvp->len > len ) {
			rc = HCF_ERR_LEN;
			ltvp->len = len;
		}
		len = ltvp->len;
		while ( --len ) *p++ = *q++;
		break;
#endif // MSF_COMPONENT_ID

#if defined HCF_DLV
	  case 0x880:		//;?create mnemonics
	  case CFG_UBI_SUP_RANGE:				//0x0881		//USB Converter board
	  case 0x882:  		//;?create mnemonics 
	  case CFG_UDI_SUP_RANGE:				//0x0883		//USB Bootloader 
	  case 0x884:		//;?create mnemonics
					//pointer to length in bytes of (CFG_USB_MNGMT + overhead)
		p = (hcf_16*)(UNUGetUsbInfo(  ) )->ucLucentVersion;
		i = 2;		//compensate for overhead in Configuration Management Descriptor (CFG_USB_MNGMT)
					//convert word-offset to byte-offset and compensate length of CFG_USB_MNGMT for overhead
		while ( 2*i < (hcf_16)(*p-4) ) {
			if ( type == p[i] ) {	//match found, so copy this sub-LTV
				NdisMoveMemory( &ltvp->len, &p[i-1], (p[i-1]+1 ) * 2);
			}
			i += p[i-1]+1;		//add L of current sub-LTV + 1 (to reflect T)
		}
		break;

	  case CFG_PROD_DATA:
		HCFASSERT( len > 4, len )
		ltvp->val[0] = CFG_MEM_READ;		//action
		ltvp->val[1] = len - 1;				//len in words, including T  ;?hopefully this suffice
		ltvp->val[2] = 0;					//offset
		rc = UNUSendHcfReq( type, &ltvp->val[ 0], 10, ltvp);
		break;
#endif // HCF_DLV

	  default:											//all "unknown" ones are passed to Hermes
		rc = UNUSendHcfReq( type, NULL, 0, ltvp);
	}
	return rc;
}/* get_info_usb */
#endif // HCF_TYPE_USB


/**************************************************************************************************************

.MODULE			hcf_put_info
.DESCRIPTION
	Transfers operation information and transient and persistent configuration information to the Card.

.ARGUMENTS
  int hcf_put_info( IFBP ifbp, LTVP ltvp )
  Card Interrupts disabled

.RETURNS
	HCF_SUCCESS				CFG_TICK_TIME (but Nothing happens)
	HCF_SUCCESS				<= CFG_RID_CFG_MIN
	HCF_SUCCESS				>= CFG_RID_CFG_MAX
	>>download				CFG_DLNV_START <= ..... <= CFG_DL_STOP
	HCF_SUCCESS				<= CFG_RID_CFG_MIN <= .... <= CFG_RID_CFG_MAX
	put_info				<= CFG_RID_CFG_MIN <= .... <= CFG_RID_CFG_MAX

.NARRATIVE
	parameters:
		ifbp		address of the Interface Block
		ltvp		specifies the RID (as defined by Hermes I/F) or pseudo-RID (as defined by WCI)

.NOTICE
 Remarks:
	The Hermes puts additional sequence constraints on the usage of hcf_put_info,
	e.g. CFG_NOTIFY is only allowed after hcf_enable

 Remarks:  In case of Hermes Configuration LTVs, the codes for the type are "cleverly" chosen to
	be identical to the RID. Hermes Configuration information is copied from the provided data
	structure into the Card.
	In case of HCF Configuration LTVs, the type values are chosen in a range which does not overlap
	the RID-range.


 Remarks:  In order to make the changes sustain over activities like hcf_diagnose and recovery from PCMCIA
 	card insertion, Hermes Configuration LTVs are saved in the IF-block.

	The mechanism to HCF_ASSERT on invalid typ-codes in the LTV record is based on the following strategy:
	  -	some codes (e.g. CFG_REG_MB) are explicitly handled by the HCF which implies that these codes
		are valid
	  - all other codes are passed to the Hermes.  Before the put action is executed, hcf_get_info is called
	  	with an LTV record with a value of 1 in	the L-field and the intended put action type in the Typ-code
	  	field. If the put action type is valid, it must definitely be valid as a get action type code, so
	  	the HCF_ASSERT logic of hcf_get_info should not catch.


.DIAGRAM
02:	see NOTICE in hcf_get_info()

15:	Like stated in hcfifbp->IFB_nitialize, in any particular situation the decision to save or not to save an
	LTV in case of an error during the configuration, could probably be made. In general
	there does not seem to be "the" right answer, hence here is chosen for the easiest implementation, i.e.
	to ignore errors as far as storing the LTV into the Configuration Table is concerned.
15: 
19:	if HCF_ASSERT is not defined, the else clause absorbs the "return rc;" statement. This is plainly not the
	intention, hence the semi-colon after the first HCFASSERT, so there will be at least an empty statement
	after the else.
	Note that I consider it an error of those compilers who do not flag this problem, e.g. MSVC 4 as used
	for the Miniport (jumps over the mov ax, [bp-n] without complaining about "no return value"), MSVC 1.5 as
	used for the DOS ODI (inserts a NOP and has all if/if_else branches jump to the mov ax, [bp-6], which is
	what I wanted but not what I asked for), Borland 3.1 as used for WavePoint (which even gets confused to
	the level where it leaves out necessary instructions in the if_else branches, skips the mov ax, [bp-n]
	without complaints)

 Remarks:
 	In the past - and apparently these days again - the validity of the T was asserted by calling
	hcf_get_info. This does not work in case hcf_action( HCF_ACT_CARD_IN ) has not yet been called.
	Are the benefits considered not too small to adjust the ASSERT to cope with this additional problem? ;?


.NOTICE
	Future enhancements in the functionality offered by the WCI and/or implementation aspects of the HCF
	may warrant filtering on the type-field of the LTV to recognize non-MSF accessible records, e.g. CFG_TICK

**************************************************************************************************************/

int
hcf_put_info( IFBP ifbp, LTVP ltvp )
{

int		rc = HCF_SUCCESS;
hcf_16	i, k;

	HCFLOGENTRY( HCF_TRACE_PUT_INFO, ltvp->typ )
	HCFASSERT( ifbp->IFB_Magic == HCF_MAGIC, ifbp->IFB_Magic )
	HCFASSERT_INT
	HCFASSERT( ltvp, 0 )
	HCFASSERT( ltvp->len - 1 < HCF_MAX_LTV, ltvp->len )

											//all codes between 0xFC00 and 0xFCFF are passed to Hermes with
											//exception of CFG_TICK_TIME to suppress tampering with the NIC timer
#if defined HCF_DLV
//note to Nico from Roland : what about CFG_DL_STAT ;?
	if ( CFG_DLNV_START <= ltvp->typ && ltvp->typ <= CFG_PROG ) {
#if (HCF_TYPE) & HCF_TYPE_USB
		rc = UNUSendHcfReq( ltvp->typ, &ltvp->val[ 0], (USHORT)(ltvp->len - 1), NULL);								
		
		// missing for USB WKR 3-12-2002		
		if (ltvp->typ == CFG_DL_STOP) {
			if (rc == HCF_SUCCESS) rc = (hcf_16)init( ifbp );
		}
		// missing for USB WKR 3-12-2002		
									
#else //HCF_TYPE_USB
		rc = download( ifbp, ltvp );
#endif //HCF_TYPE_USB
	} else 
#endif // HCF_DLV
	switch (ltvp->typ) {
#if defined HCF_ASSERT  	
 	  case CFG_REG_ASSERT_RTNP:											//Register MSF Routines
		ifbp->IFB_AssertRtn = ((CFG_REG_ASSERT_RTNP_STRCT FAR*)ltvp)->rtnp;
		ifbp->IFB_AssertLvl = ((CFG_REG_ASSERT_RTNP_STRCT FAR *)ltvp)->lvl;
		ifbp->IFB_AssertLvl = 1;										//;? not yet supported so default
		break;
#endif // HCF_ASSERT    	
	  case CFG_REG_INFO_LOG:											//Register Log filter
		ifbp->IFB_RIDLogp = ((CFG_RID_LOG_STRCT FAR*)ltvp)->recordp;
		break;
	  case CFG_REG_MB:													//Register MailBox          	
	  	HCFASSERT( (*(long FAR *)ltvp->val & 01) == 0, ltvp->val[0] )
		ifbp->IFB_MBp = *(wci_recordp FAR *)&ltvp->val[0];
		ifbp->IFB_MBSize = ifbp->IFB_MBp == NULL ? 0 : ltvp->val[2];	/* if no MB present, size must be zero
																		 * for put_info_mb to work correctly */
		ifbp->IFB_MBWp = ifbp->IFB_MBRp	= 0;
		ifbp->IFB_MBp[0] = 0;											//flag the MailBox as empty
		ifbp->IFB_MBInfoLen = 0;
		ifbp->IFB_HCF_Tallies.NoBufMB = 0;															
		HCFASSERT( ifbp->IFB_MBSize >= 60 || ifbp->IFB_MBp == NULL, ifbp->IFB_MBSize )
		break;               

#if ! ((HCF_TYPE) & HCF_TYPE_USB)
	  case CFG_CMD_NIC:
		OPW( HREG_PARAM_2, ((CFG_CMD_NIC_STRCT FAR *)ltvp)->parm2 );
		OPW( HREG_PARAM_1, ((CFG_CMD_NIC_STRCT FAR *)ltvp)->parm1 );
	  	rc = cmd_exe( ifbp, ((CFG_CMD_NIC_STRCT FAR *)ltvp)->cmd, ((CFG_CMD_NIC_STRCT FAR *)ltvp)->parm0 );
		((CFG_CMD_NIC_STRCT FAR *)((CFG_CMD_NIC_STRCT FAR *)ltvp))->hcf_stat = (hcf_16)rc;
		((CFG_CMD_NIC_STRCT FAR *)ltvp)->stat = IPW( HREG_STAT );
		((CFG_CMD_NIC_STRCT FAR *)ltvp)->resp0 = IPW( HREG_RESP_0 );
		((CFG_CMD_NIC_STRCT FAR *)ltvp)->resp1 = IPW( HREG_RESP_1 );
		((CFG_CMD_NIC_STRCT FAR *)ltvp)->resp2 = IPW( HREG_RESP_2 );
		((CFG_CMD_NIC_STRCT FAR *)ltvp)->ifb_err_cmd = ifbp->IFB_ErrCmd;
		((CFG_CMD_NIC_STRCT FAR *)ltvp)->ifb_err_qualifier = ifbp->IFB_ErrQualifier;
		break;
#endif // HCF_TYPE_USB

	  case CFG_CMD_HCF:
		HCFASSERT( ((CFG_CMD_HCF_STRCT FAR *)ltvp)->cmd == 0x0001, ((CFG_CMD_HCF_STRCT FAR *)ltvp)->cmd )				//only Rx-monitor supported
		if ( ((CFG_CMD_HCF_STRCT FAR *)ltvp)->cmd == 0x0001 ) {
			 if ( ( ifbp->IFB_Monitor = ((CFG_CMD_HCF_STRCT FAR *)ltvp)->mode ) != 0x0000 ) {
					ifbp->IFB_Monitor = ((CFG_CMD_HCF_STRCT FAR *)ltvp)->add_info;
			 }
		}
		break;

#if ! ((HCF_TYPE) & HCF_TYPE_HII )
#if (HCF_TYPE) & HCF_TYPE_AP
#if ! ((HCF_TYPE) & HCF_TYPE_USB) // this may be taken together with CFG_CMD_NIC !!!
	  case CFG_NOTIFY:
		if ( ( i = get_fid( ifbp ) ) != 0 ) {
			if ( ( rc = strio( ifbp, IO_OUT, i, 0, &ltvp->len, ltvp->len + 1 BE_PAR(2) ) ) == HCF_SUCCESS ) {
				 rc = cmd_exe( ifbp, HCMD_NOTIFY | HCMD_RECL, i );
			}
		}
		break;
#endif // HCF_TYPE_USB
#endif // HCF_TYPE_AP
#endif // HCF_TYPE_HII
#if (HCF_TYPE) & HCF_TYPE_SSN
	  case CFG_ADD_TKIP_DEFAULT_KEY:
		i = CNV_LITTLE_TO_INT(((CFG_ADD_TKIP_DEFAULT_KEY_STRCT FAR *)ltvp)->tkip_key_id_info);
		k = ( i & KEY_ID ) * 8 + 1;
		if ( i & TX_KEY ) ifbp->IFB_MICKey[0] = k;
		for ( i = 0; i < 8; i++ ) {
			ifbp->IFB_MICKey[i+k] = CNV_LITTLE_TO_INT(((CFG_ADD_TKIP_DEFAULT_KEY_STRCT FAR *)ltvp)->tx_mic_key[i]); 
		}
																//;? Dirty code, fall through in RxMicKey
#endif // HCF_TYPE_SSN	
	  default:
#if (HCF_TYPE) & HCF_TYPE_USB
		rc = put_info( ifbp, ltvp );		//pass everything unknown to the Dongle
#else // HCF_TYPE_USB
											//pass everything unknown above the "FID" range to the Hermes
		if ( CFG_RID_CFG_MIN <= ltvp->typ && ltvp->typ <= CFG_RID_CFG_MAX ) {
			rc = put_info( ifbp, ltvp );
		} else {
			HCFASSERT( DO_ASSERT, ltvp->typ )
/*19*/		/*NOP*/;															      	
		}
#endif // HCF_TYPE_USB
#if (HCF_TYPE) & HCF_TYPE_SSN
//      	if ( ltvp->typ ==  CFG_REMOVE_TKIP_DEFAULT_KEY &&
//			 ifbp->IFB_MICKey[0] == CNV_INT_TO_LITTLE(((CFG_REMOVE_TKIP_DEFAULT_KEY_STRCT FAR *)ltvp)->tkip_key_id) ) {
//			ifbp->IFB_MICKey[0] = 0;		//disable MIC-engine
//		}
      	if(ltvp->typ ==  CFG_REMOVE_TKIP_DEFAULT_KEY) {
      		k = (CNV_INT_TO_LITTLE(((CFG_REMOVE_TKIP_DEFAULT_KEY_STRCT FAR *)ltvp)->tkip_key_id) * 8) + 1 ;
	      	if (k == ifbp->IFB_MICKey[0]) {
				ifbp->IFB_MICKey[0] = 0;		//disable MIC-engine
			}
		}
#endif // HCF_TYPE_SSN	
	}
	HCFASSERT( rc == HCF_SUCCESS, rc )
	HCFLOGEXIT( HCF_TRACE_PUT_INFO )
	return rc;
}/* hcf_put_info */


/*******************************************************************************************************************

.MODULE			hcf_rcv_msg
.DESCRIPTION

.ARGUMENTS
  int hcf_rcv_msg( IFBP ifbp, DESC_STRCT *descp, int offset )

.RETURNS
	HCF_SUCCESS		No SSN error ( or HCF_ERR_MIC already reported by hcf_service_nic)
	HCF_ERR_MIC		

.NARRATIVE

 Parameters:
  ifbp		address of the Interface Block
  descp		address of 
  offset	offset 0 corresponds with 802.3 DestAddr field
   			when offset is within lookahead, data is copied from lookahead
			when offset is beyond lookahead, data is read directly from RxFS in NIC with disregard of
			the actual value of offset

  Description:
Hcf_rcv_msg starts the copy process at the offset requested by the parameter offset, relative to 
HFS_ADDR_DEST, e.g offset 0 starts copying from the Destination Address, the very begin of the 802.3
frame message. Offset must either lay within the part of the 802.3 frame as stored by hcf_service_nic
in the lookahead buffer or be just behind it, i.e. the first byte not yet read.

.DIAGRAM

.NOTICE
 - by using unsigned int as type for offset, no need to worry about negative offsets
 - Asserting on being enabled/present is superfluous, since a non-zero IFB_lal implies that hcf_service_nic 
   was called and detected a Rx-message. A zero IFB_lal will set the buf_cnt field of at least the first
   descriptor to zero.
.ENDOC				END DOCUMENTATION

***************************************************************************************************************/
int
hcf_rcv_msg( IFBP ifbp, DESC_STRCT *descp, unsigned int offset )
{
int			rc = HCF_SUCCESS;
#if ! ((HCF_TYPE) & HCF_TYPE_USB)
int 		tot_len = ifbp->IFB_RxLen - offset;		//total length
wci_bufp	lap = ifbp->IFB_lap + offset;			//start address in LookAhead Buffer 
hcf_16		lal = ifbp->IFB_lal - offset;			//available data within LookAhead Buffer
wci_bufp	cp;
hcf_16		i, j;

	HCFLOGENTRY( HCF_TRACE_RCV_MSG, offset )
 	HCFASSERT( ifbp->IFB_Magic == HCF_MAGIC, ifbp->IFB_Magic )
	HCFASSERT_INT
	HCFASSERT( descp, HCF_TRACE_RCV_MSG )
	HCFASSERT( ((long)descp & 3 ) == 0, (hcf_16)(long)descp )
	HCFASSERT( ifbp->IFB_RxLen, HCF_TRACE_RCV_MSG )
	HCFASSERT( ifbp->IFB_RxLen >= offset, ifbp->IFB_RxLen )
	HCFASSERT( ifbp->IFB_RxLen >= offset, offset )

	if ( tot_len < 0 ) {
		lal = 0; tot_len = 0;	//suppress all copying activity in the do--while loop
	}
	do {					//loop over all available fragments
		HCFASSERT( ((long)descp & 3 ) == 0, (hcf_16)(long)descp )
		cp = descp->buf_addr;		
		j = min( (hcf_16)tot_len, descp->buf_cntl.buf_dim[1] );	//minimum of "what is available" and what fits in the fragment
		descp->buf_cntl.buf_dim[0] = j;
		tot_len -= j;											//adjust length still to go
		if ( lal ) {											//if lookahead Buffer not yet completely copied
			i = min( lal, j );									//minimum of "what is available" in lookahead Buffer and what fits in the fragment	
			lal -= i;											//adjust lenght still available in LookAhead Buffer
			j -= i;												//adjust lenght still available in current fragment
			while ( i-- ) *cp++ = *lap++;		//;?could be improved by moving words at a time but that is complicated on platforms with alignment requirements;?
		}
		if ( j ) rcv_msg_frag( ifbp, cp, j );	//if LookAhead Buffer exhausted but still space in fragment, then copy directly from NIC RAM
	} while ( descp = descp->next_desc_addr );
#if HCF_TYPE & HCF_TYPE_SSN  
	if ( ifbp->IFB_RxFID ) rc = check_mic( ifbp );		//prevents MIC error report if hcf_service_nic already consumed all
#endif //HCF_TYPE_SSN
	hcf_action( ifbp, HCF_ACT_RX_ACK );		//only 1 shot to get the data, so free the resources in the NIC
 	HCFASSERT( rc == HCF_SUCCESS, rc )
	HCFLOGEXIT( HCF_TRACE_RCV_MSG )
#else // HCF_TYPE_USB
hcf_8	mic[8];									//area to save rcvd MIC
hcf_16	flag = 0x0000;							//0x01: decapsulation, 0x02: MIC check
hcf_8	*p8 = (hcf_8*)descp->buf_addr;			//char oriented working pointer
#if HCF_ENCAPSULATION == 0x0001
hcf_16	*dp16 = (hcf_16*)&p8[HFS_LEN];			//word oriented destination pointer
hcf_16	*sp16 = (hcf_16*)&p8[HFS_TYPE];			//word oriented source pointer
#endif // HCF_ENCAPSULATION
hcf_32	*p32  = (hcf_32*)&p8[HFS_ADDR_DEST];	//double-word oriented MIC working pointer
int		i;
											/* length without MAC header and without MIC, 
											 * but possibly with Encapsulation header
											 */
hcf_16	len  =  p8[HFS_DAT_LEN]+(p8[HFS_DAT_LEN+1]<<8);
	
											//determine total number of bytes (DA, SA, L, data (no encapsulation, no MIC))
	descp->buf_cntl.buf_dim[0] = len + HFS_DAT - HFS_ADDR_DEST;
	
	//restore asap HCFASSERT( GET_BUFCOUNT( descp ) <= GET_BUFSIZE(descp) , len )
	//restore asap HCFASSERT( GET_BUFCOUNT( descp ) <= GET_BUFSIZE(descp), GET_BUFSIZE(descp) )

#if HCF_ENCAPSULATION == 0x0001
											//determine decapsulation sub-flag	
	i = p8[HFS_STAT+1] & (HFS_STAT_MSG_TYPE>>8);
	if ( ( i == (HFS_STAT_TUNNEL>>8) ) ||	
		 ( i == (HFS_STAT_1042>>8) && hcf_encap( (wci_bufp)&p8[HFS_TYPE] ) != ENC_TUNNEL ) ) {
		flag = 0x0001;
											// perform decapsulation, subtract 8 from the length
		 									// note that MIC is not included in lenght, hence no comparable correction for MIC
		descp->buf_cntl.buf_dim[0] -= 8;
	}
#endif // HCF_ENCAPSULATION
	ifbp->IFB_MICRxRtn = update_mic_dummy;
#if (HCF_TYPE) & HCF_TYPE_SSN
											//if MIC present
	if ( p8[HFS_STAT] & HFS_STAT_MIC ) { 
		   									//.  set MIC sub-flag
		flag |= 0x02;
		ifbp->IFB_MICRxRtn = update_mic;
											//.  initialize LR for MIC calculation
		i = ((p8[HFS_STAT+1] & (HFS_STAT_MIC_KEY_ID>>8))>>00) + 5;// coincidentally no shift needed for i itself*/;
		*(hcf_32*)&ifbp->IFB_MICRx[0] = *(hcf_32*)&ifbp->IFB_MICKey[i];
		*(hcf_32*)&ifbp->IFB_MICRx[4] = *(hcf_32*)&ifbp->IFB_MICKey[i+2];
											//.  save rcvd MIC and add padding to rcvd msg
		for ( i = 0; i < 8; i++ ) {
			mic[i] = p8[ HFS_DAT + len  + i];
			p8[ HFS_DAT + len  + i] = mic_pad[i];
		}

		   									//.  skip LEN in MIC calculation
											//.  calculate MIC over next 8 bytes (either just 8 data bytes or
		   									//	   							   	  Encapsulation + E-II Type )
		for ( i = 0; i < (12+8)/4; i++ ) {
			MIC_RX_RTN( (hcf_32*)ifbp->IFB_MICRx, *p32++ );
			if ( i == 2 ) {					// splice 4*0 after Len/Type; skip L/T;
				MIC_RX_RTN( (hcf_32*)ifbp->IFB_MICRx, 0x00000000 );
				p32  = (hcf_32*)&p8[HFS_DAT];
			}
		}
	}
#endif // HCF_TYPE_SSN	
											//if MIC
											//.  calculate MIC over remaining data + padding 
											//if decapsulation
											//.  move remaining data forward in buffer
	for ( i = len / 4; i; i-- ) {
												/* Note that in the previous for-loop already the 8 bytes 
												 * after the L/T field, which are counted in len, are processed
												 * without affecting the i in this loop, so this loop consumes
												 * 5 to 8 bytes of the padding inserted after the 802.3 frame.
												 * Due to the up-to 3 bytes truncation caused by dividing len
												 * by 4, the padding is truncated to the appropriate amount of
												 * 0x00 after 0x5a, 0x00, 0x000, 0x000, 0x000
												 */
		MIC_RX_RTN( (hcf_32*)ifbp->IFB_MICRx, *p32++ );
												/* if len == 4*m+3, then moving len/4 times 2 words misses 
												 * 3 bytes at the end
												 * since len includes HFS_LEN field and 6 bytes SNAP header, 
												 * len is 8 bytes more than really need to be moved, 
												 * so at least 5 and most 8 bytes too much are moved in case
												 * of decapsulation
												 */
#if HCF_ENCAPSULATION == 0x0001
		if ( flag & 0x01 ) { *dp16++ = *sp16++; *dp16++ = *sp16++; }
#endif // HCF_ENCAPSULATION
	}
											//check MIC against received MIC
#if (HCF_TYPE) & HCF_TYPE_SSN
	if ( flag & 0x02 ) {
		for ( i = 0; i < 8 && ifbp->IFB_MICRx[i] == mic[i]; i++ ) /* nop */;
		if ( i != 8 ) rc = HCF_ERR_MIC;
	}
#endif // HCF_TYPE_SSN	
	ifbp->IFB_RxLen = descp->buf_cntl.buf_dim[0];
#endif // HCF_TYPE_USB

	return rc;
} /* hcf_rcv_msg */


/******************************************************************************************************************

.MODULE			hcf_send_msg
.DESCRIPTION	Transfers a message from Host to NIC and/or initiates transmission

.ARGUMENTS
  int hcf_send_msg( IFBP ifbp, DESC_STRCT *descp, hcf_16 tx_cntl )
  Card Interrupts disabled

.RETURNS
	HCF_SUCCESS
	IFB_DefunctStat
	HCF_ERR_TIME_OUT

.NARRATIVE

 Parameters:
  ifbp		address of the Interface Block
  descp		pointer to the descriptor list or NULL
  tx_cntl	indicates MAC-port and (Hermes) options

.DIAGRAM
 4:	for the normal case (i.e. no HFS_TX_CNTL_TX_DELAY option active), a fid is acquired via the
 	routine get_fid.  If no FID is acquired, the remainder is skipped without an error notification.
 	After all, the MSF is not supposed to call hcf_send_msg when no Resource is available.
 7: The ControlField of the TxFS is written.  Since strio can only return the fatal Defunct or "No NIC",
 	the return status can be ignored because when it fails, cmd_wait will fail as well.
	(see also the note on the need for a return code below).
	Note that HFS_TX_CNTL has different values for H-I, H-I/SSN and H-II and
	HFS_ADDR_DEST has different values for H-I (regardless of SSN) and H-II.
	By writing 17, 1 or 2 ( implying 16, 0 or 1 garbage word after HFS_TX_CNTL) the BAP just gets to 
	HFS_ADDR_DEST for H-I, H-I/SSN and H-II respectively.
10:	if neither encapsulation nor MIC calculation is needed, splitting the first fragment in two does not 
	really help but it makes the flow easier to follow to do not optimize on this difference

  hcf_send_msg checks whether the frame is an Ethernet-II rather than an "official" 802.3 frame. 
  The E-II check is based on the length/type field in the MAC header. If this field has a value larger than 
  1500, E-II is assumed. The implementation of this test fails if the length/type field is not in the first 
  descriptor.  If E-II is recognized, a SNAP header is inserted. This SNAP header represents either RFC1042 
  or Bridge-Tunnel encapsulation, depending on the return status of the support routine hcf_encap.
	
.NOTICE
  hcf_send_msg leaves the responsibility to only send messages on enabled ports at the MSF level.
  This is considered the strategy which is sufficiently adequate for all "robust" MSFs, have the least
  processor utilization and being still acceptable robust at the WCI !!!!!
	
  hcf_send_msg does not NEED a return value to report NIC absence or removal during the execution of 
  hcf_send_msg(), because the MSF and higher layers must be able to cope anyway with the NIC being removed 
  after a successful completion of hcf_send_msg() but before the actual transmission took place.
  To accommodate user expectations the current implementation does report NIC absence.
  Defunct blocks all NIC access and will (also) be reported on a number of other calls.

  hcf_send_msg does not check for transmit buffer overflow because the Hermes does this protection.
	In case of a transmit buffer overflow, the surplus which does not fit in the buffer is simply dropped.
	Note that this possibly results in the transmission of incomplete frames.

  After some deliberation with F/W team, it is decided that - being in the twilight zone of not
  knowing whether the problem at hand is an MSF bug, HCF buf, F/W bug, H/W malfunction or even something
  else - there is no "best thing to do" in case of a failing send, hence the HCF considers the TxFID ownership
  to be taken over by the F/W and hopes for an Allocate event in due time

.ENDOC				END DOCUMENTATION

***************************************************************************************************************/
/* figure out what the exact boundaries are to assert the length
 * HCFASSERT( len + sizeof(snap_header) + 2 <= HCF_MAX_MSG + (tx_cntl&HFS_TX_CNTL_TYPE_TKIP)/2, len )
 *			too bad that some of those MS peple try to outsmart normal people, otherwise above assert
 *			would compile without errors
 */			

int
hcf_send_msg( IFBP ifbp, DESC_STRCT *descp, hcf_16 tx_cntl )
{

int			rc = HCF_SUCCESS;
DESC_STRCT	*p = descp;				//working pointer
hcf_16 		fid = 0;				// 0-value causes USB to skip Tx-cmd						
hcf_16 		len;					// total byte count 
hcf_16 		i;

#if (HCF_TYPE) & HCF_TYPE_USB
DESC_STRCT	*p0 = descp;			//fixed pointer to "accumulating" descriptor in case of USB
	//p0->buf_cntl.buf_dim[0] = 0;
	usb_simp = (wci_recordp)(descp++->buf_addr);	//!!! note USB ONLY! ++ modifies parameter descp !!!
#else
hcf_32		prot_cnt = ifbp->IFB_TickIni;
	HCFASSERT( ifbp->IFB_RscInd || descp == NULL, 0 )
#endif // HCF_TYPE_USB

	HCFLOGENTRY( HCF_TRACE_SEND_MSG, tx_cntl )
 	HCFASSERT( ifbp->IFB_Magic == HCF_MAGIC, ifbp->IFB_Magic )
	HCFASSERT_INT
	HCFASSERT( ((long)descp & 3 ) == 0, (hcf_16)(long)descp )
	HCFASSERT( descp || tx_cntl & HFS_TX_CNTL_TX_DELAY, 0 )
	HCFASSERT( (tx_cntl & ~HFS_TX_CNTL_BITS) == 0, tx_cntl )

	if ( descp ) ifbp->IFB_TxFid = 0;				//cancel a pre-put message
									/* the following initialization code is redundant for a pre-put message
									 * but moving it inside the "if fid" logic makes the merging with the 
									 * USB flow awkward  
									 */
	ifbp->IFB_tx_tlen = 0; //needed independent of HCF_TYPE_SSN as long as send_msg_frag is (mis)used to workaround byte I/O
	ifbp->IFB_MICTxRtn = update_mic_dummy;
#if (HCF_TYPE) & HCF_TYPE_SSN
	if ( ifbp->IFB_MICKey[0] ) {		//initialize MIC-engine
		tx_cntl |= HFS_TX_CNTL_TYPE_TKIP | (ifbp->IFB_MICKey[0] - 1)<<8;  
		*(hcf_32*)&ifbp->IFB_MICTx[0] = *(hcf_32*)&ifbp->IFB_MICKey[ifbp->IFB_MICKey[0]];			
		*(hcf_32*)&ifbp->IFB_MICTx[4] = *(hcf_32*)&ifbp->IFB_MICKey[ifbp->IFB_MICKey[0]+2];			
		ifbp->IFB_MICTxRtn = update_mic;
	}
#endif // HCF_TYPE_SSN	
	IF_NOT_USB( if ( (fid = ifbp->IFB_TxFid) == 0  && ( fid = get_fid( ifbp ) ) != 0 ) )		/* 4 */
			/* skip the next compound statement if:
			   - pre-put message or
			   - no fid available (which should never occur if the MSF adheres to the WCI)
			 */
	{		// to match the closing curly bracket of above "if" in case of HCF_TYPE_USB
#if defined LLB	//!!!!MIC Debug Only, can't easily be merged with the other len calculation because BAP is modified
		len = 0;
		p = descp;
		do len += p->buf_cntl.buf_dim[0]; while ( p = p->next_desc_addr );
		(void)strio( ifbp, IO_OUT, fid, HFS_SWSUP, &len, 1 BE_PAR(0)); 
#endif //LLB	
		i = tx_cntl & (HFS_TX_CNTL_MASK | HFS_TX_CNTL_MIC_KEY_ID | HFS_TX_CNTL_TYPE_TKIP );
#if (HCF_TYPE) & HCF_TYPE_USB
		*usb_simp++ = i;													//add to USB frame 
#else
/*7*/	(void)strio( ifbp, IO_OUT, fid, HFS_TX_CNTL, &i, (HFS_ADDR_DEST - HFS_TX_CNTL)/2  BE_PAR(1) );
#endif // HCF_TYPE_USB		
		/* ;? Note that we still pay the expense of calculating MIC */
											//calculate total length 
		len = 0;
		p = descp;
		do len += p->buf_cntl.buf_dim[0]; while ( p = p->next_desc_addr );
			p = descp;
		HCFASSERT( len <= HCF_MAX_MSG, len )
		HCFASSERT( p->buf_cntl.buf_dim[0] >= 14, p->buf_cntl.buf_dim[0] )
											/* assume DestAddr/SrcAddr/Len/Type ALWAYS contained in 1st fragment
											 * otherwise life gets too cumbersome for MIC and Encapsulation !!!!!!!!
		if ( p->buf_cntl.buf_dim[0] >= 14 ) {	alternatively: add a safety escape !!!!!!!!!!!! }	*/
/*10*/										//write DA, SA with MIC calculation
			send_msg_frag( ifbp, p->buf_addr, 12 );
											//add quad-byte of 0x00 to MIC calculation
		MIC_TX_RTN( (hcf_32*)ifbp->IFB_MICTx, 0x00000000L );
											//if encapsulation needed
#if HCF_ENCAPSULATION == 0x0001
											//write length (with SNAP-header,Type, without DA,SA,Length ) no MIC calculation
		if ( ( snap_header[sizeof(snap_header)-1] = hcf_encap( &p->buf_addr[12] ) ) != ENC_NONE ) {
			OPW( HREG_DATA_0, CNV_END( len + (sizeof(snap_header) + 2) - ( 2*6 + 2 ) ) ); 
											//write splice with MIC calculation
			(void)send_msg_frag( ifbp, snap_header, sizeof(snap_header) );
			len += (sizeof(snap_header) + 2 );		/* in fact superfluous except for DCWA for pre-SSN	*/
			i = 12;						
		} else 
#endif // HCF_ENCAPSULATION
		{
			i = (hcf_16)p->buf_addr[12]*0x100 + p->buf_addr[13]; /*;?probably optimizable by moving body of hcf_encap 
														  in line and re-arranging testing and/or CNV_END use*/
			OPW( HREG_DATA_0, CNV_END( i ) ); 
			i = 14;
		}
											//complete 1st fragment starting with Type with MIC calculation
		send_msg_frag( ifbp, &p->buf_addr[i], p->buf_cntl.buf_dim[0] - i );
											//do the remaining fragments with MIC calculation
		while ( ( p = p->next_desc_addr ) != NULL ) { 
			HCFASSERT( ((long)p & 3 ) == 0, (hcf_16)(long)p )
			send_msg_frag( ifbp, p->buf_addr, p->buf_cntl.buf_dim[0] );
		}
											//pad message, finalize MIC calculation and write MIC and DCP to NIC
		len += 4 - ifbp->IFB_tx_tlen;	/* compensate bytes written in send_mic_dcp, 
									 * superfluous except for DCWA
									 * could bu used to determine p0->buf_cntl.buf_dim[0] for USB
									 * may be handy if we are going to support dynamically H-I WPA and non-WPA*/
		send_mic_dcp( ifbp );
		IF_USB(( p0->buf_cntl.buf_dim[0] = (char*)usb_simp - p0->buf_addr));
#if !( (HCF_TYPE) & HCF_TYPE_USB )
#if  !( (HCF_TYPE) & (HCF_TYPE_HII|HCF_TYPE_SSN ) )			//DCWA for pre-SSN
		(void)strio( ifbp, IO_IN, fid, len + HFS_ADDR_DEST, &i, 1 BE_PAR(1) );
		if ( i != 0xCAFE ) {
			HCFASSERT( DO_ASSERT, i )
			HCFASSERT( DO_ASSERT, len )
			// since DCP is not reported HCF_ERR_DCWA might just as well be skipped rc = HCF_ERR_DCWA;
			fid = 0;				//cheap way out to skip transmission
		} 
#endif // HCF_TYPE_HII | HCF_TYPE_SSN
		if ( tx_cntl & HFS_TX_CNTL_TX_DELAY ) {
			ifbp->IFB_TxFid = fid;
			fid = 0;
		}
	}
	if ( fid ) {
/*16*/	rc = cmd_exe( ifbp, HCMD_BUSY | HCMD_TX | HCMD_DCP | HCMD_RECL, fid );
		ifbp->IFB_TxFid = 0;
			/* probably this (i.e. no RscInd AND "HREG_EV_ALLOC") at this point in time occurs so infrequent, 
			 * that it might just as well be acceptable to skip this
			 * "optimization" code and handle that additional interrupt once in a while
			 */  
/*20*/	if ( ifbp->IFB_RscInd == 0 ) ifbp->IFB_RscInd = get_fid( ifbp );
#endif //HCF_TYPE_USB		
	}
	HCFLOGEXIT( HCF_TRACE_SEND_MSG )
	return rc;
} /* hcf_send_msg */

#if ! ((HCF_TYPE) & HCF_TYPE_USB)
/*******************************************************************************************************************

.MODULE			hcf_service_nic
.DESCRIPTION	Provides received message, handles (most) Hermes events

.ARGUMENTS
  hcf_service_nic( IFBP ifbp, wci_bufp bufp, int len )
  Card Interrupts disabled

.RETURNS
	HCF_SUCCESS
	HCF_ERR_MIC

.NARRATIVE

 Parameters:
  ifbp	address of the Interface Block
  bufp	address of buffer, sufficiently large to hold the first part of the RxFS up through HFS_TYPE
  len	lenghth in bytes of buffer specified by bufp

  MSF-accessible fields of Result Block
	IFB_RxLen			0 or Frame size
	IFB_MBInfoLen
	IFB_RscInd
	IFB_HCF_Tallies

	hcf_service_nic is primarily intended to be part of the Interrupt Service Routine.
	hcf_service_nic is presumed to neither interrupt other HCF-tasks nor to be interrupted by other HCF-tasks.
	A way to achieve this is to precede hcf_service_nic as well as all other HCF-tasks with a call to
	hcf_action to disable the card interrupts and, after all work is completed, with a call to hcf_action to
	restore (which is not necessarily the same as enabling) the card interrupts.
	In case of a polled environment, it is assumed that the MSF programmer is sufficiently familiar with the
	specific requirements of that environment to translate the interrupt strategy to a polled strategy.

	hcf_service_nic services the following Hermes events:
		HREG_EV_INFO		Asynchronous Information Frame
		HREG_EV_INFO_DROP	WMAC did not have sufficient RAM to build Unsolicited Information Frame
		HREG_EV_ALLOC		Asynchronous part of Allocation/Reclaim completed while out of resources at
							completion of hcf_send_msg/notify
		HREG_EV_RX			the detection of the availability of received messages
							including WaveLAN Management Protocol (WMP) message processing


	A received message can be either "NOS" messages or "WMP" (aka DUIF) messages.
	A WMP message is recognized by the Hermes and reported as such in the Status field of the Rx Frame
	Structure. A number of WMP messages are handled by the Hermes and are never passed to the Host at all.
	The error free WMP messages which are passed to the Host are stored in the MailBox (assuming one with
	sufficient free space is available). As a side effect, when IFB_MBInfoLen is zero, it is updated to
	reflect the size of the MailBox Information Block used to store the WMP message.
	;? If Monitor Mode is active, a WMP message SHOULD be considered a NOS message. Current implementation
	moves all ERRORFREE messages (also those addressed to other stations) into the MailBox. Frames
	containing errors are passed to the MSF as is (without decapsulation). Note that the
	Hermes does NOT pass those WMP frames to the host to which the Hermes generated a response.

	An available NOS messages is a message which:
	 -	is received by the LAN Controller with an OK status and which is not an WMP message
	 -	All messages (;?including WMP messages and messages with an error status) when Monitor Mode is enabled

	If a message is available, its length is reflected by the IFB_RxLen field of the IFB. This length
	reflects the data itself and the Destination Address, Source Address and DataLength/Type field but not the
	SNAP-header in case of decapsulation by the HCF.
	If no message is available, IFB_RxLen is zero.

  **Rx Buffer free strategy
	When hcf_service_nic reports the availability of a message, the MSF can access that message by means of 
	hcf_rcv_msg. It must be prevented that the LAN Controller writes new data in the NIC buffer before the MSF 
	is finished with the current message. The NIC buffer is returned to the LAN Controller when:
	 - hcf_rcv_msg is called or
	 - hcf_action with HCF_ACT_RX is called or
	 - hcf_service_nic is called again
	;? matters may get more complicated to explain if the strategy changes to have service_nic ACK the
	receiver when the complete frame fits in the lookahead buffer
	It can be reasoned that hcf_action( INT_ON ) should not be given before the MSF has completely processed 
	a reported Rx-frame. The reason is that the INT_ON action is guaranteed to cause a (Rx-)interrupt (the 
	MSF is processing a Rx-frame, hence the Rx-event bit in the Hermes register must be active). This 
	interrupt will cause hcf_service_nic to be called, which will cause the ack-ing of the "last" Rx-event 
	to the Hermes, causing the Hermes to discard the associated NIC RAM buffer.


.DIAGRAM
 1:	IFB_LinkStat is cleared, if a LinkStatus frame is received, IFB_LinkStat will be updated accordingly
 	by isr_info.
 2: IFB_RxLen must be cleared before the NIC presence check otherwise:
 	 -	this value may stay non-zero if the NIC is pulled out at an inconvenient moment. 
	 -	the RxAck on a zero-FID needs a zero-value for IFB_RxLen to work
	 Note that as side-effect of the hcf_action call, the remainder of Rx related info is re-initialized 
	 as well.
 4:	In case of Defunct mode, the information supplied by Hermes is unreliable, so the body of 
	hcf_service_nic is skipped
 	To prevent that hcf_service_nic reports bogus information to the MSF with all - possibly difficult to
	debug - undesirable side effects, it is paramount to check the NIC presence based on the Hermes 
	register HREG_SW_0.
	Note that in polled environments Card Removal is not detected by INT_OFF which makes the check in 
	hcf_service_nic even more important
 8:	The event status register of the Hermes is sampled
	The assert checks for unexpected events. 
	 - HREG_EV_INFO_DROP is explicitly excluded from the acceptable HREG_EV_STAT bits because it indicates 
	   a too heavily loaded system.
	 - HREG_EV_REPAIR_ACK is 0x0000 for H-I (and hopefully H-II.5)
	 - HREG_EV_RES is 0x0000 for H-II
	HREG_EV_TX_OK and HREG_EV_TX_EX are excepted if and only if HCF_EX_INT defined so at compile time.
 	The following activities are handled:
	 -	Alloc events are handled by hcf_send_msg (and notify). Only if there is no "spare" resource, the 
	 	alloc event is superficially serviced by hcf_service_nic to create a pseudo-resource with value
		0x001. This value is recognized by get_fid (called by hcf_send_msg and notify) where the real 
		TxFid is retrieved and the Hermes is acked and - hopefully - the "normal" case with a spare TxFid 
		in IFB_RscInd is restored.
	 -	Info drop events are handled by incrementing a tally
	 -	LinkEvent (including solicited and unsolicited tallies) are handled by procedure isr_info.
	 -	TxEx/TxOK (if selected at compile time) is handled by copying the significant part of the TxFS 
	 	into the mailbox, to allow the MSF to determine the MAC address associated with the failure/success. 
		Note the complication of the zero-FID protection sub-scheme in DAWA. 
		The Assert validates the HCF assumption about Hermes implementation upon which the range of 
		Pseudo-RIDs is based.
	Note, the Ack of all of above events is handled at the end of hcf_service_nic
14:	All the non-Rx/non-Cmd/non-Alloc activities are acknowledged. Combining all these acknowledgements to a 
	single place, is considered an optimization.
	Note that the Rx-acknowledgement is explicitly not included, as justified in "Rx Buffer free strategy" above.
	Note that the Cmd-acknowledgement is explicitly not included, because all command handling is handled
	in line.
	Note that the Alloc-acknowledgement is explicitly not included, as described just above
16:	If an Rx-frame is available, first the FID of that frame is read, including the complication of the 
	zero-FID protection sub-scheme in DAWA. Note that such a zero-FID is acknowledged at the end of 
	hcf_service_nic and that this depends on the IFB_RxLen initialization in the begin of hcf_service_nic.
	The Assert validates the HCF assumption about Hermes implementation upon which the range of 
	Pseudo-RIDs is based.
	Then the control fields up to the start of the 802.3 frame are read from the NIC into the lookahead buffer.
	The status field is converted to native Endianess. 
	The length is, after implicit Endianess conversion if needed, and adjustment for the size of the 
	802.3 MAC header, stored in IFB_RxLen.
	In MAC Monitor mode, 802.11 control frames with a TOTAL lenght of 14 are received.
	As a coincidence, it is a convenient I/F to add 14 ( space occupied by MAC addresses and Length) to 
	the Hermes reported payload. So the maintenance programmer is forewarned when considering changing
	this strategy. 
	No MIC calculation processes are associated with the reading of these Control fields.
	If the Hermes frame status reflects an error - which can only occur in promiscuous mode - none of the
	following tests is true, hence the frame is not further processed and control is passed back to the MSF
20:	WMP messages are processed by copying them to the MailBox. 
	In Monitor mode all messages are copied to the Mailbox and are dropped as far as the Protocol Stack
	is concerned
	Note: No filtering on Hermes engineering facilities, hence these frames (which should only be observed
	on Agere premises) end up in the MailBox !!!!!!!!
24:	If not in Monitor mode and if the message is not a WMP message, the HCF checks whether decapsulation 
	is needed i.e.:
	  - the Hermes reported Tunnel encapsulation or
	  - the Hermes reported 1042 Encapsulation and hcf_encap reports that the HCF would not have used
		1042 as the encapsulation mechanism
	The decapsulation check needs sufficient data to represent DA, SA, L, SNAP and Type which amounts to
	22 bytes. In MAC Monitor mode, 802.11 control frames with a smaller lenght are received. To prevent 
	that the implementation goes haywire, a check on the lenght is needed.
	The actual decapsulation takes place on the fly in the copying process by overwriting the SNAP header.
	Note that in case of decapsulation the SNAP header is not passed to the MSF, hence IFB_RxLen must be
	compensated for the SNAP header length
26:	This lenght test feels like superfluous robustness against malformed frames, but it turned out to be
	needed in the real (hostile) world.
28:	If Hermes reported MIC-presence, than the MIC engine is initialized with the appropraite key.
32:	The 12 in the no-SSN branch corresponds with rcv_msg_frag, 2 with IPW of the SSN branch
36:	The L-field and 6 byte SNAP header is discarded, so IFB_RxLen must be adjusted by 8.
	The Type field is the only word kept (after moving) of the just read 8 bytes, but it is moved to the
	L-field which was already incorporated in buf_addr simultanuously with the DA and SA. Therefo buf_addr 
	must also be adjusted by 8.
40: Determine how much of the frame (starting with DA) fits in the Lookahead buffer, then read the not-yet 
	read data into the lookahead buffer
44:	Since the complete message is copied from NIC RAM to PC RAM, the Rx can be acknowledged to the Hermes
	to optimize the flow ( a better chance to get new Rx data in the next pass through hcf_service_nic ).
	This acknowledgement can not be done via hcf_action( HCF_ACT_RX_ACK ) because this also clears 
	IFB_RxLEN thus corrupting the I/F to the MSF.


.ENDOC				END DOCUMENTATION

***************************************************************************************************************/
int
hcf_service_nic( IFBP ifbp, wci_bufp bufp, unsigned int len )
{

int			rc = HCF_SUCCESS;
hcf_16		i, stat;
wci_bufp	buf_addr;

	HCFLOGENTRY( HCF_TRACE_SERVICE_NIC, ifbp->IFB_IntOffCnt )
	HCFASSERT( ifbp->IFB_Magic == HCF_MAGIC, ifbp->IFB_Magic )
	HCFASSERT_INT
	HCFASSERT( bufp, len ) 
#if HCF_ENCAPSULATION == 0x0001
	HCFASSERT( len >= HFS_DAT + 2 + sizeof(snap_header), len )
#else
	HCFASSERT( len >= HFS_DAT + 2, len )
#endif // HCF_ENCAPSULATION

	ifbp->IFB_LinkStat = 0;																				/* 1*/
	hcf_action( ifbp, HCF_ACT_RX_ACK );																	/* 2*/
	if ( !(ifbp->IFB_CardStat & CARD_STAT_DEFUNCT) && IPW( HREG_SW_0) == HCF_MAGIC ) {					/* 4*/
		stat = IPW( HREG_EV_STAT );																		/* 8*/
		HCFASSERT( !( stat & ~( HREG_EV_TICK | HREG_EV_RES | HREG_EV_REPAIR_ACK	|
								/* HREG_EV_INFO_DROP |*/ HREG_EV_INFO | HREG_EV_CMD |
								  HREG_EV_ALLOC | HCF_EX_INT | HREG_EV_RX ) ), stat )
		if ( stat & HREG_EV_ALLOC && ifbp->IFB_RscInd == 0 ) ifbp->IFB_RscInd = 1;
		if ( stat & HREG_EV_INFO_DROP ) ifbp->IFB_HCF_Tallies.NoBufInfo++;
		if ( stat & HREG_EV_INFO ) isr_info( ifbp );
#if HCF_EX_INT & (HCF_EX_INT_TX_EX | HCF_EX_INT_TX_OK)
		if ( stat & (HREG_EV_TX_EX | HREG_EV_TX_OK) && 
			 ( ifbp->IFB_MB_FID = IPW( HREG_TX_COMPL_FID ) ) != 0 /*DAWA*/ ) {
#if defined LLB	//!!!!MIC Debug Only
			call the code temporarily dumped below this routine waiting to be turned into a routine;
#endif //LLB		
			put_info_mb( ifbp, CFG_EX_INT_TX, NULL, HFS_DAT/2 + 1 );
		}
#endif // HCF_EX_INT
/*16*/	if ( stat & HREG_EV_RX ) {
			if ( ( ifbp->IFB_RxFID = IPW( HREG_RX_FID ) ) != 0 ) { //if 0 then DAWA_ACK( RX );
				OPW( HREG_RX_FID, 0);	/*DAWA*/
				HCFASSERT( ifbp->IFB_RxFID < CFG_TX_FRAME, ifbp->IFB_RxFID)
				(void)strio(ifbp, IO_IN, ifbp->IFB_RxFID, 0, (wci_recordp)bufp, HFS_ADDR_DEST/2 BE_PAR(1) );
				ifbp->IFB_lap = buf_addr = bufp + HFS_ADDR_DEST;
				stat = *(wci_recordp)&bufp[HFS_STAT] & ( HFS_STAT_MSG_TYPE | HFS_STAT_ERR );
				ifbp->IFB_RxLen = (hcf_16)(bufp[HFS_DAT_LEN] + (bufp[HFS_DAT_LEN+1]<<8) + 2*6 + 2); 
#if defined HCF_MB_ON
#if defined HCF_MONITOR_MODE				
				if ( ifbp->IFB_Monitor ) {									
					if ( ifbp->IFB_Monitor < ifbp->IFB_RxLen ) ifbp->IFB_RxLen = ifbp->IFB_Monitor;
					stat = HFS_STAT_WMP_MSG;
				}
#endif // HCF_MONITOR_MODE				
/*20*/			if ( stat == HFS_STAT_WMP_MSG ) {									
					ifbp->IFB_MB_FID = ifbp->IFB_RxFID;
					put_info_mb( ifbp, CFG_WMP, NULL, (hcf_16)DIV_BY_2(ifbp->IFB_RxLen + HFS_ADDR_DEST + 2) );
					ifbp->IFB_RxLen = 0;		//Rx_Ack at the end of hcf_service_nic
				} else
/*24*/			
#endif // HCF_MB_ON
				{
					ifbp->IFB_MICRxRtn = update_mic_dummy;
					ifbp->IFB_rx_tlen = 0; //;?needed as long as you do not really skip MIC calculation
/*26*/				if ( ifbp->IFB_RxLen >= 2*6 + 2 + 6 + 2 ) {
													//if MIC received
#if (HCF_TYPE) & HCF_TYPE_SSN
/*28*/					if ( *(wci_recordp)&bufp[HFS_STAT] & HFS_STAT_MIC ) {
													//.  initialize MIC housekeeping
							ifbp->IFB_MICRxRtn = update_mic;
													/* coincidentally no shift needed for i itself */
							i = ( (*(wci_recordp)&bufp[HFS_STAT] & HFS_STAT_MIC_KEY_ID ) >>8 ) + 5;
							*(hcf_32*)&ifbp->IFB_MICRx[0] = *(hcf_32*)&ifbp->IFB_MICKey[i];
							*(hcf_32*)&ifbp->IFB_MICRx[4] = *(hcf_32*)&ifbp->IFB_MICKey[i+2];
						}
													//.  get DA,SA and calculate MIC
						rcv_msg_frag( ifbp, buf_addr, 12 );
													//.  add quad-byte of 0x00 to MIC calculation
//						ifbp->IFB_MICRxRtn( (hcf_32*)ifbp->IFB_MICRx, 0x00000000L );
						MIC_RX_RTN( (hcf_32*)ifbp->IFB_MICRx, 0x00000000L );
													//.  get Length without MIC calculation
						*(wci_recordp)(buf_addr+12) = IPW( HREG_DATA_0 );
#else
													//.  get DA,SA and Len/Type
/*30*/					rcv_msg_frag( ifbp, buf_addr, 12 + 2 );
#endif // HCF_TYPE_SSN	
						buf_addr += 12 + 2;
													// get next 8 bytes and calculate MIC over it
													 /* (either Encapsulation + E-II Type or just 8 data bytes */
#if HCF_ENCAPSULATION == 0x0001
						rcv_msg_frag( ifbp, buf_addr, 8 );
						buf_addr += 8;
						if ( stat == HFS_STAT_TUNNEL ||
							 ( stat == HFS_STAT_1042 && hcf_encap( (wci_bufp)&bufp[HFS_TYPE] ) != ENC_TUNNEL ) ) {
													//.  copy E-II Type to 802.3 LEN field
							bufp[HFS_LEN  ] = bufp[HFS_TYPE  ];
							bufp[HFS_LEN+1] = bufp[HFS_TYPE+1];
													//.  discard Snap by overwriting with data
/*36*/						ifbp->IFB_RxLen -= (HFS_TYPE - HFS_LEN); //;? - 2;							
							buf_addr -= ( HFS_TYPE - HFS_LEN );
						}
#endif // HCF_ENCAPSULATION
					}
					HCFASSERT( (int)ifbp->IFB_RxLen <= HCF_MAX_MSG + (bufp[HFS_STAT]&HFS_STAT_MIC)/2, ifbp->IFB_RxLen )
					HCFASSERT( (int)ifbp->IFB_RxLen <= HCF_MAX_MSG + (bufp[HFS_STAT]&HFS_STAT_MIC)/2, ifbp->IFB_RxFID )
					ifbp->IFB_lal = min( (hcf_16)(len - HFS_ADDR_DEST), ifbp->IFB_RxLen );
					rcv_msg_frag( ifbp, buf_addr, ifbp->IFB_lal - ( buf_addr - ( bufp + HFS_ADDR_DEST ) ) );
#if HCF_TYPE & HCF_TYPE_SSN  
					if ( ifbp->IFB_lal == ifbp->IFB_RxLen ) rc = check_mic( ifbp );
#endif //HCF_TYPE_SSN
				}
			}
/*44*/		if ( len - HFS_ADDR_DEST >= ifbp->IFB_RxLen ) ifbp->IFB_RxFID = 0;	
			/* IFB_RxFID is cleared, so  you do not get another Rx_Ack at next entry of hcf_service_nic */
			else stat &= (hcf_16)~HREG_EV_RX;	//don't ack Rx if processing not yet completed
		}
		stat &= (hcf_16)~( HREG_EV_CMD | HREG_EV_REPAIR_ACK | HREG_EV_ALLOC );
/*14*/	if ( stat ) DAWA_ACK( ifbp, stat );	/*DAWA*/															
	}
	HCFLOGEXIT( HCF_TRACE_SERVICE_NIC )
	return rc;
}/* hcf_service_nic */
#endif // HCF_TYPE_USB


#if defined LLB	//!!!!MIC Debug Only
													//read control and header info from RxFS into bufp
			(void)strio( ifbp, IO_IN, ifbp->IFB_MB_FID, 0, (wci_recordp)bufp, HFS_ADDR_DEST/2 BE_PAR(1) );
													//initialize MIC housekeeping
			ifbp->IFB_lap = buf_addr = bufp + HFS_ADDR_DEST;
			ifbp->IFB_rx_tlen = 0;					/* also needed to get the byte->word I/O correct */
//			ifbp->IFB_MICRxRtn = update_mic;
													/* coincidentally no shift needed for i itself */
			i = ( (*(wci_recordp)&bufp[HFS_TX_CNTL] & HFS_STAT_MIC_KEY_ID ) >>8 ) + 5;
			*(hcf_32*)&ifbp->IFB_MICRx[0] = *(hcf_32*)&ifbp->IFB_MICKey[i];
			*(hcf_32*)&ifbp->IFB_MICRx[4] = *(hcf_32*)&ifbp->IFB_MICKey[i+2];
													//if LLB length > than 12 bytes (which implies it is a real frame)
			i = bufp[HFS_SWSUP] + ((hcf_16)bufp[HFS_SWSUP+1]<<8);
			if ( i > 12 ) {
													//.  get DA,SA and calculate MIC
				rcv_msg_frag( ifbp, buf_addr, 12 );
													//.  get Length without MIC calculation
				*(wci_recordp)&buf_addr[12] = IPW( HREG_DATA_0 );
				buf_addr += 14;
				i -= 14;
			}
													// get remaining bytes and calculate MIC over it
			rcv_msg_frag( ifbp, buf_addr, i );
			ifbp->IFB_lap[-HFS_ADDR_DEST] = HFS_STAT_MIC;	//fake condition needed by check_mic
			rc = check_mic( ifbp );
#endif //LLB		



/**************************************************************************************************************
************************** H C F   S U P P O R T   R O U T I N E S ********************************************
**************************************************************************************************************/


#if 0 //just to try it out ! ((HCF_TYPE) & HCF_TYPE_USB)
/******************************************************************************************************************

.MODULE			alloc
.DESCRIPTION	allocates a (TX or Notify) FID

.ARGUMENTS
  void alloc( IFBP ifbp )

.RETURNS

.NARRATIVE

 Parameters:
  ifbp	address of the Interface Block

  ASSERTS:
  o fid == 0, which indicates cmd_exe failed or the alloc event did not occur in time

.DIAGRAM
 1:	execute the allocate command by calling cmd_exe
 2: wait till either the alloc event or a time-out occurs
 4: since alloc is only called after an Hermes initialize command but before the Hermes enable, a failing
	allocation is considered a H/W failure, hence the Miscellaneous Error tally is incremented. It is
	assumed that no more severe actions are needed because other malfunctions will yield appropriate
	error information to the MSF
 6: if the alloc event occurs,
 	- read the FID and save it in IFB_RscInd to be used as "spare FID"
 	- acknowledge the alloc event
 8:	do another "half" allocate to complete the "1.5 Alloc scheme"
.ENDOC				END DOCUMENTATION

***************************************************************************************************************/
void
alloc( IFBP ifbp )
{

hcf_32 prot_cnt = ifbp->IFB_TickIni;

	if ( cmd_exe( ifbp, HCMD_ALLOC, 0 ) == HCF_SUCCESS ) {											/* 1 */
/*2*/	HCF_WAIT_WHILE( (IPW( HREG_EV_STAT ) & HREG_EV_ALLOC) == 0 );							
/*4*/	if ( prot_cnt == 0 ) ifbp->IFB_HCF_Tallies.MiscErr++;									
		else {
			ifbp->IFB_RscInd = get_fid( ifbp ); //IPW( HREG_ALLOC_FID );
///			OPW( HREG_ALLOC_FID, 0 ); /*DAWA*/
///			S_DAWA_ACK( ifbp, HREG_EV_ALLOC );
/*8*/		cmd_exe( ifbp, HCMD_ALLOC, 0 );														
		}
	}
	HCFASSERT( ifbp->IFB_RscInd, 0)
	return;
}/* alloc */
#endif // HCF_TYPE_USB


/*******************************************************************************************************************

.MODULE			assert
.DESCRIPTION	filters assert on level and interfaces to the MSF supplied assert routine

.ARGUMENTS
  void assert( IFBP ifbp, unsigned int line_number, int q )

.RETURNS
  void

.NARRATIVE

 Parameters:
  ifbp			address of the Interface Block
  line_number	line number of the line which caused the assert
  q				qualifier, additional information which may give a clue about the problem

.DIAGRAM

.ENDOC				END DOCUMENTATION

**************************************************************************************************************/
#if defined HCF_ASSERT
void
assert( IFBP ifbp, unsigned int line_number, int q )
{

hcf_16	run_time_flag = ifbp->IFB_AssertLvl;

	if ( run_time_flag /* > ;?????? */ ) { //prevent recursive behavior, later to be extended to level filtering
		if ( ifbp->IFB_AssertRtn ) {
			ifbp->IFB_AssertRtn( (wci_bufp)assert_strct.val, line_number, ifbp->IFB_AssertTrace, q );
		}
#if HCF_ASSERT & 0x0002
		if ( ifbp->IFB_IOBase ) {	//;? without this it crashes if hcf_disable at entry ASSERT catches
			IPW( HREG_SW_2 );			//useless read flags trace record as Assert record
			OPW( HREG_SW_2, line_number );
			OPW( HREG_SW_2, ifbp->IFB_AssertTrace );
			OPW( HREG_SW_2, q );
		}
#endif // HCF_ASSERT & 0x0002

#if defined HCF_MB_ON && HCF_ASSERT & 0x0004
		ifbp->IFB_AssertLvl = 0;									// prevent recursive behavior
		assert_strct.trace = ifbp->IFB_AssertTrace;
		assert_strct.qualifier = (hcf_16)q;
		assert_strct.line_number = (hcf_16)line_number;
		put_info_mb( ifbp, CFG_MB_ASSERT, (wci_recordp)&assert_strct.trace, assert_strct.len );
		ifbp->IFB_AssertLvl = run_time_flag;						// restore appropriate filter level
#endif // !HCF_MB_ON && HCF_ASSERT & 0x0004
	}
}/* assert */
#endif // HCF_ASSERT

#if ((HCF_TYPE) & (HCF_TYPE_USB | HCF_TYPE_HII) ) == 0
/*******************************************************************************************************************

.MODULE			aux_cntl
.DESCRIPTION	Prepares/terminates access via AUX-mechanism to the NIC RAM.

.ARGUMENTS
  int aux_cntl( IFBP ifbp, int cmd )

.RETURNS
	HCF_SUCCESS
	HCF_ERR_DEFUNCT_AUX

.NARRATIVE

 Parameters:
  ifbp	address of the Interface Block
  cmd
 		HREG_CNTL_AUX_ENA_CNTL	enables AUX-port access mechanism
 		HREG_CNTL_AUX_DIS_CNTL	disables ,,  ,,     ,,      ,,

  Description:
	Prepares/terminates access via AUX-mechanism to the NIC RAM.

.DIAGRAM

 5:	the code "OUT_PORT_WORD( i, IN_PORT_WORD( i ) & ~HREG_CNTL_AUX_ENA | HREG_CNTL_AUX_ENA_CNTL );"
 	could just as well have been written as:
				t = IN_PORT_WORD( i ) & ~HREG_CNTL_AUX_ENA;
				OUT_PORT_WORD( i, t | HREG_CNTL_AUX_ENA_CNTL );
	The compiler is in "default release" configuration (minimal size, full optimization)
	clever enough to see that in the "clear" form, t is overwritten with TickIni after
	its only use in OUT_PORT_WORD so it is not saved in [bp-2] (which happens to be
	the location of t).  However the compiler moves the value of t into register cx
	and out again for no good reason. In the obscure form the compiler leaves the
	value of t in register ax throughout the process.  I guess this is typical Micro$oft.

.ENDOC				END DOCUMENTATION

***************************************************************************************************************/
int
aux_cntl( IFBP ifbp, hcf_16 cmd )
{

int		rc = HCF_SUCCESS;
hcf_32	prot_cnt = ifbp->IFB_TickIni;
hcf_io	io_port = (hcf_io/*;?NDIS*/)ifbp->IFB_IOBase + HREG_CNTL;

	OPW( HREG_PARAM_0, AUX_MAGIC_0 );
	OPW( HREG_PARAM_1, AUX_MAGIC_1 );
	OPW( HREG_PARAM_2, AUX_MAGIC_2 );
	OUT_PORT_WORD( io_port,
				   ( IN_PORT_WORD( io_port ) & (hcf_16)~(HREG_CNTL_AUX_ENA|HREG_CNTL_AUX_DSD) ) | cmd );/* 5 */
	cmd = (hcf_16)(cmd & HREG_CNTL_AUX_ENA_CNTL ? HREG_CNTL_AUX_ENA_STAT : HREG_CNTL_AUX_DIS_STAT);
	HCF_WAIT_WHILE( (IN_PORT_WORD( io_port ) & HREG_CNTL_AUX_ENA) != cmd );
	if ( prot_cnt == 0 ) {
		ifbp->IFB_HCF_Tallies.MiscErr++;
		rc = ifbp->IFB_DefunctStat = HCF_ERR_DEFUNCT_AUX;
		ifbp->IFB_CardStat |= CARD_STAT_DEFUNCT;
	}
	return rc;
}/* aux_cntl */
#endif //HCF_TYPE_USB, HCF_TYPE_HII


#if ! ((HCF_TYPE) & HCF_TYPE_USB)
/******************************************************************************************************************

.MODULE			calibrate
.DESCRIPTION	calibrates the S/W protection counter against the Hermes Timer tick

.ARGUMENTS
  void calibrate( IFBP ifbp )

.RETURNS
	N.A.

.NARRATIVE

 Parameters:
  ifbp	address of the Interface Block

  Description: calibrates the S/W protection counter against the Hermes Timer tick
	IFB_TickIni is the value used to initialize the S/W protection counter such that the
	expiration period more or less independent of the processor speed. If IFB_TickIni is not yet
	calibrated, it is done now. This calibration is "reasonably" accurate because the Hermes is in a
	quiet state as a result of the Initialize command.

.DIAGRAM

 1:	IFB_TickIni is initialized at INI_TICK_INI by hcf_connect. If calibrate succeeds, IFB_TickIni is 
 	guaranteed to be changed. As a consequence there will be only 1 shot at calibration (regardless of
	the number of init calls) under normal circumstances.
 2:	Calibration is done HCF_PROT_TIME times. This diminish the effects of jitter and interference, 
 	especially in a pre-emptive environment. 
 	Each cycle is limited to at most INI_TICK_INI samples of the TimerTick status of the Hermes. 
	Since the start of calibrate is unrelated to the Hermes Internal Timer, the first
 	interval may last from 0 to the normal interval, all subsequent intervals should be the full length of
 	the Hermes Tick interval. The Hermes Timer Tick is not reprogrammed by the HCF, hence it is running
 	at the default of 10 k microseconds.
 3: Acknowledge the Timer Tick Event. For the first time through the loop there may or there may not be
 	a Timer Tick Event, this does not influence the observation made under 2 above.
 4:	If no Timer Tick Event occurred before the protection counter expired, reset IFB_TickIni to INI_TICK_INI,
 	set the defunct bit of IFB_CardStat (thus rendering the Hermes inoperable) and exit the calibrate routine.
	
.NOTICE
 o  Although there are a number of viewpoints possible, calibrate() uses now as error strategy that a single
	failure of the Hermes TimerTick is considered fatal. 
 o	If the Timer Tick Event is ALWAYS present (e.g. due to an error, we read 0xFFFF continuously), 
	IFB_TickIni stays zero so neither cmd_exe nor strio will be able to execute. This is considered
	adequate protection against this unlikely event.
 o  The Hermes does not restart nor reset its timing logic at an Initialize command. This implies that the
 	very first timing period measured after an Initialize command may be anything between 0 and whatever
 	the Hermes Timer was programmed by means of CFG_TICK_TIME. In the past the HCF programmed the Hermes
 	timer to 1 second. This definitely jeopardizes the calibration as implemented in calibrate().
 	By executing calibrate() only once in the lifetime of the driver would (supposedly) solve the problem.
 	This solution is not robust, due to the fragilIty of the Hermes implementation of the Initialize, Disable
 	and Enable commands as well as due to MSF sequences of hcf_connect, hcf_disconnect.
 o	There is no hard and concrete time-out value defined for Hermes activities. The 2.5 seconds is believed
 	to be sufficiently "relaxed" for real life and to be sufficiently short to be still useful in an
 	environment with humans.
 o	To diminish the chance that in a pre-emptive environment IFB_TickIni is calibrated too low because the HCF
	just happens to loose control during this calibration, the calibration is performed 10 times and the
	largest value is used.
 o	In case of failure, IFB_TickIni ends up as INI_TICK_INI, which renders the HCF inoperable

.ENDOC				END DOCUMENTATION

***************************************************************************************************************/
void
calibrate( IFBP ifbp )
{

int		cnt = 8;
hcf_32	prot_cnt;

	HCFTRACE( ifbp, HCF_TRACE_CALIBRATE );
	if ( ifbp->IFB_TickIni == INI_TICK_INI ) {													/* 1 */
/*2*/	ifbp->IFB_TickIni = 0;																		
			while ( cnt-- ) {
				prot_cnt = INI_TICK_INI;
/*3*/			OPW( HREG_EV_ACK, HREG_EV_TICK );													
				while ( (IPW( HREG_EV_STAT ) & HREG_EV_TICK) == 0 && --prot_cnt ) {
					ifbp->IFB_TickIni++;
				}
/*4*/			if ( prot_cnt == INI_TICK_INI ) {										
					ifbp->IFB_TickIni = INI_TICK_INI;
					ifbp->IFB_DefunctStat = HCF_ERR_DEFUNCT_TIMER;
					ifbp->IFB_CardStat |= CARD_STAT_DEFUNCT;
					HCFASSERT( DO_ASSERT, ifbp->IFB_CardStat )
				}
			}
/*8*/	ifbp->IFB_TickIni *= HCF_PROT_TIME/8;															
	}
	HCFTRACE( ifbp, HCF_TRACE_CALIBRATE | HCF_TRACE_EXIT );
} /* calibrate */
#endif // HCF_TYPE_USB




#if defined MSF_COMPONENT_ID
/******************************************************************************************************************

.MODULE			check_comp
.DESCRIPTION	Checks compatibility of the HCF as actor with a specific the NIC supplier feature

.ARGUMENTS
  int check_comp( IFBP ifbp, CFG_RANGES_STRCT *p, hcf_16 type )

.RETURNS
  0		incompatible
  <>0	low order byte: Bottom		high order byte: High

.NARRATIVE

 Parameters:
  ifbp	address of the Interface Block
  *p	address of the actor specification
  type	supplier specification

 Description: Check_comp is a support routine to check the compatibility of the HCF as actor
  with a specific feature supplied by the NIC
  It is used for the following features
  o cfg_drv_act_ranges_pri
  o cfg_drv_act_ranges_hsi
  o cfg_drv_act_ranges_sta

 Remarks:
  o For backward compatibility with NICs which do not support the HSISupRange LTV, the HCF will fake an
  	HSISupRange if the Hermes indicates "not available"

.DIAGRAM
 1:	0x000 as initial value for the unsigned variable i to cater for the case where the RID corresponding
	with type can not be retrieved
 2:	The supplier information is requested by building an LTV with the appropriate Type field (as specified
 	by parameter type) and passing this LTV to hcf_get_info
 4:	Old NICs do not support the HSISupRange, therefor this LTV is faked if and only if the F/W gives a
 	"does not exist" return status (L of LTV = 0)
	As a negative consequence of this choice to cope with old NICs, this logic to fake acceptance of those old
	NICs, must be removed when the Bottom Compatibility is raised above 1.
	The "dynamic" approach to replace the statement "if ( type == CFG_NIC_HSI_SUP_RANGE ) i = 0;" with
	"if ( type == CFG_NIC_HSI_SUP_RANGE && cfg_drv_sup_range_hsi.variant[0].bottom == 1 ) i = 0;"
	is not suitable because cfg_drv_sup_range_hsi does not exist when MSF_COMPONENT_ID is not defined.
 8:	The retrieved (or counterfeited) supplier structure is together with actor parameter, passed to the
 	WCI function mmd_check_comp to actually check the compatibility.

.ENDOC				END DOCUMENTATION

***************************************************************************************************************/
hcf_16 check_comp( IFBP ifbp, CFG_RANGES_STRCT *actp, hcf_16 type )
{

hcf_16				i = 0;																				/* 1 */
static CFG_RANGES_STRCT	BASED sup;			//;?Is this use/location O.K for Dos

	sup.len = sizeof(CFG_RANGES_STRCT)/sizeof(hcf_16)-1;												/* 2 */
	sup.typ = type;

	if ( hcf_get_info( ifbp, (LTVP)&sup ) == HCF_SUCCESS ) {
////??! is this the right strategy
//#if !defined HCF_HSI_VAR_0					// do not fake HSI for Controlled Deployment
//										// when Primary Variant 1 is no longer supported, faking can be removed
//		if ( sup.len == 0 && type == CFG_NIC_HSI_SUP_RANGE ) {
//			sup.len = sizeof(CFG_RANGES_STRCT)/sizeof(hcf_16)-1;										/* 4 */
//			sup.role = COMP_ROLE_SUPL;
//			//sup.id = don't care;
//			sup.variant[0].number = sup.variant[0].top = sup.variant[0].bottom = 1;
//		}
//#endif //HCF_HSI_VAR

#if 1 //;?rlav
		if ( mmd_check_comp( actp, (CFG_RANGES_STRCT /*NEAR */ * )&sup ) ) {	//cast needed when compiling as *.cpp under DOS ODI
			i = CNV_LITTLE_TO_INT(sup.variant[0].bottom) | (CNV_LITTLE_TO_INT(sup.variant[0].top) << 8);
		}
#else // 0
	i = 0xFF00;
#endif // 0
	}
	return i;
}/* check_comp */

#endif //MSF_COMPONENT_ID


#if (HCF_TYPE) & HCF_TYPE_SSN && ! ((HCF_TYPE) & HCF_TYPE_USB)
/*******************************************************************************************************************

.MODULE			check_mic
.DESCRIPTION	verifies the MIC of a received non-USB frame

.ARGUMENTS
  int check_mic( IFBP ifbp )

.RETURNS
	HCF_SUCCESS		
	HCF_ERR_MIC		

.NARRATIVE

 Parameters:
  ifbp		address of the Interface Block

  Description:
 4: test whether or not a MIC is reported by the Hermes
 6:	presume ccumulator IFB_rx_32 contains either 4 garabage bytes or 1, 2 or 3 final data bytes and 3, 2 or 1 
 	bytes of the received MIC.  The loop:
	 - saves the 4 garbage bytes or the 3, 2, or 1 MIC bytes in x8.
	 - append 4, 3, 2, or 1 bytes of the padding sequence 0x5A000000 to the final data bytes (if any) in IFB_rx_32
	In case 4 garbage bytes are saved in x8, the index i is cleared, causing the MIC bytes read at point 12
	to be deposited starting at the begin of x8. In the other 3 cases, the index i is not modified, causing
	the MIC bytes to be appended.
10:	calculate the MIC over the first 4 padding bytes or over the last data byte(s) + padding	
11:	calculate the MIC over 4 more 0x00 padding bytes
	Net effect: 0x5a and at least 4 times 0x00 and at most 7 times 0x00 is appended to the message.
12:	in the while loop at point 6, at most 3 bytes of the received MIC are read, now read the missing 
	5, 6, 7 or 8 bytes.
	Since words are read this corresponds with 3, 3, 4 or 4 words readed at byte offset 3, 2, 1 or 0 in x8[],
	resulting in a total of 9, 8, 9 and 8 bytes in x8[]
14:	the calculated MIC and the received MIC are compared, the return status is set when there is a mismatch

.DIAGRAM

.NOTICE
The check_mic algorithm is intimately interwoven with lookahead mechanism (IFB_lap) and
byte-gathering logic (IFB_rx_tlen) that despite the communality with the USB-specific flow
(splicing in the padding and the actual compare itself), the USB-specific check is integrated
into hcf_rcv_msg.
.ENDOC				END DOCUMENTATION

***************************************************************************************************************/
int
check_mic( IFBP ifbp )
{
int		rc = HCF_SUCCESS;
int		i = 0;
union { hcf_32 x32; hcf_16 x16[4]; hcf_8 x8[8+1]; } x;	//* area to save rcvd MIC + 1 extra (see *12 )
hcf_8	*p;
hcf_16	tmpvar;
			
	if ( ifbp->IFB_lap[-HFS_ADDR_DEST] & HFS_STAT_MIC ) {										/* 4 */
/*6*/	while ( ifbp->IFB_rx_tlen < 4 ) {																
			x.x8[i] = ifbp->IFB_rx_32[ifbp->IFB_rx_tlen];
				ifbp->IFB_rx_32[ifbp->IFB_rx_tlen++] = mic_pad[i++];
			}
		i &= 0x03;	//if ifbp->IFB_rx_tlen was 0, then still 8 MIC bytes must be read
//*10*/	ifbp->IFB_MICRxRtn( (hcf_32*)ifbp->IFB_MICRx, *(hcf_32*)&ifbp->IFB_rx_32[0] );				
//*11*/	ifbp->IFB_MICRxRtn( (hcf_32*)ifbp->IFB_MICRx, 0x00000000 );													
/*10*/	MIC_RX_RTN( (hcf_32*)ifbp->IFB_MICRx, *(hcf_32*)&ifbp->IFB_rx_32[0] );				
/*11*/	MIC_RX_RTN( (hcf_32*)ifbp->IFB_MICRx, 0x00000000 );													
/*12*/	while ( i < 8 ){																	
			tmpvar = IPW( HREG_DATA_0 );
#if defined HCF_BIG_ENDIAN
			x.x8[i++] = tmpvar>>8;
			x.x8[i++] = tmpvar&0xFF;
#else
			x.x8[i++] = tmpvar&0xFF;
			x.x8[i++] = tmpvar>>8;
#endif // HCF_BIG_ENDIAN
		}
/*14*/	for ( i = 0; i < 8; i++ ) {															
			if ( x.x8[i] != ifbp->IFB_MICRx[i] ) rc = HCF_ERR_MIC;
		}
	}
	return rc;
} // check_mic
#endif // HCF_TYPE_SSN / HCF_TYPE_USB


#if ! ((HCF_TYPE) & HCF_TYPE_USB)
/**************************************************************************************************************

.MODULE			cmd_cmpl
.DESCRIPTION	waits for Hermes Command Completion

.ARGUMENTS
  int cmd_cmpl( IFBP ifbp )

.RETURNS
	IFB_DefunctStat
	HCF_ERR_TIME_OUT
	HCF_ERR_DEFUNCT_CMD_SEQ
	HCF_SUCCESS

.NARRATIVE

 Parameters:
  ifbp		address of the Interface Block

  Description: 

.DIAGRAM

2:	Once cmd_cmpl is called, the Busy option bit in IFB_Cmd must be cleared
4:	If Status register and command code don't match either:
	 - the Hermes and Host are out of sync ( a fatal error) 
	  - error bits are reported via the Status Register. 
	Out of sync is considered fatal and brings the HCF in Defunct mode
 	Errors reported via the Status Register should be caused by sequence violations in Hermes command
 	sequences and hence these bugs should have been found during engineering testing. Since there is no
 	strategy to cope with this problem, it might as well be ignored at run time. Note that for any
 	particular situation where a strategy is formulated to handle the consequences of a particular bug
 	causing a particular Error situation reported via the Status Register, the bug should be removed
 	rather than adding logic to cope with the consequences of the bug.
	There have been HCF versions where an error report via the Status Register even brought the HCF in
	defunct mode (although it was not yet named like that at that time). This is particular undesirable
	behavior for a general library. 
	Simply reporting the error (as "interesting") is debatable. There also have been HCF versions with this
	strategy using the "vague" HCF_FAILURE code.
	The error is reported via:
	 - MiscErr tally of the HCF Tally set
	 - the (informative) fields IFB_ErrCmd and IFB_ErrQualifier
	 - the assert mechanism
6: 	This is a workaround for (supposedly legacy) H-I code where in some F/W versions the download would 
	do an implicit initialize, thus clearing the Command register.
8:	Here the Defunct case and the Status error are seperately treated


.ENDOC				END DOCUMENTATION

***************************************************************************************************************/
int
cmd_cmpl( IFBP ifbp )
{

hcf_32	prot_cnt = ifbp->IFB_TickIni;
int		rc = HCF_SUCCESS;
hcf_16	stat;

	ifbp->IFB_Cmd &= ~HCMD_BUSY;												/* 2 */
	HCF_WAIT_WHILE( (IPW( HREG_EV_STAT) & HREG_EV_CMD) == 0 );					/* 4 */
	stat = IPW( HREG_STAT );
	if ( prot_cnt == 0 ) {
		ifbp->IFB_HCF_Tallies.MiscErr++;
		rc = HCF_ERR_TIME_OUT;
		HCFASSERT( DO_ASSERT, ifbp->IFB_Cmd )
	} else {
		DAWA_ACK( ifbp, HREG_EV_CMD );
/*4*/	if ( stat != (ifbp->IFB_Cmd & HCMD_CMD_CODE) ) {
#if !((HCF_TYPE) & HCF_TYPE_HII)
/*6*/		if ( ifbp->IFB_Cmd != HCMD_PROGRAM ) 
#endif // HCF_TYPE_HII
/*8*/		if ( ( (stat ^ ifbp->IFB_Cmd ) & HCMD_CMD_CODE) != 0 ) {				
				rc = ifbp->IFB_DefunctStat = HCF_ERR_DEFUNCT_CMD_SEQ;
				ifbp->IFB_CardStat |= CARD_STAT_DEFUNCT;
			}
			ifbp->IFB_HCF_Tallies.MiscErr++;
			ifbp->IFB_ErrCmd = stat;
			ifbp->IFB_ErrQualifier = IPW( HREG_RESP_0 );
			HCFASSERT( DO_ASSERT, ifbp->IFB_Cmd )
			HCFASSERT( DO_ASSERT, ifbp->IFB_ErrCmd )
			HCFASSERT( DO_ASSERT, ifbp->IFB_ErrQualifier )
		}
	}
	return rc;
} // cmd_cmpl
#endif // HCF_TYPE_USB


/**************************************************************************************************************

.MODULE			cmd_exe
.DESCRIPTION	Executes synchronous part of Hermes Command and - optionally - waits for Command Completion

.ARGUMENTS
  int cmd_exe( IFBP ifbp, int cmd_code, int par_0 )

.RETURNS
	IFB_DefunctStat
	HCF_ERR_DEFUNCT_CMD_SEQ
	HCF_SUCCESS
	HCF_ERR_TO_BE_ADDED	<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<

.NARRATIVE

 Parameters:
  ifbp		address of the Interface Block
  cmd_code
  par_0

  Description: Executes synchronous Hermes Command and waits for Command Completion

.NARRATIVE  
	The general HCF strategy is to wait for command completion. As a consequence:
 	 - the read of the busy bit before writing the command register is superfluous
	 - the Hermes requirement that no Inquiry command may be executed if there is still an unacknowledged 
	   Inquiry command outstanding, is automatically met.
	The Tx command uses the "Busy" bit in the cmd_code parameter to deviate from this general HCF strategy. 
	The idea is that by not busy-waiting on completion of this frequently used command the processor 
	utilization is diminished while using the busy-wait on all other seldom used commands the flow is kept
	simple.
	
.DIAGRAM

 1:	skip the body of cmd_exe when in defunct mode or when - based on the S/W Support register write and 
 	read back test - there is apparently no NIC.
	Note: we gave up on the "old" strategy to write the S/W Support register at magic only when needed. Due 
	to the intricateness of Hermes F/W varieties ( which behave differently as far as corruption of the 
	S/W Support register is involved), the increasing number of Hermes commands which do an implicit
	initialize (thus modifying the S/W Support register) and the workarounds of some OS/Support S/W induced
	aspects (e.g. the System Soft library at WinNT which postpones the actual mapping of I/O space up to
	30 seconds after giving the go-ahead), the "magic" strategy is now reduced to a simple write and read
	back. This means that problems like a bug tramping over the memory mapped Hermes registers will no
	longer be noticed as side effect of the S/W Support register check.
 2: check whether the preceeding command skipped the busy wait and if so, check for command completion
 4:	cope with the behavior of some F/W versions to do an implicit Initialize on commands like Download, 
 	Initialize, Program

.NOTICE
.ENDOC				END DOCUMENTATION

***************************************************************************************************************/

int
cmd_exe( IFBP ifbp, hcf_16 cmd_code, int par_0 )	//if HCMD_BUSY of cmd_code set, then do NOT wait for completion
{

int		rc;

	HCFLOGENTRY( HCF_TRACE_CMD_EXE, cmd_code )
	HCFASSERT( (cmd_code & HCMD_CMD_CODE) != HCMD_TX || cmd_code & HCMD_BUSY, cmd_code ) //Tx must have Busy bit set
#if (HCF_TYPE) & HCF_TYPE_USB
	rc = put_usb_nic_cmd( ifbp, cmd_code &~HCMD_BUSY, (hcf_16)par_0, 0, 0 );
#else // HCF_TYPE_USB
	OPW( HREG_SW_0, HCF_MAGIC );
	if ( IPW( HREG_SW_0) == HCF_MAGIC ) rc = ifbp->IFB_DefunctStat;						/* 1 */
	else rc = HCF_ERR_NO_NIC;
	if ( rc == HCF_SUCCESS ) {
		//;?is this a hot idea, better MEASURE performance impact
/*2*/	if ( ifbp->IFB_Cmd & HCMD_BUSY ) rc = cmd_cmpl( ifbp );
		OPW( HREG_PARAM_0, par_0 );
//HWi! Changed back till know who will change (HCF or FW)
//HWi!		OPW( HREG_CMD, cmd_code /*&~HCMD_BUSY*/ );
		OPW( HREG_CMD, cmd_code &~HCMD_BUSY );
		ifbp->IFB_Cmd = cmd_code;
		if ( (cmd_code & HCMD_BUSY) == 0 ) {	//;?is this a hot idea, better MEASURE performance impact
			rc = cmd_cmpl( ifbp );		//;?see note below 
/*4*/		OPW( HREG_SW_0, HCF_MAGIC );
		}
	}
#endif // HCF_TYPE_USB
	HCFASSERT( rc == HCF_SUCCESS, cmd_code )
	HCFLOGEXIT( HCF_TRACE_CMD_EXE )
	return rc;

		//this is the "note below"
		//;?definitely a complication, init wipes most of the IFB, then ini_hermes uses
		//cmd_exe to initialize while apparently a send is still not finished, i.e. cmd_reg contains 800b
		//when checked at cmd completion rather than the expected 0
} // cmd_exe


/******************************************************************************************************************

.MODULE			debug_trigger
.DESCRIPTION	Trace Analyzer support

.ARGUMENTS
  int debug_trigger( IFBP ifbp, hcf_16 type, hcf_16 value )

.RETURNS
  0 (FALSE)

.NARRATIVE

 Parameters:
  ifbp	address of the Interface Block
  type
  value

  debug_trigger has a long history of being adopted to the tools which at a particular moment
  seem adequate to handle the (debug) job at hand.
  The name debug_trigger is not very adequate for the current implementation.

  In the current implementation, debug_trigger is invoked via the HCFTRACE and HCFTRACEVALUE macros.
  
  In the current implementation, debug_trigger writes information to the S/W support Hermes register.
  These registers are R/W registers which do not influence the behavior of the Hermes. The Host will
  read back whatever value the host has lastly written to it. These registers are made available to have
  a means of communication for two concurrent S/W modules which intend to access the same NIC but are
  unaware of one another, an example of this used to be the WaveLAN-I Netware server drivers. In addition
  these registers can be used to write arbitrary information to be picked up via a StateAnalyzer hooked up
  to a FAT NIC. This way we can follow the flow through the HCF and profile the HCF code.

  Earlier versions used the parallel port to talk to the Podalyzer.

  Depending on the requirements of the moment and the facilities offered by the tool (e.g. 8 bit port
  versus 16 bit ports), the I/F and the manipulation of the parameter change

.DIAGRAM
.ENDOC				END DOCUMENTATION

***************************************************************************************************************/

#if HCF_DEBUG //any logging feauture (currently only HREG_SW#)
int
debug_trigger( IFBP ifbp, hcf_16 type, hcf_16 value )
{
	if ( ifbp->IFB_IOBase && ifbp->IFB_Magic == HCF_MAGIC ) {	//some precaution won't harm
		OPW( type, value );
	}
	return 0;			//return 0 to assure catching HCF_ASSERT
} /* debug_trigger */
#endif // HCF_DEBUG & 0x0018


#if ! ((HCF_TYPE) & HCF_TYPE_USB)
#if (HCF_TYPE) & HCF_TYPE_HII
/*********************************************************************************************************************

.MODULE			download
.DESCRIPTION

.ARGUMENTS
  int download( IFBP ifbp, LTVP ltvp )

.RETURNS

.NARRATIVE

 Parameters:
	ifbp		address of the Interface Block
	ltvp		specifies the pseudo-RID (as defined by WCI)

.DIAGRAM
 1:	First, Ack everything to unblock a (possibly) blocked cmd pipe line
	Note 1: it is very likely that an Alloc event is pending and very well possible that a (Send) Cmd event 
	is pending
	Note 2: it is assumed that this strategy takes away the need to ack every conceivable event after an 
	Hermes Initialize


*********************************************************************************************************************/
#if defined HCF_DLNV 
not yet supported;
#endif // HCF_DLNV
#if defined HCF_DLV
int
download(IFBP ifbp, LTVP ltvp )
{

hcf_16				i;
int					rc = HCF_SUCCESS;
CFG_PROG_STRCT FAR 	*p = (CFG_PROG_STRCT FAR *)ltvp;	
wci_bufp			cp;
hcf_io				io_port = (hcf_io/*;?NDIS*/)ifbp->IFB_IOBase + HREG_AUX_DATA;

	HCFLOGENTRY( HCF_TRACE_DL, p->typ )
													//if initial "program" LTV
	if ( ifbp->IFB_DLMode == CFG_PROG_STOP && p->mode == CFG_PROG_VOLATILE) {
													//.  switch Hermes to initial mode
/*1*/	OPW( HREG_EV_ACK, 0xFFFF );
		rc = cmd_exe( ifbp, HCMD_INI, 0 );	/* HCMD_INI can not be part of init() because that is called on 
											 * other occasions as well */
		rc = init( ifbp );
	}
													//if final "program" LTV
	if ( p->mode == CFG_PROG_STOP && ifbp->IFB_DLMode == CFG_PROG_VOLATILE) {
													//.  start tertiary (or secondary)
		OPW( HREG_PARAM_1, p->high_addr );
		rc = cmd_exe( ifbp, HCMD_EXECUTE, p->low_addr );
		if (rc == HCF_SUCCESS) rc = init( ifbp );	/*;? do we really want to skip init if cmd_exe failed, i.e.
													 *	 IFB_FW_Comp_Id is than possibly incorrect */
													//else (non-final)
	} else {
													//.  if mode == Readback SEEPROM
		if ( p->mode == CFG_PROG_SEEPROM_READBACK ) {
			OPW( HREG_PARAM_1, p->high_addr );
			OPW( HREG_PARAM_2, MUL_BY_2(p->len - 4));
													//.  .  perform Hermes prog cmd with appropriate mode bits
			rc = cmd_exe( ifbp, HCMD_PROGRAM | p->mode, p->low_addr);
													//.  .  set up NIC RAM addressability according Resp0-1
			OPW( HREG_AUX_PAGE,   IPW( HREG_RESP_1) );
			OPW( HREG_AUX_OFFSET, IPW( HREG_RESP_0) );
													//.  .  set up L-field of LTV according Resp2
			i = ( IPW( HREG_RESP_2 ) + 1 ) / 2;  // i contains max buffer size in words, a probably not very usefull piece of information ;?
/*Nico's code based on i is the "real amount of data available"
			if ( p->len - 4 < i ) rc = HCF_ERR_LEN;
			else p->len = i + 4;
*/
/* Rolands code based on the idea that a MSF should not ask for more than is available
			// check if number of bytes requested exceeds max buffer size
			if ( p->len - 4 > i ) {
				rc = HCF_ERR_LEN;
				p->len = i + 4;
			}
*/
													//.  .  copy data from NIC via AUX port to LTV
			cp = (wci_bufp)p->dl_data;						/*IN_PORT_STRING macro may modify its parameters*/
			i = p->len - 4;
			io_port	= (hcf_io/*;?NDIS*/)ifbp->IFB_IOBase + HREG_AUX_DATA;
			IN_PORT_STRING( io_port, cp, i );		//!!!WORD length, cp MUST be a char pointer	// $$ char
													//.  else (non-final programming)
		} else {
													//.  .  get number of words to program
			i = p->len - 4;
													//.  .  copy data (words) from LTV via AUX port to NIC
			cp = (wci_bufp)p->dl_data;						//OUT_PORT_STRING macro may modify its parameters
													//.  .  if mode == volatile programming
			if ( p->mode == CFG_PROG_VOLATILE ) {
													//.  .  .  set up NIC RAM addressability via AUX port
				OPW( HREG_AUX_PAGE, p->high_addr<<9 | p->low_addr>>7 );
				OPW( HREG_AUX_OFFSET, p->low_addr & 0x007E );
				OUT_PORT_STRING( io_port, cp, i );		//!!!WORD length,  cp MUST be a char pointer
			}
		}
	}
	ifbp->IFB_DLMode = p->mode;					//save state in IFB_DLMode
	HCFASSERT( rc == HCF_SUCCESS, rc )
	HCFLOGEXIT( HCF_TRACE_DL )
	return rc;
}/* download */
#endif // HCF_DLV
#endif // HCF_TYPE_HII

#if !( (HCF_TYPE) & HCF_TYPE_HII)
/*********************************************************************************************************************

.MODULE			download
.DESCRIPTION

.ARGUMENTS
  int download( IFBP ifbp, LTVP ltvp )

.RETURNS

.NARRATIVE

 Parameters:
	ifbp		address of the Interface Block
	ltvp		specifies the pseudo-RID (as defined by WCI)

.DIAGRAM
19:	if HCF_ASSERT is not defined, the else clause absorbs the "return rc;" statement. This is plainly not the
	intention, hence the semi-colon after the first HCFASSERT, so there will be at least an empty statement
	after the else.



*********************************************************************************************************************/
STATIC void download_data( IFBP ifbp, LTVP ltvp, hcf_16 offset, hcf_16 page );
#if defined HCF_DLNV 
not yet supported;
#endif // HCF_DLNV
#if defined HCF_DLV
int
download(IFBP ifbp, LTVP ltvp )
{


hcf_16				i = ltvp->len - 1;
int					rc = HCF_SUCCESS;
CFG_DL_BUF_STRCT	strct = { sizeof(CFG_DL_BUF_STRCT)/sizeof(hcf_16)-1, CFG_DL_BUF };
	switch (ltvp->typ) {												// function dispatch table
	  case CFG_DLV_START:												// Setup Download RAM
		/* this HCFASSERT never catches, the side effects are needed	*/
		HCFASSERT( ( ifbp->IFB_DLTarget[0] = ifbp->IFB_DLTarget[1] = 0 ) == 0, 0 )
						//bring AP back to STA, bring STA back to PRI
												//.  Ack everything to unblock a (possibly) blocked cmd pipe line
												/* Note 1: it is very likely that an Alloc event is pending
												 *   and very well possible that a (Send) Cmd event is pending 
												 *   on non-initial HCF_ACT_CARD_IN calls  
												 * Note 2: it is assumed that this strategy takes away the need
												 *   to ack every conceivable event after an Hermes Initialize
												 */
		OPW( HREG_EV_ACK, 0xFFFF );
						//let HCMD_INI in init bring both STA and PRI to STA
		if ( (rc = cmd_exe( ifbp, HCMD_INI|HCMD_INI_0x0100, 0 ) ) != HCF_SUCCESS ) break;
		if ( (rc = cmd_exe( ifbp, HCMD_INI, 0 ) ) != HCF_SUCCESS ) break;
		if ( rc == HCF_SUCCESS ) rc = HI_AUX_CNTL( ifbp, HREG_CNTL_AUX_ENA_CNTL );
		if ( rc == HCF_SUCCESS ) {
			OPW( HREG_PARAM_1, ltvp->val[1] );
			rc = cmd_exe( ifbp, HCMD_PROGRAM_ENABLE_VOLATILE | HCMD_PROGRAM, ltvp->val[0] );
		}
		break;
	  case CFG_DLV_ADDR:												//Download RAM Addressing information
		ifbp->IFB_DLTarget[0] = ltvp->val[0];
		ifbp->IFB_DLTarget[1] = ltvp->val[1];
		break;
	  case CFG_DLV_DATA:												//Download RAM Data
		HCFASSERT( ifbp->IFB_DLTarget[0] | ifbp->IFB_DLTarget[1], 0 )

//;?d:\develop\consolidate\hcf\hcf.c(3711) : warning C4761: integral size mismatch in argument; conversion supplied
		download_data( ifbp,ltvp, (hcf_16)(ifbp->IFB_DLTarget[0] & 0x007E),
						(hcf_16)((ifbp->IFB_DLTarget[0]>>7) + (ifbp->IFB_DLTarget[1]<<9)) );
		break;
	  case CFG_DL_STOP:														//Cleanup Download
		rc = cmd_exe( ifbp, HCMD_PROGRAM | HCMD_PROGRAM_DISABLE, 0 );
		i = (hcf_16)HI_AUX_CNTL( ifbp, HREG_CNTL_AUX_DIS_CNTL );
		if ( rc == HCF_SUCCESS ) rc = i;
		i = (hcf_16)init( ifbp );	//, INI_COMPLETE );
		if ( rc == HCF_SUCCESS /*&& i != HCF_ERR_INCOMP_STA*/ ) rc = i;
		break;
	  default:
	  	HCFASSERT( DO_ASSERT, ltvp->typ )
/*19*/	 /*NOP*/;																               
	  	//!rc = HCF_SUCCESS;
	}
	HCFASSERT( rc == HCF_SUCCESS, rc )
	HCFLOGEXIT( HCF_TRACE_DL )
	return rc;
}/* download */

/*********************************************************************************************************************

.MODULE			download_data
.DESCRIPTION

.ARGUMENTS
  void download_data( IFBP ifbp, LTVP ltvp, hcf_16 offset, hcf_16 page )

.RETURNS
	HCF_SUCCESS

.NARRATIVE

 Parameters:
	ifbp		address of the Interface Block
	ltvp		specifies the pseudo-RID (as defined by WCI)
	offset
	page

.DIAGRAM
*********************************************************************************************************************/
void
download_data( IFBP ifbp, LTVP ltvp, hcf_16 offset, hcf_16 page )
{

hcf_16		len	= ltvp->len - 1;
wci_bufp	cp = (wci_bufp)ltvp->val;
hcf_io		io_port	= (hcf_io/*;?NDIS*/)ifbp->IFB_IOBase + HREG_AUX_DATA;
hcf_32		tl;						//to ease the 32 bits calculation

	HCFASSERT( ifbp->IFB_DLTarget[0] | ifbp->IFB_DLTarget[1], 0 )
		
	tl = (hcf_32)MUL_BY_2( len );
	tl += ifbp->IFB_DLTarget[0] + ((hcf_32)ifbp->IFB_DLTarget[1] << 16);
	ifbp->IFB_DLTarget[0] = (hcf_16)tl;
	ifbp->IFB_DLTarget[1] = (hcf_16)(tl >> 16);

	OPW( HREG_AUX_OFFSET, offset );
	OPW( HREG_AUX_PAGE, page );
	OUT_PORT_STRING( io_port, cp, len );		//!!!WORD length,  cp MUST be a char pointer
}/* download_data */
#endif // HCF_DLV 
#endif // HCF_TYPE_HII
#endif // HCF_TYPE_USB


#if ! ((HCF_TYPE) & HCF_TYPE_USB)
/******************************************************************************************************************

.MODULE			get_fid
.DESCRIPTION	get allocated FID for either transmit or notify

.ARGUMENTS
  hcf_16 get_fid( IFBP ifbp )

.RETURNS
	0	no FID available
	<>0	FID number

.NARRATIVE

 Parameters:
  ifbp		address of the Interface Block

.DIAGRAM
 2:	card check needed here to prevent I/O from either of the two places where get_fid is called
 4:	The preference is to use a "pending" alloc. If no alloc is pending, then -if available-
 	the "spare" FID is used.
	Note that due to the H/W problems which are intended to be worked around by DAWA, the Alloc
	bit in the Event register is no longer a reliable indication of the presence/absence of
	an FID, so the test on the Alloc bit might just as well be skipped
	If the pending alloc is used, the alloc event must be acknowledged to the Hermes.
	If the spare FID is used, IFB_RscInd (representing the spare FID) must be cleared
.ENDOC				END DOCUMENTATION

***************************************************************************************************************/
hcf_16
get_fid( IFBP ifbp )
{

hcf_16 fid = 0;

//;?HCF_CARD_CHECK {																				/* 2 */
		if ( IPW( HREG_EV_STAT) & HREG_EV_ALLOC) { 		   	/* 4 */
			fid = IPW( HREG_ALLOC_FID );		/*resolve race condition in the above if */
			OPW( HREG_ALLOC_FID, 0 ); /*DAWA*/
			S_DAWA_ACK( ifbp, HREG_EV_ALLOC );
			if ( ifbp->IFB_RscInd == 1 ) ifbp->IFB_RscInd = 0;
		} else {
			fid = ifbp->IFB_RscInd;
			ifbp->IFB_RscInd = 0;
		}
//;?}
	return fid;
} //get_fid
#endif // HCF_TYPE_USB


/*******************************************************************************************************************

.MODULE			init
.DESCRIPTION    ..........., in addition, a light-weight diagnostic test of the NIC

.ARGUMENTS
  int init( IFBP ifbp, int cntl )

.RETURNS
	HCF_ERR_INCOMP_PRI
	>>hcf_get_info
	HCF_ERR_INCOMP_STA
	>>put_info


.NARRATIVE

  Parameters:
	ifbp		address of the Interface Block
	cntl		INI_PARTIAL, minimal initialization as needed for download and before diagnose
				INI_COMPLETE, complete initialization

  Condition Settings:
	Card Interrupts: Disabled as side effect of ini_hermes() (Note that the value of IFB_IntOffCnt is unchanged)

  init will successively:
  -	initialize the NIC by calling ini_hermes
  -	calibrate the S/W protection timer against the Hermes Timer
  - check HSI/Primary compatibility with the HCF
  - collect the parameters to be used in the non-volatile download process
  - check Station compatibility with the HCF
  - re-configure the Hermes based on IFB_CfgTbl


.DIAGRAM
 2:	Clear all fields of the IFB except those which need to survive init. This is intended to make
 	the behavior and - as a consequence - the debugging of the HCF more reproducible.
	Especially IFB_DefunctStat must be cleared (new round, new chances)
 6:	Initialize the Hermes (to the extend possible via the Host I/F) by calling ini_hermes(). The interrupt
 	generation facility is disabled as side effect of ini_hermes(). If this initialization fails, return
 	to the caller with appropriate error status.
 	Note that ini_hermes checks for Card Presence, so init does not need to do this.
 8: drop all error status bits in IFB_CardStat since they are expected to be re-evaluated.
10:	Calibrate the S/W time-out protection mechanism by calling calibrate(). Note that possible errors
	in the calibration process are nor reported by init but will show up via the defunct 
	mechanism in subsequent hcf-calls.
14:	check_comp() is called to check the Supplier Range of the Host-S/W I/F (HSI) and the Primary Firmware
	and checked against the Top and Bottom level supported by this HCF.
	If either of these tests fails, the CARD_STAT_INCOMP_PRI bit of IFB_CardStat is set and control returns
 	to the caller with appropriate error status.
30: The necessary initialization of the NIC as well as the necessary information gathering for download
	is now done, so skip all station and configuration stuff if we are preparing for download
32: In case of a HCF compiled for station functionality, the Supplier Range of the Station Firmware function
	is retrieved from the Hermes and checked against the Top and Bottom level supported by this HCF.
	Note that the Firmware can have multiple Variants, but that the HCF currently only supports a single
	variant.
34:	
40:	re-configure the Hermes based on IFB_CfgTbl. In any particular situation the decision to stop on an error
	for a single configuration or to continue with the remaining configurations could probably be made. In
	general there does not seem to be "the" right answer, hence here is chosen for the easiest
	implementation, i.e. to ignore errors.

.NOTICE
 o  init disables the card interrupts as side effect of ini_hermes(), however it does NOT influence
 	IFB_IntOffCnt. This way it is symmetrical with hcf_enable, which does NOT enable the card interrupts.
*/

int
init( IFBP ifbp )
{

int		rc = HCF_SUCCESS;	//hardly interesting in case of H-II since cmd_exe is then skipped ;?
hcf_8	*q;

	HCFLOGENTRY( HCF_TRACE_INITIALIZE, 0 )
	HCFASSERT( ifbp->IFB_Magic == HCF_MAGIC, ifbp->IFB_Magic )
	HCFASSERT_INT

												//.  Ack everything to unblock a (possibly) blocked cmd pipe line
												/* Note 1: it is very likely that an Alloc event is pending
												 *   and very well possible that a (Send) Cmd event is pending 
												 * Note 2: it is assumed that this strategy takes away the need
												 *   to ack every conceivable event after an Hermes Initialize
												 */
	IF_NOT_USB( OPW( HREG_EV_ACK, 0xFFFF ); )
#if ! ( (HCF_TYPE) & HCF_TYPE_HII )
	cmd_exe( ifbp, HCMD_INI, HCMD_INI_0x0100 );	//assume if this one fails, the next one fails too
	if ( (rc = cmd_exe( ifbp, HCMD_INI, 0 ) ) == HCF_SUCCESS )
#endif //HCF_TYPE_HII
	{	
/*8*/	ifbp->IFB_CardStat = 0;		//;?to be superfluous in the end
/*10*/	IF_NOT_USB( calibrate( ifbp ); ) 						
#if defined MSF_COMPONENT_ID 																			/* 14*/
		ifbp->IFB_HSISup.len = sizeof(CFG_NIC_HSI_SUP_RANGE_STRCT)/sizeof(hcf_16) - 1;
		ifbp->IFB_HSISup.typ = CFG_NIC_HSI_SUP_RANGE;
#if (HCF_TYPE) & HCF_TYPE_USB
		ifbp->IFB_UBICfg = check_comp( ifbp, (CFG_RANGES_STRCT*)&cfg_drv_act_ranges_ubi, CFG_UBI_SUP_RANGE );
		ifbp->IFB_UDICfg = check_comp( ifbp, (CFG_RANGES_STRCT*)&cfg_drv_act_ranges_udi, CFG_UDI_SUP_RANGE );
 		if ( ifbp->IFB_UBICfg == 0 || ifbp->IFB_UDICfg == 0 ) rc = ifbp->IFB_CardStat = HCF_ERR_INCOMP_PRI;
		else //be aware of this fall through scheme
#endif // HCF_TYPE_USB
//			cast of cfg_drv_act_ranges_hsi is needed to get around Microsoft compiler problem in DOS ODI
//			ifbp->IFB_HSICfg = check_comp( ifbp, (CFG_RANGES_STRCT*)&cfg_drv_act_ranges_hsi, CFG_NIC_HSI_SUP_RANGE );
		rc = hcf_get_info( ifbp, (LTVP)&ifbp->IFB_HSISup.len );		//detect HCF_ERR_NO_NIC as side effect
		if ( rc == HCF_SUCCESS ) {
            /* cbz */
            ifbp->IFB_HSISup.role    = CNV_LITTLE_TO_INT( ifbp->IFB_HSISup.role );
            ifbp->IFB_HSISup.id      = CNV_LITTLE_TO_INT( ifbp->IFB_HSISup.id );
            ifbp->IFB_HSISup.variant = CNV_LITTLE_TO_INT( ifbp->IFB_HSISup.variant );
            ifbp->IFB_HSISup.bottom  = CNV_LITTLE_TO_INT( ifbp->IFB_HSISup.bottom );
            ifbp->IFB_HSISup.top     = CNV_LITTLE_TO_INT( ifbp->IFB_HSISup.top );

			ifbp->IFB_PRIIdentity.len = sizeof(CFG_FW_IDENTITY_STRCT)/sizeof(hcf_16) - 1;
			ifbp->IFB_PRIIdentity.typ = CFG_PRI_IDENTITY;
			hcf_get_info( ifbp, (LTVP)&ifbp->IFB_PRIIdentity.len );
			/* cbz */
			ifbp->IFB_PRIIdentity.comp_id       = CNV_LITTLE_TO_INT( ifbp->IFB_PRIIdentity.comp_id );
			ifbp->IFB_PRIIdentity.variant       = CNV_LITTLE_TO_INT( ifbp->IFB_PRIIdentity.variant );
			ifbp->IFB_PRIIdentity.version_major = CNV_LITTLE_TO_INT( ifbp->IFB_PRIIdentity.version_major );
			ifbp->IFB_PRIIdentity.version_minor = CNV_LITTLE_TO_INT( ifbp->IFB_PRIIdentity.version_minor );

//			ifbp->IFB_PRICfg = check_comp( ifbp, (CFG_RANGES_STRCT*)&cfg_drv_act_ranges_pri, CFG_PRI_SUP_RANGE );
			ifbp->IFB_PRISup.len = sizeof(CFG_PRI_SUP_RANGE_STRCT)/sizeof(hcf_16) - 1;
			ifbp->IFB_PRISup.typ = CFG_PRI_SUP_RANGE;
			hcf_get_info( ifbp, (LTVP)&ifbp->IFB_PRISup.len );
			/* cbz */
			ifbp->IFB_PRISup.role    = CNV_LITTLE_TO_INT( ifbp->IFB_PRISup.role );
			ifbp->IFB_PRISup.id      = CNV_LITTLE_TO_INT( ifbp->IFB_PRISup.id );
			ifbp->IFB_PRISup.variant = CNV_LITTLE_TO_INT( ifbp->IFB_PRISup.variant );
			ifbp->IFB_PRISup.bottom  = CNV_LITTLE_TO_INT( ifbp->IFB_PRISup.bottom );
			ifbp->IFB_PRISup.top     = CNV_LITTLE_TO_INT( ifbp->IFB_PRISup.top );

		}
		ifbp->IFB_FWIdentity.len = sizeof(CFG_FW_IDENTITY_STRCT)/sizeof(hcf_16) - 1;
		ifbp->IFB_FWIdentity.typ = CFG_FW_IDENTITY;
		hcf_get_info( ifbp, (LTVP)&ifbp->IFB_FWIdentity.len );
		/* cbz */
		ifbp->IFB_FWIdentity.comp_id       = CNV_LITTLE_TO_INT( ifbp->IFB_FWIdentity.comp_id );
		ifbp->IFB_FWIdentity.variant       = CNV_LITTLE_TO_INT( ifbp->IFB_FWIdentity.variant );
		ifbp->IFB_FWIdentity.version_major = CNV_LITTLE_TO_INT( ifbp->IFB_FWIdentity.version_major );
		ifbp->IFB_FWIdentity.version_minor = CNV_LITTLE_TO_INT( ifbp->IFB_FWIdentity.version_minor );

		ifbp->IFB_FWSup.len = sizeof(CFG_FW_SUP_RANGE_STRCT)/sizeof(hcf_16) - 1;
		ifbp->IFB_FWSup.typ = CFG_FW_SUP_RANGE;
		if ( ifbp->IFB_FWIdentity.len > 1 ) {
			hcf_get_info( ifbp, (LTVP)&ifbp->IFB_FWSup.len );
			/* cbz */
			ifbp->IFB_FWSup.role    = CNV_LITTLE_TO_INT( ifbp->IFB_FWSup.role );
			ifbp->IFB_FWSup.id      = CNV_LITTLE_TO_INT( ifbp->IFB_FWSup.id );
			ifbp->IFB_FWSup.variant = CNV_LITTLE_TO_INT( ifbp->IFB_FWSup.variant );
			ifbp->IFB_FWSup.bottom  = CNV_LITTLE_TO_INT( ifbp->IFB_FWSup.bottom );
			ifbp->IFB_FWSup.top     = CNV_LITTLE_TO_INT( ifbp->IFB_FWSup.top );

		} else {		//fake AP Identity and supplier range
			/* you could make this compilation conditional because it can only occur for H-I
			   review this and the corresponding DN
			   */
//			ifbp->IFB_FWIdentity.len = sizeof(CFG_FW_IDENTITY_STRCT)/sizeof(hcf_16) - 1;
//			ifbp->IFB_FWIdentity.variant 		= 1;
//			ifbp->IFB_FWIdentity.version_major	= 1;
//			ifbp->IFB_FWIdentity.version_minor	= 1;
			ifbp->IFB_FWSup.role 				= COMP_ROLE_SUPL;
			ifbp->IFB_FWSup.id   				= COMP_ID_APF;
			ifbp->IFB_FWSup.variant				= 1;
			ifbp->IFB_FWSup.bottom				= 1;
			ifbp->IFB_FWSup.top  				= 1;
		}
	// ;??  add some stuff like check_comp( ifbp, (CFG_RANGES_STRCT*)&cfg_drv_act_ranges_sta, CFG_FW_SUP_RANGE );
#endif //MSF_COMPONENT_ID 
#if !((HCF_TYPE) & HCF_TYPE_USB)
		if ( 30 < ifbp->IFB_FWIdentity.comp_id && ifbp->IFB_FWIdentity.comp_id < 40 ) { //skip this on H-II if only primary present 
hcf_32 prot_cnt = ifbp->IFB_TickIni;
/*8*/
			if ( cmd_exe( ifbp, HCMD_ALLOC, 0 ) == HCF_SUCCESS ) {
/*10*/			HCF_WAIT_WHILE( (IPW( HREG_EV_STAT ) & HREG_EV_ALLOC) == 0 );							
/*12*/			if ( prot_cnt == 0 ) ifbp->IFB_HCF_Tallies.MiscErr++;									
				else {
					ifbp->IFB_RscInd = get_fid( ifbp );
/*14*/				cmd_exe( ifbp, HCMD_ALLOC, 0 );														
				}
			}
			HCFASSERT( ifbp->IFB_RscInd, 0)
		}
#endif // HCF_TYPE_USB
	}
	HCFASSERT( rc == HCF_SUCCESS, rc )
	HCFLOGEXIT( HCF_TRACE_INITIALIZE )
	return rc;
}/* init */
//#endif // HCF_TYPE


#if ! ((HCF_TYPE) & HCF_TYPE_USB)
/*********************************************************************************************************************

.MODULE			isr_info
.DESCRIPTION

.ARGUMENTS
  void isr_info( IFBP ifbp )

.RETURNS
  void

	.NARRATIVE

 Parameters:
  ifbp		address of the Interface Block

  Description:


.DIAGRAM
 1:	First the FID number corresponding with the InfoEvent is determined. 
	Note the complication of the zero-FID protection sub-scheme in DAWA. 
 	Next the L-field and the T-field are fetched into scratch buffer info.
 2: In case of tallies, the 16 bits Hermes values are accumulated in the IFB into 32 bits values. Info[0]
 	is (expected to be) HCF_NIC_TAL_CNT + 1. The contraption "while ( info[0]-- >1 )" rather than
 	"while ( --info[0] )" is used because it is dangerous to determine the length of the Value field by
 	decrementing info[0]. As a result of a bug in some version of the Hermes, info[0] may be 0, resulting
 	in a very long loop in the pre-decrement logic.
 	In case of the Lucent AccessPoint, the tallies are put in the MailBox rather than in the IFB.
 4: In case of a link status frame, the information is copied to the IFB field IFB_linkStat
 6:	All other than Tallies (including "unknown" ones and link status which is also put in IFB_LinkStat) are
 	put in the MailBox.  Although put_info_mb is robust against a len-parameter with value zero, it accepts
 	any bogus value for the type-parameter.

.ENDOC				END DOCUMENTATION

***************************************************************************************************************/
void
isr_info( IFBP ifbp )
{

hcf_16	info[2], tmp;
hcf_32	*p;

	HCFTRACE( ifbp, HCF_TRACE_ISR_INFO );                          										/* 1 */
//xxtmp = IPW( HREG_INFO_FID );
	if ( ( tmp = IPW( HREG_INFO_FID ) ) != 0 ) {
		OPW( HREG_INFO_FID, 0 ); /*DAWA*/
		(void)strio(ifbp, IO_IN, tmp, 0, info, 2 BE_PAR(2) );
		HCFASSERT( info[0] <= HCF_MAX_LTV, info[0] )
		HCFASSERT( info[0] <= HCF_MAX_LTV, info[1] )
#if (HCF_TYPE) == HCF_TYPE_AP || MSF_COMPONENT_ID != COMP_ID_AP1	//pre-SSN ORiNOCO AP
		if ( info[1] == CFG_TALLIES ) {
/*2*/		if ( info[0] > HCF_NIC_TAL_CNT ) info[0] = HCF_NIC_TAL_CNT + 1;								
			p = (hcf_32*)&ifbp->IFB_NIC_Tallies;
			while ( info[0]-- >1 ) *p++ += IPW( HREG_DATA_0 );	//request may return zero length
		}
		else
#endif //COMP_ID_AP1
		{
/*4*/		if ( info[1] == CFG_LINK_STAT ) ifbp->IFB_LinkStat = IPW( HREG_DATA_0 );							
#if defined HCF_MB_ON
/*6*/		ifbp->IFB_MB_FID = tmp;                     												
			put_info_mb( ifbp, info[1], NULL, info[0] );
#endif // HCF_MB_ON
		}
		HCFTRACE( ifbp, HCF_TRACE_ISR_INFO | HCF_TRACE_EXIT );
	}
	return;
}/* isr_info */
#endif // HCF_TYPE_USB


/*********************************************************************************************************************

.MODULE			put_info
.DESCRIPTION	stores Hermes configuration information in the CfgTbl of the IFB

.ARGUMENTS
  int put_info( IFBP ifbp, LTVP ltvp )

.RETURNS
	HCF_ERR_NO_NIC
	0 (Restraint violation)
	>>strio
	>>cmd_wait

.NARRATIVE

 Parameters:
	ifbp		address of the Interface Block
	ltvp		address in NIC RAM where LVT-records are located

 Remarks:
	In case of HCF-light, it may seem that it would be an improvement to integrate this code into
	hcf_put_info, however in the full blown HCF there is more logic involved in the handling of
	configuration, which makes this integration unattractive.
	Safe keeps Hermes configuration information. This information is used to restore the Hermes
	configuration after download and diagnose
	The Configuration information is a series of LTV-records, terminated with an L-field with a zero-value.

.DIAGRAM
20:	do not write RIDs to NICs which are incompatible with respect to Primary or Station Firmware level
22: only write RIDs when the NIC is present
24: If the RID does not exist, the L-field is set to zero. If the RID exists, the L-field is left at
 	one. Note that some RIDs can not be read, e.g. the pseudo RIDs for direct Hermes commands and CFG_DEFAULT_KEYS
28:	If the RID is written successful, pass it to the NIC by means of an Access Write command
**************************************************************************************************************/
int
put_info( IFBP ifbp, LTVP ltvp	)
{

int						rc = HCF_SUCCESS;
CFG_NIC_PROFILE_STRCT	x;


//	if ( (ifbp->IFB_CardStat & (CARD_STAT_INCOMP_PRI|CARD_STAT_INCOMP_STA) ) == 0 ) {						/* 20*/
//#if defined HCF_CARD_CHECK_ON
//		if ( !(ifbp->IFB_CardStat & CARD_STAT_PRESENT) ) /* !HCF_CARD_CHECK */ { rc = HCF_ERR_NO_NIC; } else			 	//misindent to accommodate HCF-light	/* 22*/
//#endif // HCF_CARD_CHECK_ON
//		{		/* !! be careful with this curly bracket when changing this code !! */
#if !defined HCF_CARD_CHECK_OFF
//;?superfluos, strio will report this	/* ! */ HCF_CARD_CHECK ; else rc = HCF_ERR_NO_NIC; 
#endif // HCF_CARD_CHECK_OFF
	if ( rc == HCF_SUCCESS && (ifbp->IFB_CardStat & (CARD_STAT_INCOMP_PRI|CARD_STAT_INCOMP_STA) ) == 0 ) {	/* 20*/
#if defined HCF_ASSERT
		if ( ltvp->typ != CFG_CMD_NIC      && 
			 ltvp->typ != CFG_DEFAULT_KEYS && ltvp->typ != CFG_ADD_TKIP_DEFAULT_KEY	) {
			x.len = 2;
			x.typ = ltvp->typ;
			hcf_get_info( ifbp, (LTVP)&x );
/*24*/		HCFASSERT( x.len, ltvp->typ )															
		}
#endif //HCF_ASSERT
//			if ( rc == HCF_SUCCESS ) {															/* 26*/
#if (HCF_TYPE) & HCF_TYPE_USB
		rc = UNUSendHcfReq( ltvp->typ, (PUSHORT)&ltvp->val[ 0], (USHORT)(ltvp->len - 1), NULL);	
#else
		rc = strio( ifbp, IO_OUT, ltvp->typ, 0, /*(wci_bufp)*/&ltvp->len, ltvp->len + 1 BE_PAR(2) );
							//for convenience, ignore DCWA at put_info, consequences are considered acceptable
/*28*/	if ( rc == HCF_SUCCESS ) rc = cmd_exe( ifbp, HCMD_ACCESS + HCMD_ACCESS_WRITE, ltvp->typ );		
#endif // HCF_TYPE_USB
//			}
	}
	return rc;
}/* put_info */



/*******************************************************************************************************************

.MODULE			put_info_mb
.DESCRIPTION
	copies either a PC-RAM located buffer or a RID/FID as a single Info block into the MailBox

.ARGUMENTS
  void put_info_mb( IFBP ifbp, hcf_16 type, wci_recordp bufp, hcf_16 len )
  Card Interrupts disabled

.RETURNS
  void

.NARRATIVE

 parameters:
	ifbp		address of the Interface Block
	type		in case of a V record in PC RAM:
				CFG_DIAG, CFG_WMP, CFG_UNK, CFG_DL_STAT or CFG_MB_ASSERT
				in case of RID/FID:
				CFG_SCAN, CFG_LINK_STAT, CFG_ASSOC_STAT
	bufp		pointer to a V record in PC RAM or don't care in case of RID/FID
	len			length in words of T field + V record in PC RAM

  Description:
	Note that the definition of len may not be 100% intuitive at first glance, but it is historically grown
	(in relation ship with the use of put_info_mb to copy LTVs directly from NIC RAM rather than PC RAM) and
	changing is expected to give more problems than solve them.

	If the data does not fit (including no MailBox is available), the IFB_MBTally is incremented and an
	HCF_ASSERT catches, but no error status is returned.
	Calling put_info_mb when their is no MailBox available, is considered a design error in the MSF.


	Note that there is always at least 1 word of unused space in the mail box.
	As a consequence:
	- no problem in pointer arithmetic (MB_RP == MB_WP means unambiguously mail box is completely empty
	- There is always free space to write an L field with a value of zero after each MB_Info block.  This
	  allows for an easy scan mechanism in the "get MB_Info block" logic.

.DIAGRAM
 2:	First the free space in the MailBox is calculated (2a: free part from Write Ptr to Read Ptr,
 	2b: free part turns out to wrap around) . If this space suffices to store the number of words
	reflected by len (T-field + Value-field) plus the additional MailBox Info L-field + a trailing 0 to act
	as the L-field of a trailing dummy or empty LTV record, then a MailBox Info block is build in the MailBox
	consisting of
	  -	the value len in the first word
	  -	type in the second word
	  - a copy of the contents of bufp/RID/FID in the second and higher word

	There is a simple mapping from the publicized WCI I/F for hcf_put_info to put_info_mb
	The WMP frame mode is recognized based on the IFB_MB_FID field of the IFB being non-zero.

	In the WMP mode, the MailBox Info block is build by means of input from NIC RAM rather than copied from a
	buffer. Since the (future) size of WMP frames is not known, it is considered to be an unattractive choice
	to copy a WMP frame first into a temporary buffer and then use the "standard" put_info_mb to copy the
	contents of the temporary buffer into the MailBox. Note that there are more arguments for the chosen
	implementation, e.g. efficiency

 4: Since put_info_mb() can more or less directly be called from the MSF level, the I/F must be robust
	against out-of-range variables. As failsafe coding, the MB update is skipped by changing tlen to 0 if
	len == 0; This will indirectly cause an assert as result of the violation of the next if clause.
 6:	Check whether the free space in MailBox suffices (this covers the complete absence of the MailBox).
	Note that len is unsigned, so even MSF I/F violation works out O.K.
	The '2' in the expression "len+2" is used because 1 word is needed for L itself and 1 word is needed
	for the zero-sentinel
 8:	update MailBox Info length report to MSF with "oldest" MB Info Block size

 5:	In case of an Hermes Information frame ( CFG_SCAN, CFG_LINK_STAT and CFG_ASSOC_STAT, as reflected by
	"type >= CFG_INFO_FRAME_MIN"), the L and T are already copied into the MailBox, hence reading the
	Information frame must start at offset 4.
 	In case of an Hermes Receive frame ( CFG_WMP ), reading must start at offset 0.
!!!	;? now we are in trouble, if this "if" clause was skipped (because the ReadPointer is larger than the
	WritePointer), this type dependent code is not executed. Simply copying this code to the 2nd strio
	does not work either because then it may be executed twice ;?


.NOTICE
	boundary testing depends on the fact that IFB_MBSize is guaranteed to be zero if no MailBox is present,
	and to a lesser degree, that IFB_MBWp = IFB_MBRp = 0

.ENDOC				END DOCUMENTATION

***************************************************************************************************************/
#if defined HCF_MB_ON
//put_info_mb_sup is a support routine for put_info_mb, however due to the "T"-logic it does not
//necessarily write to the MailBox
void
put_info_mb_sup( IFBP ifbp, hcf_16 offset, wci_recordp sp, wci_recordp dp, hcf_16 len )
{
#if ! ((HCF_TYPE) & HCF_TYPE_USB)
//	if ( sp == NULL )	//;?is this more robust against reantry problem of writing assert to mailbox if mailbox is full
	if ( ifbp->IFB_MB_FID )
		(void)strio( ifbp, IO_IN, ifbp->IFB_MB_FID, offset, /*(wci_bufp)*/dp, len BE_PAR(0) );
	else
#endif // HCF_TYPE_USB
		while ( len-- ) *dp++ = *sp++;	
} // put_info_mb_sup

/* as discussed with Martin at 20010522: if IFB_MB_FID non-zero AND strio asserts, then you may end
 * up with something like L1T1L2T2V1 in the MB.
 *
 * //hcf_16	run_time_flag = ifbp->IFB_AssertLvl;
 * //		ifbp->IFB_AssertLvl = 0;									// prevent recursive behavior
 */

/* due to the "T"-logic, put_info_mb does not necessarily write to the MailBox, but as a coincidemce 
 * MsfBuffer also abreviates to mb
 */
void
put_info_mb( IFBP ifbp, hcf_16 type, wci_recordp sp, hcf_16 len )
{

hcf_16		i;                     	//byte offset to start writing in MailBox
wci_recordp	dp;	         			//destination pointer (in MailBox)
hcf_16		tlen;										//free length/working length/offset in WMP frame
/* Offset controls the copying from NIC RAM
 * In case of an LTV, skip LT of LTV, but skip nothing in case of WMP frames
 * not really used for USB
 */
hcf_16		offset = (hcf_16)(type >= CFG_INFO_FRAME_MIN ? 4 : 0);
RID_LOGP	ridp = ifbp->IFB_RIDLogp;					// pointer to (zero-terminated array of) RID_LOG structures

#if (HCF_TYPE) & HCF_TYPE_USB
	HCFASSERT( sp, type )
#else
	HCFASSERT( ( ifbp->IFB_MB_FID == 0 ) ^ (sp == NULL ), (hcf_16)(hcf_32) sp )	//either IFB_MB_FID or sp
	HCFASSERT( ( ifbp->IFB_MB_FID == 0 ) ^ (sp == NULL ), ifbp->IFB_MB_FID )	//(but not both) should be non-zero
	HCFASSERT( ( ifbp->IFB_MB_FID == 0 ) ^ (sp == NULL ), type )
#endif // HCF_TYPE_USB	
	HCFASSERT( ifbp->IFB_MBp != 0, 0 )
	HCFASSERT( ifbp->IFB_MBSize, 0 )

												//reset lenght and pointers if MailBox empty
//		if ( ( ifbp->IFB_MBInfoLen = *q ) == 0 ) {
//			*ifbp->IFB_MBp = ifbp->IFB_MBRp = ifbp->IFB_MBWp = 0;
//		}

	//;?this excludes tallies from the Tee on non-APs, which seems the easiest way out for the short term
	if ( ridp != NULL ) {
		while ( ridp->typ ) {
			if ( ridp->typ == type ) {
				if ( ( tlen =  min( ridp->len, len ) ) != 0 ) {	//;?might just as well use len iso tlen
					ridp->bufp[0] = tlen--;				//save L, then convert it to number of V-values
					ridp->bufp[1] = type;				//save T
					put_info_mb_sup( ifbp, offset, sp, &ridp->bufp[2], tlen);
				}
/*
 * return;		//;? I like a solution as on the next line but that has some flow impact I want to
							//solve later when the bug is found and corrected
				len = 0;	//data goes either to MSF Buffer or to MailBox
*/				
			}
			ridp++;
		}
	}
	if ( ifbp->IFB_MBRp > ifbp->IFB_MBWp ) tlen = ifbp->IFB_MBRp - ifbp->IFB_MBWp;						/* 2a*/
	else { 
/* optimization add later after Wrapping algorithm is tested
   		if ( ifbp->IFB_MBRp == ifbp->IFB_MBWp ) ifbp->IFB_MBRp = ifbp->IFB_MBWp = 0;
*/		
		tlen = ifbp->IFB_MBSize - ifbp->IFB_MBWp;														/* 2b*/
		if ( ( tlen <= len + 2 ) && ( len + 2 < ifbp->IFB_MBRp ) ) {	//if trailing space is too small but
																	//	 leading space is sufficiently large
			ifbp->IFB_MBp[ifbp->IFB_MBWp] = 0xFFFF;					//flag dummy LTV to fill the trailing space
			ifbp->IFB_MBWp = 0;										//reset WritePointer to begin of MailBox
			tlen = ifbp->IFB_MBRp;									//get new available space size
		}
	}
	i = ifbp->IFB_MBWp;
	dp = &ifbp->IFB_MBp[i];
	if ( len == 0 ) tlen = 0; //;? what is this good for
	if ( len + 2 >= tlen ){																				/* 6 */
		//Do Not ASSERT, this is a normal condition
		ifbp->IFB_HCF_Tallies.NoBufMB++;
	} else {
/*8*/	if ( ifbp->IFB_MBInfoLen == 0 )  ifbp->IFB_MBInfoLen = len;
		/* alternative ifbp->IFB_MBInfoLen = ifbp->IFB_MBp[ifbp->IFB_MBRp]; at the end of this routine */
		*dp++ = len--;									/*write Len (= size of T+V in words to MB_Info block
														 *and compensate len for writing Type just below	*/
		*dp++ = type;										/*write Type to MB_Info block						*/

		i += len + 2;										/*update WritePointer of MailBox 					*/
		ifbp->IFB_MBWp = i;
		ifbp->IFB_MBp[i] = 0;							//to assure get_info_mb stops
		put_info_mb_sup( ifbp, offset, sp, dp, len);
	}
	IF_NOT_USB( ifbp->IFB_MB_FID = 0; )
}/* put_info_mb */

#endif // HCF_MB_ON


#if (HCF_TYPE) & HCF_TYPE_USB
/*******************************************************************************************************************

.MODULE			put_usb_nic_cmd
.DESCRIPTION

.ARGUMENTS
  hcf_16 put_usb_nic_cmd( IFBP ifbp, hcf_16 h16Cmd, hcf_16 h16Par0, hcf_16 h16Par1, hcf_16 h16Par2)

.RETURNS

.NARRATIVE

 Parameters:
  ifbp	address of the Interface Block
  h16Cmd
  h16Par0
  h16Par1
  h16Par2

 Description:


.DIAGRAM

.NOTICE
.ENDOC				END DOCUMENTATION

***************************************************************************************************************/
hcf_16	put_usb_nic_cmd( IFBP ifbp,
						 hcf_16 h16Cmd,
						 hcf_16 h16Par0,
						 hcf_16 h16Par1,
						 hcf_16 h16Par2)

{
	CFG_CMD_NIC_STRCT	CCNS;
	hcf_16				rc;

	if ( h16Cmd == HCMD_DIAG ) h16Par1 = ~ h16Par0;
	if ( h16Cmd == HCMD_INQUIRE ) h16Par1 = 0x3FFF;

	CCNS.len = sizeof( CCNS)/sizeof( hcf_16) - 1;
	CCNS.typ = CFG_CMD_NIC;
	CCNS.cmd = h16Cmd;
	CCNS.parm0 = h16Par0;
	CCNS.parm1 = h16Par1;
	CCNS.parm2 = h16Par2;
	rc = (hcf_16)put_info( ifbp, (LTVP)&CCNS );
	return( rc );
} // put_usb_nic_cmd
#endif // HCF_TYPE_USB

#if ! ((HCF_TYPE) & HCF_TYPE_USB)
/*******************************************************************************************************************

.MODULE			rcv_msg_frag
.DESCRIPTION

.ARGUMENTS
  void rcv_msg_frag( IFBP ifbp, wci_bufp bufp, int len )

.RETURNS

.NARRATIVE

 Parameters:
  ifbp	address of the Interface Block
  bufp	address of buffer
  len	lenghth in bytes of buffer specified by bufp

  Description: process the extranuous bytes (if applicable) read by the previous rcv_msg_frag 
  together with the bytes read from the RxFS by the current rcv_msg_frag by passing them in groups of 4 
  to the MIC calculation.
  Only the bytes read by the current rcv_msg_frag are stored in bufp.

.DIAGRAM
 2:	process remainder of previous rcv_msg_frag, i.e. if the previous rcv_msg_frag consumed only 1, 2 or 3
 	bytes of the n times 4 bytes which were read, the complete group of 4 bytes is copied to bufp and the final 
	group of 4 bytes must still be passed to the MIC calculation engine.
 	This last group of 4 bytes is saved in the accumulator IFB_rx_32.
 	If and only if IFB_rx_32 is fully consumed by this rcv_msg_frag, IFB_rx_32 is passed to the MIC Calculation 
	engine.
	Note that this code must cope with zero-lenght fragments and non-zero fragments which are nevertheless
	too small to completely consume the contents of IFB_rx_32. This is achieved by the check on len in the 
	while loop and the test on IFB_rx_tlen == 4 in the if.
	Note that the MIC calculation over the contents of IFB_rx_32 can NOT be done when these bytes are read 
	from the NIC in the while loop in point 6, because these bytes MAY be the first byte(s) of the MIC
	already.
 6:	The remaining space in bufp is filled by reading in groups of 2 times 2 bytes from the RxFS in NIC
 	memory. This reading stops at the last group of 4 bytes which fits at least partly in bufp.
	Each group which fits completely in bufp is copied to bufp and passed to the MIC Calculation engine. 
10:	If the last 4 byte group did not fit in bufp, there is still space for one or more (at most 3) bytes 
	in bufp. These bytes are copied from IFB_rx_32.
 	This last group of 4 bytes is completely saved in the accumulator IFB_rx_32.

.NOTICE
	It turns out DOS ODI uses zero length fragments. The HCF code can cope with it, but as a 
	consequence, no Assert on len is possible 
	
.ENDOC				END DOCUMENTATION

***************************************************************************************************************/
void
rcv_msg_frag( IFBP ifbp, wci_bufp bufp, int len )
{

union { hcf_32 x32; hcf_16 x16[2]; hcf_8 x8[4]; } x;

	if ( ifbp->IFB_rx_tlen ) {																				/* 2 */
		while ( ifbp->IFB_rx_tlen < 4 && len ) {
			*bufp++ = ifbp->IFB_rx_32[ifbp->IFB_rx_tlen++];
			len--;	//don't be clever and don't move this -- into the while
		}
		if ( ifbp->IFB_rx_tlen == 4 ) {		//cope with extremely small fragments
#if (HCF_TYPE) & HCF_TYPE_SSN		//this if can probably be deleted once MIC_RX_RTN is changed
//			ifbp->IFB_MICRxRtn( (hcf_32*)ifbp->IFB_MICRx, *(hcf_32*)ifbp->IFB_rx_32 );
			MIC_RX_RTN( (hcf_32*)ifbp->IFB_MICRx, *(hcf_32*)ifbp->IFB_rx_32 );
#endif //HCF_TYPE_SSN
			ifbp->IFB_rx_tlen = 0;
		}
	}
	while ( len ) {																					/* 6 */
		x.x16[0] = IPW( HREG_DATA_0 );
		x.x16[1]= IPW( HREG_DATA_0 );
		if ( len < 4 ) break;
#if HCF_ALIGN == 1
		*(hcf_16 FAR*)bufp = CNV_LITTLE_TO_INT(x.x16[0]);
		*(hcf_16 FAR*)(bufp+2)= CNV_LITTLE_TO_INT(x.x16[1]);
#else
		bufp[0 + EOC] = x.x8[0];
		bufp[1 - EOC] = x.x8[1];
		bufp[2 + EOC] = x.x8[2];
		bufp[3 - EOC] = x.x8[3];
#endif // HCF_ALIGN
#if (HCF_TYPE) & HCF_TYPE_SSN		//this if can probably be deleted once MIC_RX_RTN is changed
//		ifbp->IFB_MICRxRtn( (hcf_32*)ifbp->IFB_MICRx, x.x32 );
		MIC_RX_RTN( (hcf_32*)ifbp->IFB_MICRx, x.x32 );
#endif //HCF_TYPE_SSN
		bufp += 4;
		len -= 4;
	}
	if ( len ) {																					/* 10*/
		*(hcf_16 FAR *)&ifbp->IFB_rx_32[0] = CNV_LITTLE_TO_INT(x.x16[0]);
		*(hcf_16 FAR *)&ifbp->IFB_rx_32[2] = CNV_LITTLE_TO_INT(x.x16[1]);
		//! ifbp->IFB_rx_tlen = 0;
		while ( len-- ) *bufp++ = ifbp->IFB_rx_32[ifbp->IFB_rx_tlen++];
	}
} // rcv_msg_frag
#endif // HCF_TYPE_USB


/*******************************************************************************************************************

.MODULE			send_msg_frag
.DESCRIPTION

.ARGUMENTS
  void send_msg_frag( IFBP ifbp, wci_bufp bufp, int len )

.RETURNS

.NARRATIVE

 Parameters:
  ifbp	address of the Interface Block
  bufp	address of buffer
  len	lenghth in bytes of buffer specified by bufp

  Description: process the last bytes (if applicable) of the previous send_msg_frag and the bytes of the
  current send_msg_frag by passing them in groups of 4 to the MIC calculation, and output them in groups
  of 2 to the TxFS on the NIC

.DIAGRAM
 2:	process remainder of previous send_msg_frag, i.e. if any bytes of the previous send_msg_frag are still 
 	in the accumulator IFB_tx_32, waiting for some more bytes, to go as an quartet to the MIC calculation engine, 
	then as many bytes as needed to form this 4-some are copied from the message fragment to IFB_tx_32.
	Note that this code must cope with zero-lenght fragments and non-zero fragments which are nevertheless
	too small to extend the contents of IFB_tx_32 to a quad byte. This is achieved by the check on len in the 
	while loop and the test on IFB_tx_tlen == 4 in the if.
 	In addition to the MIC Calculation, the contents of IFB_tx_32 is written as 2 words to the NIC
 6:	The remaining bytes are processed in groups of 4 by the MIC Calculation engine. Any group of 4 bytes
 	is written as 2 words to the NIC
10:	If there are one or more (at most 3) bytes left, these bytes are stored in IFB_tx_32 to be processed
	with the next send_msg_frag or send_mic_dcp.

.NOTICE
	It turns out DOS ODI uses zero length fragments. The HCF code can cope with it, but as a 
	consequence, no Assert on len is possible 
.ENDOC				END DOCUMENTATION

***************************************************************************************************************/
#if 1
void
send_msg_frag( IFBP ifbp, wci_bufp bufp, int len )
{

union { hcf_32 x32; hcf_16 x16[2]; hcf_8 x8[4]; } x;

	HCFASSERT( len < HCF_MAX_MSG, len )

	if ( ifbp->IFB_tx_tlen ) {																				/* 2 */
		while ( ifbp->IFB_tx_tlen < 4 && len ) {
			ifbp->IFB_tx_32[ifbp->IFB_tx_tlen++] = *bufp++;
			len--;
		}
		if ( ifbp->IFB_tx_tlen == 4 ) {
			ifbp->IFB_tx_tlen = 0;
#if HCF_TYPE & HCF_TYPE_SSN  
			ifbp->IFB_MICTxRtn( (hcf_32*)ifbp->IFB_MICTx, *(hcf_32 BASED *)ifbp->IFB_tx_32 );
#endif //HCF_TYPE_SSN
			x.x16[0] = CNV_INT_TO_LITTLE(*(hcf_16 FAR *)&ifbp->IFB_tx_32[0] );
			OPW( HREG_DATA_0, x.x16[0] );
			x.x16[1] = CNV_INT_TO_LITTLE(*(hcf_16 FAR *)&ifbp->IFB_tx_32[2] );
			OPW( HREG_DATA_0, x.x16[1] );
		}
	}
	while ( len >= 4 ) {																			/* 6 */
#if defined HCF_BIG_ENDIAN
//NvR   tmpvar = (hcf_16)( *bufp | (hcf_16)(*(bufp+1) << 8));
#endif // HCF_BIG_ENDIAN
#if HCF_ALIGN == 1				//this saves 16 bytes code
		x.x16[0] = CNV_INT_TO_LITTLE(*(hcf_16 FAR*)bufp);
		x.x16[1] = CNV_INT_TO_LITTLE(*(hcf_16 FAR*)(bufp+2));
#else
		x.x8[0] = bufp[0 + EOC];
		x.x8[1] = bufp[1 - EOC];
		x.x8[2] = bufp[2 + EOC];
		x.x8[3] = bufp[3 - EOC];
#endif // HCF_ALIGN
		OPW( HREG_DATA_0, x.x16[0] );
		OPW( HREG_DATA_0, x.x16[1] );
#if HCF_TYPE & HCF_TYPE_SSN  
		ifbp->IFB_MICTxRtn( (hcf_32*)ifbp->IFB_MICTx, x.x32 );
#endif //HCF_TYPE_SSN
		bufp += 4;
		len -= 4;
	}
//	if ( len ) {
/*10*/	while ( len-- ) ifbp->IFB_tx_32[ifbp->IFB_tx_tlen++] = *bufp++;												
//	}
} // send_msg_frag
#else 
/*	this SEEMS a bit / a lot shorter and - depending on your taste - more elegant.
	However, it seems to make adifference of less than 20 bytes in code size.
	It does not function as is, except for multiple of 4 bytes.
	It is difficult to "see" whether it is correct with all weird boundary conditions caused by
	multiple fragments, including zero and sub-4 lenghts 
*/
void
send_msg_frag( IFBP ifbp, wci_bufp bufp, int len )
{

int			i = 0;
wci_bufp	p = bufp;
union { hcf_32 x32; hcf_16 x16[2]; hcf_8 x8[4]; } x;

	HCFASSERT( bufp, len )

	if ( ifbp->IFB_tx_tlen ) {		//process remainder of previous send_msg_frag
		while ( ifbp->IFB_tx_tlen < 4 && len ) {
			ifbp->IFB_tx_32[ifbp->IFB_tx_tlen++] = *bufp++;
			len--;
		}
		if ( ifbp->IFB_tx_tlen == 4 ) {		//cope with extremely small fragments
			ifbp->IFB_tx_tlen = 0;
			p = ifbp->IFB_tx_32;
			i = -4;
	}
	}
	while ( i < len  ) {
#if HCF_ALIGN == 1
		x.x16[0] = CNV_INT_TO_LITTLE(*(hcf_16 FAR*)p);
		x.x16[1] = CNV_INT_TO_LITTLE(*(hcf_16 FAR*)(p+2));
#else
		x.x8[0] = p[0 + EOC];
		x.x8[1] = p[1 - EOC];
		x.x8[2] = p[2 + EOC];
		x.x8[3] = p[3 - EOC];
#endif // HCF_ALIGN
		OPW( HREG_DATA_0, x.x16[0] );
		OPW( HREG_DATA_0, x.x16[1] );
#if HCF_TYPE & HCF_TYPE_SSN  
		ifbp->IFB_MICTxRtn( (hcf_32*)ifbp->IFB_MICTx, x.x32 );
#endif // HCF_TYPE_SSN
		i += 4;
		p = bufp + i;
	}
//	if ( len ) {
		while ( len-- ) ifbp->IFB_tx_32[ifbp->IFB_tx_tlen++] = *p++;
//	}
} // send_msg_frag
#endif //0

/*******************************************************************************************************************

.MODULE			send_mic_dcp
.DESCRIPTION

.ARGUMENTS
  void send_mic_dcp( IFBP ifbp )

.RETURNS

.NARRATIVE

 Parameters:
  ifbp	address of the Interface Block

  Description: finalize the MIC calculation with the padding pattern, output the last bytes (if applicable)
  of the message and the MIC to the TxFS

.DIAGRAM
 2:	0 - 3 bytes of the last send_msg_frag are still in IFB_tx_32, so at least 1 and most 4 bytes of the 
 	minimum padding of 0x5A000000 can be written to IFB_tx_32.
	The 1st call to the MIC calculation routine feeds these remaining bytes (if any) of send_msg_frag 
	and the first bytes of the padding to the MIC calculation engine.
	The 2nd call to the MIC calculation routine feeds 4 0x00 bytes of padding to the MIC calculation engine,
	thus completing the required 5 to 8 bytes padding.
 6:	process remainder -if any - of last send_msg_frag. Note that this remainder in IFB_tx_32 is undisturbed by 
 	point 2. Depending on the number of bytes (1-3) of the last send_msg_frag, 3 - 1 bytes of the MIC can 
	be copied from the IFB to IFB_tx_32 and then IFB_tx_32 is written as 2 words to NIC RAM.
 8:	write the remainder of the MIC and possible some garbage to NIC RAM
	Note: i is always 4 (a loop-invariant of the while in point 2)
10:	Write the DCP pattern, in case of SSN, the F/W checks the pattern, in case of pre-SSN, the remainder
	of the DCWA logic is contained in hcf_send_msg

.NOTICE

  DataCorruptionWorkAround scheme (ref:	DCWA W2DN176B1 "another layer of workaround")
The problem is that without known cause, the Hermes internal NIC RAM pointer misses its auto-increment causing
two successive writes to NIC RAM to address the same location. As a consequence word <n> is overwritten and
a word <n+j> is written at position <n+j-1>. Since the Hermes is unaware of this, nothing in the system is
going to catch this, e.g. the frame is received on the other side with correct FCS. As a workaround, the HCF
writes a number of words - the kludge pattern - after the MSF-data.  Then "something" checks the first word 
after the (supposed) MSF_data. This first word must match the first word of the kludge pattern, otherwise, 
apparently, the auto-increment failed. We write more than 1 word otherwise if the previous use of that piece 
of NIC RAM would have left by chance the right "kludge" value just after the newly but incorrectly put data, 
the test would not trigger. By overwriting the next 3 words as well, we assume the likelihood of this to be 
sufficiently small.
The frequency observed of this problem varies from 1 out of 1000 frames till too low to be noticeable. Just to
be on the safe side we assume that the chance of an error is 10E-3. By writing 4 words we reduce the
likelihood of an unnoticed problem to 10E-12, but of course this becomes tricky because we don't know whether
the chances are independent. 
Doing the checking in the HCF, requires the HCF to keep track of where the internal NIC RAM pointer is 
supposed to be, set the BAP register to the location which should contain the kludge pattern and read the 1st
word back. The required BAP setup is expensive in time (up to 400 microseconds) and the actual time needed is
not very predicatble.
Doing the checking in the F/W has no impact on the Host processor utilization and is assumed to have a low 
and very predicatble impact on the NIC performance.
Since the stability of the DCWA mechanism is now field-proven, the actual check is moved to the F/W domain.
This implementation is called DCP (Data Corruption Protection). To accomodate pre-SSN F/W, the DCWA checking
in the Host is still supported for pre-SSN F/W.
.ENDOC				END DOCUMENTATION

***************************************************************************************************************/
void
send_mic_dcp( IFBP ifbp )
{
int		i;
hcf_8	*bufp;

/*probably changed a alignment problem for an Endian Problem ;?????????????*/

#if !( (HCF_TYPE) & HCF_TYPE_SSN )
	bufp = (hcf_8*)ifbp->IFB_tx_32;		//;?add this to description
	i = 2;
#else
//#if  !( (HCF_TYPE) & HCF_TYPE_HII )		//on H-I, go always trough the MIC engine even if no SSN to get easy offset for DCWA
hcf_16	tmpvar;

	i = ifbp->IFB_tx_tlen; 																				/* 2 */
	ifbp->IFB_tx_32[i] = 0x5A;
	while ( ++i < 4 ) ifbp->IFB_tx_32[i] = 0x00;
	ifbp->IFB_MICTxRtn( (hcf_32*)ifbp->IFB_MICTx, *(hcf_32 BASED *)ifbp->IFB_tx_32 );
	ifbp->IFB_MICTxRtn( (hcf_32*)ifbp->IFB_MICTx, 0x00000000L );
	bufp = ifbp->IFB_MICTx;
	if ( ifbp->IFB_tx_tlen ) {																			/* 6 */
		while ( ifbp->IFB_tx_tlen < 4 ) {
			ifbp->IFB_tx_32[ifbp->IFB_tx_tlen++] = *bufp++;
		}
		tmpvar = CNV_INT_TO_LITTLE(*(hcf_16 FAR *)&ifbp->IFB_tx_32[0] );
		OPW( HREG_DATA_0, tmpvar );
		tmpvar = CNV_INT_TO_LITTLE(*(hcf_16 FAR *)&ifbp->IFB_tx_32[2] );
		OPW( HREG_DATA_0, tmpvar );
//		OPW( HREG_DATA_0, ifbp->IFB_tx_32[2] | ifbp->IFB_tx_32[4]<<8 );	4 bytes more code per OPW
//		OPW( HREG_DATA_0, ifbp->IFB_tx_32[0] + ifbp->IFB_tx_32[1]<<8 );	6 bytes more code per OPW
	}
//!	i = 4;																						/* 8 */
#endif //HCF_TYPE_SSN
	while ( i-- ) {
		OPW( HREG_DATA_0, *bufp | *(bufp+1)<<8 );	//;?change this, use hulpvariable, OPW may have side effects
		bufp += 2;
	}
#if  !( (HCF_TYPE) & HCF_TYPE_HII )																	/*10*/
	OPW( HREG_DATA_0, 0xCAFE );
	OPW( HREG_DATA_0, 0 );
	OPW( HREG_DATA_0, 0 );
	OPW( HREG_DATA_0, 0 );
#endif // HCF_TYPE_HII
} // send_mic_dcp


#if ! ((HCF_TYPE) & HCF_TYPE_USB)
/******************************************************************************************************************

.MODULE         strio
.DESCRIPTION    set up data access to NIC RAM via BAP_0 ports
				read read/write string with specified length from/to WaveLAN NIC RAM to/from PC RAM
				optionally converting the Endianess of a part of the words 

int strio( IFBP ifbp, int type, hcf_16 fid, hcf_16 offset, wci_bufp pc_addr,
				  int tot_len, int type [, int word_len] );

.ARGUMENTS
  IFBP			ifbp			address of I/F Block
  int			type            action code
								  -	IO_IN			read from NIC RAM
								  -	IO_OUT			write to NIC RAM
  int			fid				FID/RID
  int			offset			offset in FID/RID
  wci_bufp		pc_addr			begin address in PC RAM to write to or read from
  int			tot_len			number of WORDS (USED TO BE bytes) to write or read
  int			word_len		This parameter is only present when HCF_BIG_ENDIAN is defined
  								number of leading words of which the Endianess must be converted
  								implies offset and pc_addr are even

.RETURNS
	HCF_SUCCESS 	    		O.K
	HCF_ERR_NO_NIC				card is removed
	HCF_ERR_DEFUNCT_TIME_OUT	Fatal malfunction detected
	HCF_ERR_DEFUNCT_.....		if and only if IFB_DefunctStat <> 0
	
.NARRATIVE

  strio has the following tasks:
  -	set up data access to NIC RAM via BAP_0 ports
  - copy data from NIC RAM to Host RAM or vice versa
  - optionally convert (part of) the data from/to Little Endian format (as used by the NIC)
  	to/from the Native Endian format (as used by the Host)
	
  Data is a string with specified length copied from/to a specified offset in a specified Receive Frame
  Structure (FID), Transmit Frame Structure (FID) or Record (RID) to/from a Host RAM buffer with a specified
  begin address.
  
  A length of 0 can be specified, resulting BAP_0 setup but no data transfer. This feature played a
  	role in the optimization when 2 BAPs where used by the HCF. These days this feature services only
	as a robustness aspect.

  A non-zero return status indicates:
  - the NIC is considered nonoperational, e.g. due to a time-out of some Hermes activity in the past
  -	BAP_0 could not properly be initialized
  -	the card is removed before completion of the data transfer
  In all other cases, a zero is returned.
  If a card removal is returned, the MSF has the option to drop the message or recover in any other way it
  sees fit.
  BAP Initialization failure indicates an H/W error which is very likely to signal complete H/W failure. Once
  a BAP Initialization failure has occurred all subsequent interactions with the Hermes will return a "defunct" 
  status till the Hermes is re-initialized by means of an hcf_action with HCF_ACT_CARD_IN parameter.

  Strio prepares access via BAP-mechanism to a FID or a RID. A FID or RID is the handle supplied by the 
  Hermes to the Host to access the buffer structures located in NIC RAM.
  A BAP is a set of registers (Offset, Select and Data) offering read/write access to a particular FID or RID.
  This access is based on a auto-increment feature.
  There are two BAPs but these days the HCF uses only BAP_0 and leaces BAP_1 to the PCI Busmastering H/W.
  
  The BAP-mechanism is based on the Busy bit in the Offset register (see the Hermes definition). The waiting 
  for Busy must occur between writing the Offset register and accessing the Data register. The implementation 
  to wait for the Busy bit drop after each write to the Offset register, implies that the 
  requirement that the Busy bit is low  before the Select register is written, is automatically met. 
  !!! BE AWARE !!! of self supporting (i.e. not strio based) logic in hcf.c to manipulate the BAP 
  registers trying to do something usefull in the Busy bit settling time.
  The wait for Busy bit drop is protected by a loop counter, which is initialized with IFB_TickIni, which 
  is calibrated in init.

  BAP-setup may be time consuming (e.g. 380 usec for large offsets occurs frequently) 
  In the past a complicated scheme with S/W "shadows" of BAP0/1 Select/Offset was used to counteract these
  long times. By restructuring the DCWA and the buffer fragmentation logic, the profit of this 
  shadowing diminished so it was deleted.

  The NIC I/F is optimized for word transfer and can only handle word transfer at a word boundary in 
  NIC RAM. The intended solution for transfer of a single byte has multiple H/W flaws. There have been
  two different S/W Workaround strategies, one based on NIC RAM, and one based on temporary buffering
  of odd chars in the IFB. Due to the word oriented nature of the MIC feature, this byte logic became
  obsolete.
	
.DIAGRAM

 2:	the test on rc checks whether the HCF went into "defunct" mode ( e.g. BAP initialization or a call to 
 	cmd_wait did ever fail).
 4:	the select register and offset register are set
 	the offset register is monitored till a successful condition (no busy bit) is detected or till the 
	(calibrated) protection counter expires
	If the counter expires, this is reflected in IFB_DefunctStat, so all subsequent calls to strio
	fail immediately ( see 2)
 6:	HREG_OFFSET_ERR is ignored as error because:
	 a: the Hermes is robust against it
	 b: it is not known what causes it (probably a bug), hence no strategy can be specified which level is
	 	to handle this error in which way. In the past, it could be induced by the MSF level, e.g.  by 
		calling hcf_rcv_msg while there was no Rx-FID available. Since this is an MSF-error which is caught 
		by ASSERT, there is no run-time action required by the HCF. 
	Lumping the Offset error in with the Busy bit error, as has been done in the past turns out to be a 
	disaster or a life saver, just depending on what the cause of the error is. Since no prediction can be 
	done about the future, it is "felt" to be the best strategy to ignore this error. One day the code was 
	accompanied by the following comment:
	//	ignore HREG_OFFSET_ERR, someone, supposedly the MSF programmer ;) made a bug. Since we don't know
	//	what is going on, we might as well go on - under management pressure - by ignoring it
 8: The optional conversion of the first words from Native Endian to Little Endian format is done.
	This implies that Endian conversion can ONLY take place at word boundaries.
10: The PCMCIA card can be removed in the middle of the transfer. By depositing a "magic number" in the
	HREG_SW_0 register of the Hermes at initialization time and by verifying this register, it can be 
	determined whether the card is still present. The return status is set accordingly.
	A (relative) cheap way to prevent that failing I/O results in run-away behavior because garbage on the
	stack used as a local variable passed to strio is interpreted by the caller of strio
	irrespective of the return status van strio (e.g. hcf_service_nic has this behavior).

.NOTICE
  Depending on the selected optimization options, using "register int reg" causes some obscure
  optimization with si. As a result the code is longer. Therefore, do not invest time to optimize this code
  that way during a "maintenance cycle".

.NOTICE
  The IN_/OUT_PORT_WORD_/STRING macros are MSF-programmer defined and may or may not have side effects
  on their parameters, therefore they can not handle expressions like "len/2". As a solution all these macros
  must be called via "simple" variables, with no side effects like ++ and their contents is unpredictable
  at completion of the macro
 
.NOTICE  
  When, in a Big Endian environment, hcf_get_info is called for a non-existing RID, strio will be called 
  with a request to read 0 bytes (being the Hermes reported RID length) and 1 word to be converted to Little 
  Endian (being the assumed type). This cause ASSERT to trigger, but that is considered an acceptable nuisance.

 .ENDOC                          END DOCUMENTATION

-------------------------------------------------------------------------------------------------------------*/

int strio( IFBP ifbp, hcf_16 type, hcf_16 fid, hcf_16 offset, wci_recordp pc_addr,  int tot_len
		   BE_PAR(int word_len) ) 
{

hcf_io		data_port = (hcf_io/*;?NDIS*/)ifbp->IFB_IOBase + HREG_DATA_0;						//BAP data register
hcf_32		prot_cnt = ifbp->IFB_TickIni;
//@@initial value of cp depends on word_len, so postpone initialization: wci_bufp	cp = (wci_bufp)pc_addr;
wci_bufp	cp;
int			rc;
int			i;

	HCFTRACE( ifbp, HCF_TRACE_STRIO );
	HCFTRACEVALUE( ifbp, fid );
	HCFTRACEVALUE( ifbp, offset );
	HCFASSERT( ((long)pc_addr & (HCF_ALIGN-1) ) == 0, (hcf_16)(long)pc_addr );
	HCFASSERT( ((long)pc_addr & (HCF_ALIGN-1) ) == 0, HCF_ALIGN);
	HCFASSERT( (offset & 01) == 0, offset )
#if defined HCF_BIG_ENDIAN
	HCFASSERT( word_len == 0 || ((long)pc_addr & 1 ) == 0, (hcf_16)(long)pc_addr )
	HCFASSERT( word_len <= tot_len, word_len )
#endif // HCF_BIG_ENDIAN
/*2*/
	if ( ( rc = ifbp->IFB_DefunctStat ) == HCF_SUCCESS ) {
/*4*/	OPW( HREG_SELECT_0, fid );
		OPW( HREG_OFFSET_0, offset & 0xFFFE );
		HCF_WAIT_WHILE( IPW( HREG_OFFSET_0) & HCMD_BUSY );
/*6*/	HCFASSERT( !( IPW( HREG_OFFSET_0) & HREG_OFFSET_ERR ), offset )
		HCFASSERT( !( IPW( HREG_OFFSET_0) & HREG_OFFSET_ERR ), fid )
		if ( prot_cnt == 0 ) {
			HCFASSERT( DO_ASSERT, fid )
			HCFASSERT( DO_ASSERT, offset )
			rc = ifbp->IFB_DefunctStat = HCF_ERR_DEFUNCT_TIME_OUT;
			ifbp->IFB_CardStat |= CARD_STAT_DEFUNCT;
			ifbp->IFB_HCF_Tallies.MiscErr++;													
		}
		if ( tot_len ) {
#if defined HCF_BIG_ENDIAN
			i = word_len;
/*8*/		while ( i-- ) {
				if ( type == IO_IN ) *pc_addr = IN_PORT_WORD( data_port );
				else OUT_PORT_WORD( data_port, *pc_addr );		//@@presume no side effect on pc_addr
				pc_addr++; 										//@@now we get the right increment (by 2))
			}
			i = tot_len - word_len;
#else
			i = tot_len;
#endif //HCF_BIG_ENDIAN
			cp = (wci_bufp)pc_addr;								/* @@now we can correctly initialize the 
																 * (awkward) char pointer on BE as well as LE 
																 * note that the STRING macros are defined 
																 * as using char pointer to allow arbitrary 
																 * alignment of the buffers supplied by MSF
																 * or higher layer
																 */
																/* some people make nasty macros, better be
																   prepared by turning them into compound
																   statements
																  */ 
			if ( type == IO_IN ) { IN_PORT_STRING( data_port, cp, i ); }
			else { OUT_PORT_STRING( data_port, cp, i ); }
		}
	    if ( IPW( HREG_SW_0 ) != HCF_MAGIC ) rc =  HCF_ERR_NO_NIC;
	}	
/*10*/	
	if ( rc != HCF_SUCCESS && type == IO_IN ) {
#if defined HCF_BIG_ENDIAN
		pc_addr -= word_len;			//@@ restoration of pc_addr may be needed in case of BE
#endif //HCF_BIG_ENDIAN
		for ( i = tot_len; i; i-- ) *pc_addr++ = 0;
	}
	HCFTRACE( ifbp, HCF_TRACE_STRIO | HCF_TRACE_EXIT );
    return rc;
} // strio
#endif // HCF_TYPE_USB


/*******************************************************************************************************************

.MODULE			update_mic_dummy / update_mic
.DESCRIPTION

.ARGUMENTS
  void update_mic/update_mic_dummy( hcf_32* p, hcf_32 M )

.RETURNS

.NARRATIVE

 Parameters:
  p		address of the MIC
  M		32 bit value to be processed by the MIC calculation engine

 Description:
 	update_mic_dummy is used when MIC is not yet / not anymore active. Using a dynamic routine pointer,
	which is updated once per Tx or Rx frame, is supposed to be a faster and more elegant implementation 
	then determiming whether or not to perform an actual MIC calculation a number of times per Rx or Tx 
	frame

	update_mic is the implementation of the MIC algorithm. It is a monkey-see monkey-do copy of 
 	Michael::appendByte() 
	of Appendix C of 

	The MIC is located in the IFB (to alleviate the byte versus double-word boundary problems).
	The MIC is seperate for Tx and Rx, thus allowing hcf_send_msg to occur between hcf_service_nic and
	hcf_rcv_msg
.DIAGRAM

.NOTICE
.ENDOC				END DOCUMENTATION

***************************************************************************************************************/
void
update_mic_dummy( hcf_32* p, hcf_32 M )
{
	/*NOP*/;
} // update_mic_dummy

#if (HCF_TYPE) & HCF_TYPE_SSN

#define ROL32( A, n ) ( ((A) << (n)) | ( ((A)>>(32-(n)))  & ( (1UL << (n)) - 1 ) ) )
#define ROR32( A, n ) ROL32( (A), 32-(n) )

#define L	*p
#define R	*(p+1)

void
update_mic( hcf_32* p, hcf_32 M )
{
		L ^= M;
		R ^= ROL32( L, 17 );
		L += R;
		R ^= ((L & 0xff00ff00) >> 8) | ((L & 0x00ff00ff) << 8);
		L += R;
		R ^= ROL32( L, 3 );
		L += R;
		R ^= ROR32( L, 2 );
		L += R;
} // update_mic
#undef R
#undef L
#endif // HCF_TYPE_SSN

#if (HCF_TYPE) & HCF_TYPE_HII // lets forget all those workarounds, it is unworkable over different HCF versions 
/*******************************************************************************************************************

.MODULE			ack_the_bastard
.DESCRIPTION

.ARGUMENTS
  void ack_the_bastard( IFBP ifbp, hcf_16 mask ) 

.RETURNS

.NARRATIVE
  bufp	address of buffer
  mask

 Parameters:

 Description:

.DIAGRAM

.NOTICE
.ENDOC				END DOCUMENTATION

***************************************************************************************************************/
void
ack_the_bastard( IFBP ifbp, hcf_16 mask ) 
{
hcf_32 prot_cnt = 0;

	if ( ifbp->IFB_FWIdentity.comp_id >= COMP_ID_FW_STA	) /*;?why this test */ prot_cnt = ifbp->IFB_TickIni;
	HCF_WAIT_WHILE( ( IPW( HREG_EV_STAT ) & HREG_EV_REPAIR_ACK ) == 0 );	
	HCFASSERT(prot_cnt, mask)
	OPW( HREG_EV_ACK, mask | HREG_EV_REPAIR_ACK );
	OPW( HREG_EV_ACK, (mask &~HREG_EV_ALLOC) | HREG_EV_REPAIR_ACK );
}
#endif // HCF_TYPE_HII


//#if 0 //POtentially usefull tidbits to remember to set in updated comments on a more appropriate place
///******************************************************************************************************************
//	As side effect of the Hermes Initialize command, the interrupts are disabled (The Hermes Register HREG_INT_EN == 0)
//
// -: In some environments (e.g. Windows NT with SystemSoft extensions to support Suspend/Resume the I/O space
// 	is not mapped for a prolonged time after Resume. This causes the HCF to fail Busy bit waits etc after
// 	approx 2.5 seconds. Since at higher levels there is a 2 second recovery timer, there is a retry immediately,
// 	causing the system to hang for 20 seconds (in the view of the end-user). By checking whether there is
// 	at least something with a "write&read back" characteristic, the HCF verifies whether the Hermes is present.
// 	Note that while the I/O space is not yet mapped, this test fails, when the I/O space is mapped and a Hermes
// 	is present (and stabilized after power up, but regardless of the Initialize command), this test succeeds.
// 	If the test succeeds because the wrong card is mapped, this does not matter within the problem at hand.
// -:	Preset the return code at Time out. Since ini_hermes is part of the initialization of the card and
// 	since the strategy is to detect problems as a side effect of "necessary" actions to initialize the card,
// 	ini_hermes has, in deviation of the cmd_exe strategy, an "wait for busy bit drop" before the Hermes
// 	Initialize command is executed.
//	The additional complication that no calibrated value for the protection count can be assumed since calibrate()
//	may not yet have determined a calibrated value (a catch 22) is handled by the initial value set at INI_TICK_INI
//	by hcf_connect). This approach is considered safe, because:
//	 o the HCF does not use the pipeline mechanism of Hermes commands.
//	 o the likelihood of failure (the only time when protection count is relevant) is small.
//	 o the time will be sufficiently large on a fast machine (busy bit drops on good NIC before counter expires)
//	 o the time will be sufficiently small on a slow machine (counter expires on bad NIC before the end user
//	   switches the power off in despair
//	The time needed to wrap a 32 bit counter around is longer than many humans want to wait, hence the more or
//	less arbitrary value of 0x40000L is chosen, assuming it does not take too long on an XT and is not too short on
//	a scream-machine.
//	Note that early models of the Hermes needed some arbitrary read before the first write activity to operate
//	stable (ref tracker 27). This busy drop wait achieves that function as a side effect.
//	If the Hermes passes the superficial health test of waiting for the Busy bit drop, the Hermes Initialize
//	command is executed. The Initialize command acts as the "best possible" reset under HCF control. A more
//	"severe/reliable" reset is under MSF control via the COR register.
// -:	Initialize the Hermes. In order to "roll back" from AP mode to STA mode, the undocumented bit 0x0100 must be
//	set. This bit is controlled via the cntl parameter of init.
//	The Hermes H/W controlled bits (i.e. HREG_EV_STAT) are not guaranteed to be cleared as result of the initialize
//	command. However it is guaranteed that no more events are in the pipeline. Ack-ing indiscriminately all events
//	resolves this problem. An alternative would be to use the resulting value of
//	"IPW( HREG_EV_STAT )" rather than 0xFFFF to specifically only reset those events
//	which are set. This strategy is considered to be only more aesthetically pleasing (if that).
//#endif // 0 usefull tidbits	
//
