/*
 * File:    DHFPLUG.C
 * $Date: 2003/05/30 11:40:20 $ $Revision: 1.1.1.1 $
 *
 * Abstract: Contains functions to apply special plug-rules to customize to-plug 
 *           Production Data
 *
 * Contents: function apply_plug_rules
 *
 * Description:
 *    This software module takes care of special costumized production data
 *    items. It applies special plug-rules when a new item is introduced or
 *    the layout of an existing item has changed.
 *
 * Author:  John Meertens
 *
 * Changed:
 *
 * SOFTWARE LICENSE
 *
 * This software is provided subject to the following terms and conditions,
 * which you should read carefully before using the software.  Using this
 * software indicates your acceptance of these terms and conditions.  If you do
 * not agree with these terms and conditions, do not use the software.
 *
 * Copyright  2003 Agere Systems Inc.
 * All rights reserved.
 *
 * Redistribution and use in source or binary forms, with or without
 * modifications, are permitted provided that the following conditions are met:
 *
 * . Redistributions of source code must retain the above copyright notice, this
 *    list of conditions and the following Disclaimer as comments in the code as
 *    well as in the documentation and/or other materials provided with the
 *    distribution.
 * 
 * . Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following Disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 * . Neither the name of Agere Systems Inc. nor the names of the contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * Disclaimer
 *
 * THIS SOFTWARE IS PROVIDED AS IS AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, INFRINGEMENT AND THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  ANY
 * USE, MODIFICATION OR DISTRIBUTION OF THIS SOFTWARE IS SOLELY AT THE USERS OWN
 * RISK. IN NO EVENT SHALL AGERE SYSTEMS INC. OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, INCLUDING, BUT NOT LIMITED TO, CONTRACT, STRICT 
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGE.
 *
 */



/*-----------------------------------------------------------------------------
 * Includes
 *---------------------------------------------------------------------------*/

#include "dhf.h"

#ifndef DSF_HERMESII


#ifndef STDTYPES
#include "stdtypes.h"
#endif

#include "s3defs.h"

/*-----------------------------------------------------------------------------
 * Prototypes
 *---------------------------------------------------------------------------*/

// defined in DHF.C; see there for comments
EXTERN_C hcf_16 *find_record_in_pda( hcf_16 *pdap, hcf_16 code, hcf_16 *len );

/*-----------------------------------------------------------------------------
 * Default Plugging values ;
 *---------------------------------------------------------------------------*/

const hcf_8 byDefPDRRecord_0005[2+2+10] = { 	
	/*L*/ 0x06, 0x00, 
	/*T*/ 0x05, 0x00,
	/*V*/ 0x00, 0x00, 0x06, 0x00, 0x01, 0x00, 0x01, 0x00, 0x01, 0x00 } ;
	
const hcf_8 byDefPDRRecord_0108[2+2+4] = { 	
	/*L*/ 0x03, 0x00, 
	/*T*/ 0x08, 0x01,
	/*V*/ 0x00, 0x00, 0x00, 0x00 } ;

const hcf_8 byDefPDRRecord_0109[2+2+10] = { 	
	/*L*/ 0x06, 0x00, 
	/*T*/ 0x09, 0x01,
	/*V*/ 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00 } ;

const hcf_8 byDefPDRRecord_0150[2+2+2] = { 	
	/*L*/ 0x02, 0x00, 
	/*T*/ 0x50, 0x01,
	/*V*/ 0x00, 0x3F } ;

const hcf_8 byDefPDRRecord_0160[2+2+28] = { 	
	/*L*/ 0x0F, 0x00, 
	/*T*/ 0x60, 0x01,
	/*V*/ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
		  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
 } ;

const hcf_8 byDefPDRRecord_0161[2+2+256] = { 	
	/*L*/ 0x81, 0x00, 
	/*T*/ 0x61, 0x01,
	/*V*/ 0x3F, 0x01, 0x3F, 0x01, 0x3F, 0x01, 0x3F, 0x01, 0x3F, 0x01, 0x3F, 0x01, 0x3F, 0x01, 0x3F, 0x01, 
		  0x3F, 0x01, 0x3F, 0x01, 0x3F, 0x01, 0x3F, 0x01, 0x3F, 0x01, 0x3F, 0x01, 0x3F, 0x01, 0x3F, 0x01, 
		  0x3F, 0x01, 0x3E, 0x01, 0x3E, 0x01, 0x3D, 0x01, 0x3D, 0x01, 0x3C, 0x01, 0x3C, 0x01, 0x3B, 0x01, 
		  0x3B, 0x01, 0x3A, 0x01, 0x3A, 0x01, 0x39, 0x01, 0x39, 0x01, 0x38, 0x01, 0x38, 0x01, 0x37, 0x01, 
		  0x37, 0x01, 0x36, 0x01, 0x36, 0x01, 0x35, 0x01, 0x35, 0x01, 0x34, 0x01, 0x34, 0x01, 0x33, 0x01, 
		  0x33, 0x01, 0x32, 0x01, 0x32, 0x01, 0x31, 0x01, 0x31, 0x01, 0x30, 0x01, 0x30, 0x01, 0x7B, 0x01, 
		  0x7B, 0x01, 0x7A, 0x01, 0x7A, 0x01, 0x79, 0x01, 0x79, 0x01, 0x78, 0x01, 0x78, 0x01, 0x77, 0x01, 
		  0x77, 0x01, 0x76, 0x01, 0x76, 0x01, 0x75, 0x01, 0x75, 0x01, 0x74, 0x01, 0x74, 0x01, 0x73, 0x01, 
		  0x73, 0x01, 0x72, 0x01, 0x72, 0x01, 0x71, 0x01, 0x71, 0x01, 0x70, 0x01, 0x70, 0x01, 0x68, 0x01, 
		  0x68, 0x01, 0x67, 0x01, 0x67, 0x01, 0x66, 0x01, 0x66, 0x01, 0x65, 0x01, 0x65, 0x01, 0x57, 0x01, 
		  0x57, 0x01, 0x56, 0x01, 0x56, 0x01, 0x55, 0x01, 0x55, 0x01, 0x54, 0x01, 0x54, 0x01, 0x53, 0x01, 
		  0x53, 0x01, 0x52, 0x01, 0x52, 0x01, 0x51, 0x01, 0x51, 0x01, 0x50, 0x01, 0x50, 0x01, 0x48, 0x01, 
		  0x48, 0x01, 0x47, 0x01, 0x47, 0x01, 0x46, 0x01, 0x46, 0x01, 0x45, 0x01, 0x45, 0x01, 0x44, 0x01, 
		  0x44, 0x01, 0x43, 0x01, 0x43, 0x01, 0x42, 0x01, 0x42, 0x01, 0x41, 0x01, 0x41, 0x01, 0x40, 0x01, 
		  0x40, 0x01, 0x40, 0x01, 0x40, 0x01, 0x40, 0x01, 0x40, 0x01, 0x40, 0x01, 0x40, 0x01, 0x40, 0x01, 
		  0x40, 0x01, 0x40, 0x01, 0x40, 0x01, 0x40, 0x01, 0x40, 0x01, 0x40, 0x01, 0x40, 0x01, 0x40, 0x01
 } ;


/*-----------------------------------------------------------------------------
 * Function:    apply_plug_rules
 *
 * Abstract:    Locates the record with code in the PDA, and subsequently adjusts 
 *              the record, if needed, by applying special plug-rule(s). 
 *
 * Arguments:   pdap       - pointer to production data area.
 *              record     - pointer to specific record in PDA, may be null.
 *                           For certain codes this record is "allocated" by
 *                           this routine.
 *              code       - code of production data record to be (possibly) 
 *                           adjusted.
 *
 * Returns:     DHF_SUCCESS  - if everthing OK
 *              DHF_IGNORE   - skip plugging of this record
 *              DHF_FAILURE  - plugging error
 *
 * Description:
 *   This function will apply special plugging rules to the production data
 *   record passed in, if such a special rule exists for the record.
 *   In some cases it will create a new production data record.
 *
 * Notes:
 * - This is an internal DHF-function, i.e. it does not occur in the DHF-
 *   interface. It is not a static function as it is used in file DHF.C.
 *---------------------------------------------------------------------------*/
EXTERN_C int apply_plug_rules( hcf_16 *pdap, hcf_16 **record, hcf_16 code ) 
{
    int rc = DHF_SUCCESS;

    *record = find_record_in_pda( pdap, code, NULL );

    //!HWi return default a error if record is NOT in PDA
    if(*record == NULL) {
    	rc = DHF_FAILURE ;
    }

    // Adaption code to handle Low Power NICs
    // If PDR 0110 - Modem REFDAC Set-up Values Normal requested
    if ( code == 0x110 ) {
        // Clear error condition first
        rc = DHF_SUCCESS;
        // If PDA contains PDR 0111 - Modem REFDAC Set-up Values Outdoor
        if ( ( *record = find_record_in_pda( pdap, 0x111, NULL ) ) != NULL ) {
            // Plug PDR 0111 values in PDR 0110 related location
        } else {
            // Else If PDA contains PDR 0110 - Modem REFDAC Set-up Values Normal
            if ( ( *record = find_record_in_pda( pdap, 0x110, NULL ) ) != NULL ) {
                // Plug PDR 0110 values in PDR 0110 related location
                // No change is needed
            } else {
                //  Else PDR 0110 and PDR 0111 not in PDA
                //  Handle 'plug error
                rc = DHF_FAILURE;
            }
        }
    }

    //  If PDR 0120 - Modem VGDAC Set-up Values Normal requested
    if ( code == 0x120 ) {
        // Clear error condition first
        rc = DHF_SUCCESS;
        // If PDA contains PDR 0121 - Modem VGDAC Set-up Values Outdoor
        if ( ( *record = find_record_in_pda( pdap, 0x121, NULL ) ) != NULL ) {
            // Plug PDR 0121 values in PDR 0120 related location
        } else {
            // ElseIf PDA contains PDR 0120 - Modem VGDAC Set-up Values Normal
            if ( ( *record = find_record_in_pda( pdap, 0x120, NULL ) ) != NULL ) {
                // Plug PDR 0120 values in PDR 0120 related location
                // No change is needed
            } else {
                // Else PDR 0120 and PDR 0121 not in PDA
                // Handle plug error
                rc = DHF_FAILURE;
            }
        }
    }

    // Adaption code for plug record 0x150
    // If PDR 0150 - Antenna diversity
    if ( code == 0x150 ) {
        if ( *record == NULL ) {
        	*record = (hcf_16 *) byDefPDRRecord_0150 ;
        	/* Ignore and leave it default */
    		rc = DHF_SUCCESS ;
        }
    }

    // Adaption code for plug record 0x160
    // If PDR 0160 - Modem VCO band Set-up
    if ( code == 0x160 ) {
        if ( *record == NULL ) {
            /* Give back the defaults */
        	*record = (hcf_16 *) byDefPDRRecord_0160 ;
            rc = DHF_SUCCESS;
        }
    }

	// Adaption code for plug record 0x161
    // If PDR 0161 - Modem Rx Gain Table Values
    if ( code == 0x161 ) {
        if ( *record == NULL ) {
            /* Give back the defaults */
        	*record = (hcf_16 *) byDefPDRRecord_0161 ;
            rc = DHF_SUCCESS;
        }
    }

	
    // Adaption code for plug record 0x005
    // If PDR 0005 - HWIF Compatiblity
    if ( code == 0x005 ) {
        // If available
        if ( *record == NULL ) {
            /* Give back the defaults */
        	*record = (hcf_16 *) byDefPDRRecord_0005 ;
            rc = DHF_SUCCESS;
        }
    }

    // Adaption code for plug record 0x108
    // If PDR 0108 - PPPPSign
    if ( code == 0x108 ) {
        // If available
        if ( *record == NULL ) {
            /* Give back the defaults */
        	*record = (hcf_16 *) byDefPDRRecord_0108 ;
            rc = DHF_SUCCESS;
        }
    }

    // Adaption code for plug record 0x109
    // If PDR 0109 - PPPPProf
    if ( code == 0x109 ) {
        // If available
        if ( *record == NULL ) {
            /* Give back the defaults */
        	*record = (hcf_16 *) byDefPDRRecord_0109 ;
            rc = DHF_SUCCESS;
        }
    }

    // Adaption code for testrecords
    if ( code >= 0xFF00 ) {
        // If available
        if ( *record == NULL ) {
            /* Ignore and leave it default */
            rc = DHF_IGNORE;
        }
    }

    return rc;
}   // apply_plug_rules

 #endif // #ifndef DSF_HERMESII
