/*
 * File:    DHFPDAR.C
 * $Date: 2003/05/30 11:40:20 $ $Revision: 1.1.1.1 $
 *
 * Abstract: Contains functions to be able to access the PDA on the NIC.
 *
 * Description:
 *   This module provides functionality to load the PDA from the NIC into an
 *   internal buffer, to update specific PA records in this internal buffer by
 *   retrieving them and writing them back, and finally to program the NIC with
 *   the adjusted PDA.
 *
 * Notes: 
 * - the PDA functions provided in this file are not for public use.
 * - this file must be copied to file DHFPDA.C before building applications
 *   that want to access the PDA. Otherwise, copy file DHFPDAD.C to DHFPDA.C.
 *
 * Author: John Meertens
 *
 * Started: 26-04-2000
 *
 * Changed:
 *
 * SOFTWARE LICENSE
 *
 * This software is provided subject to the following terms and conditions,
 * which you should read carefully before using the software.  Using this
 * software indicates your acceptance of these terms and conditions.  If you do
 * not agree with these terms and conditions, do not use the software.
 *
 * Copyright  2003 Agere Systems Inc.
 * All rights reserved.
 *
 * Redistribution and use in source or binary forms, with or without
 * modifications, are permitted provided that the following conditions are met:
 *
 * . Redistributions of source code must retain the above copyright notice, this
 *    list of conditions and the following Disclaimer as comments in the code as
 *    well as in the documentation and/or other materials provided with the
 *    distribution.
 * 
 * . Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following Disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 * . Neither the name of Agere Systems Inc. nor the names of the contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * Disclaimer
 *
 * THIS SOFTWARE IS PROVIDED AS IS AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, INFRINGEMENT AND THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  ANY
 * USE, MODIFICATION OR DISTRIBUTION OF THIS SOFTWARE IS SOLELY AT THE USERS OWN
 * RISK. IN NO EVENT SHALL AGERE SYSTEMS INC. OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, INCLUDING, BUT NOT LIMITED TO, CONTRACT, STRICT 
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGE.
 *
 */


/*-----------------------------------------------------------------------------
 *
 * Includes
 *
 *---------------------------------------------------------------------------*/
#include "dhf.h"

#ifndef DHFPDA_H
#include "dhfpda.h"
#endif

#ifndef DHF_WCI

#ifdef DHF_WIN
	#ifndef UIL_WIN_H
	#include "uil_win.h"
	#endif
#endif

#ifdef DHF_DOS
	#ifndef UIL_DOS_H
	#include "uil_dos.h"
	#endif
#endif

#endif


/*-----------------------------------------------------------------------------
 *
 * Assert function
 *
 *---------------------------------------------------------------------------*/
#ifdef DSF_ASSERT
#define DSFASSERT( x ) if (!(x)) { dsf_assert( TEXT(__FILE__), TEXT(__LINE__) ); }
#else
#define DSFASSERT( x )
#endif


/*-----------------------------------------------------------------------------
 * Global variables needed by functions download_start, download_next, and
 * download_stop
 *---------------------------------------------------------------------------*/

//TODO: static seems to lead to problems in Win2000-platform/compiler (and others???)
//      so currently removed
#ifdef DSF_ASSERT
/*static*/ BOOL      dhf_connected = FALSE;      // only used for assert's
#endif
#ifdef DHF_WCI
EXTERN_C IFBP      dhf_ifbp;                     // Pointer to the ifb
#endif


/*-----------------------------------------------------------------------------
 * External functions
 *---------------------------------------------------------------------------*/

EXTERN_C hcf_16 *find_record_in_pda(                  // defined in DHF.C
    hcf_16 *pdap,            // pointer to PDA
    hcf_16 code              // code to be looked for in the PDA
);

EXTERN_C hcf_16 calc_crc( hcf_8 *cp, hcf_32 count);   // defined in DHF.C

EXTERN_C int wait_for_dl_stat( void );                // defined in DHF.C

EXTERN_C int copy_pda_rec( LTVP dest, LTVP src );     // defined in DHF.C


/*-----------------------------------------------------------------------------
 * External data
 *---------------------------------------------------------------------------*/

#ifdef DSF_ALLOC
EXTERN_C hcf_16 *cfg_prod_data;
#else
EXTERN_C hcf_16 cfg_prod_data[];
#endif



/*-----------------------------------------------------------------------------
 *
 * Function:    dhf_add_pda_record( hcf_16 code, LTVP rec )
 *
 * Abstract:    adds a PDA-record to the (internally stored) PDA
 *
 * Returns:
 *    DHF_SUCCESS               - PDA-record successfully stored in PDA
 *    DHF_FAILURE               - an error occurred
 *
 * Description:
 *   If the record to be added is allready present in the PDA, it is overwritten,
 *   else the record is added to the end of PDA. If not enough room is available
 *   an error is returned.
 *   If the record has been added to the end the PDA_END record is shifted to 
 *   the new end position.
 *
 * Notes:
 * - use dhf_program_pda to actually write the complete PDA to the NIC
 * - this function does not check rec in any way
 * - this function does not check the length of the destination record
 * - the CRC will be recalculated by function dhf_program_pda
 *---------------------------------------------------------------------------*/
int dhf_add_pda_record( 
    LTVP rec            // record to be added; may not be NULL
) {
    int rc = DHF_SUCCESS;
    hcf_16 *pdap;
    
    DSFASSERT( dhf_connected == TRUE )
    DSFASSERT( rec != NULL )
        
#ifdef DSF_ALLOC
    pdap = find_record_in_pda( (hcf_16*)cfg_prod_data + 2, rec->typ & CODEMASK );     //;?requires that pri plug records do not overlap PDA plug records
#else
    pdap = find_record_in_pda( (hcf_16*)&cfg_prod_data + 2, rec->typ & CODEMASK );    //;?requires that pri plug records do not overlap PDA plug records
#endif

    if ( pdap != NULL ) {
        rc = copy_pda_rec( (LTVP)pdap, rec );
    } else {
#ifdef DSF_ALLOC
        pdap = find_record_in_pda( (hcf_16*)cfg_prod_data + 2, PDA_END );
#else
        pdap = find_record_in_pda( (hcf_16*)&cfg_prod_data + 2, PDA_END );
#endif

        DSFASSERT( pdap != NULL )
        
        // Check if there's enough room before copying.
        // +1 to accommodate the len field.
#ifdef DSF_ALLOC
        if ( (hcf_16)(cfg_prod_data + PDA_SIZE - pdap ) < rec->len + 1 ) {
#else
        if ( (hcf_16)(&cfg_prod_data[PDA_SIZE] - pdap ) < rec->len + 1 ) {
#endif
            rc = DHF_FAILURE;
        } else {
            rc = copy_pda_rec( (LTVP)pdap, rec );
        }
        if ( rc == DHF_SUCCESS ) {
            pdap[rec->len + 1] = pdap[rec->len + 2] = 0;   //restore PDA_END record
        }
    }
    
    return rc;
}   // dhf_add_pda_record


/*-----------------------------------------------------------------------------
 *
 * Function:    dhf_get_pda_record( hcf_16 code )
 *
 * Abstract:    retrieves a PDA-record from the (internally stored) PDA
 *
 * Returns:
 *    LTV-pointer to retrieved PDA-record, or NULL if record with code was
 *    not found
 *
 * Description:
 *---------------------------------------------------------------------------*/
LTVP dhf_get_pda_record( hcf_16 code ) {
    LTVP rc = NULL;

    DSFASSERT( dhf_connected == TRUE )
     
#ifdef DSF_ALLOC
    rc = (LTVP)find_record_in_pda( (hcf_16*)cfg_prod_data + 2, code & CODEMASK );     //;?requires that pri plug records do not overlap PDA plug records
#else
    rc = (LTVP)find_record_in_pda( (hcf_16*)&cfg_prod_data + 2, code & CODEMASK );     //;?requires that pri plug records do not overlap PDA plug records
#endif

    return rc;
}   // dhf_get_pda_record


/*-----------------------------------------------------------------------------
 *
 * Function:    dhf_program_pda
 *
 * Abstract:    Programs NIC with (new) PDA.
 *
 * Returns:
 *   DHF_SUCCESS    - PDA successfully programmed
 *   DHF_ERR_NO_NIC - no NIC available
 *
 * Description:
 *   Locates the end of the PDA
 *   Calculates the CRC of the PDA (as it may have been changed by the user)
 *     The CRC is stored at the end of the PDA, directly after the end_of_pda 
 *     record with code PDA_END (see DHF.H)
 *   Plugs the new CRC in the PDA
 *   Finally programs the PDA which consists of a complete download cycle: 
 *     put a CFG_DL(N)V_START record, the actual PDA, and finally 
 *     CFG_DL_STOP; when downloading to non-volatile memory waits for the
 *     proper result to appear in the mailbox.
 *
 * Notes:
 *---------------------------------------------------------------------------*/
int dhf_program_pda( void ) {
    int rc = DHF_SUCCESS;
    hcf_16 crc;
    hcf_16 *crc_rec;
    LTVP pda_ltv;
    hcf_16 buf[20];
    LTVP tmp;

    DSFASSERT( dhf_connected == TRUE )

#ifdef DSF_ALLOC
    pda_ltv = (LTVP)cfg_prod_data;
    crc_rec = (LTVP)find_record_in_pda( (hcf_16*)cfg_prod_data + 2, PDA_END );
#else
    pda_ltv = (LTVP)&cfg_prod_data;
    crc_rec = find_record_in_pda( (hcf_16*)&cfg_prod_data + 2, PDA_END );
#endif

    DSFASSERT( crc_rec != NULL )

    if ( crc_rec != NULL ) {

        // The CRC is stored in the val the PDA_END record in the PDA. 
        crc_rec = ((LTVP)crc_rec)->val;
        
        // Recalculate and plug the CRC
        crc = calc_crc( (hcf_8*)pda_ltv->val, (crc_rec - pda_ltv->val) * sizeof(hcf_16) );
        *crc_rec = crc;

        // Start the download cycle by putting CFG_DL_START
        tmp = (LTVP)buf;
        tmp->len = LOF(CFG_DL_START_STRCT);
        tmp->typ = CFG_DLNV_START;
        tmp->val[0] = (hcf_16)(PDA_ADDRESS & 0xFFFF);
        tmp->val[1] = (hcf_16)(PDA_ADDRESS >> 16);
#ifdef DHF_WCI
        rc = hcf_put_info( dhf_ifbp, tmp );
#else
        rc = uil_put_info( tmp );
#endif

//TODO: do we always download to non-volatile memory???
        // Put the actual PDA information and wait for result in the mailbox when 
        // downloading to non-volatile memory
        if ( rc == UIL_SUCCESS ) {
            pda_ltv->typ = CFG_DLNV_DATA;   // temporarily change typ field
#ifdef DHF_WCI
            rc = hcf_put_info( dhf_ifbp, pda_ltv );
#else
            rc = uil_put_info( pda_ltv );
#endif
            pda_ltv->typ = CFG_PROD_DATA;   // restore original value
        }
        if ( rc == UIL_SUCCESS ) {
            wait_for_dl_stat();
        }
        
        // Finalize download cycle by putting CFG_DL_STOP
        if ( rc == UIL_SUCCESS ) {
            tmp->len = 1;
            tmp->typ = CFG_DL_STOP;
#ifdef DHF_WCI
            rc = hcf_put_info( dhf_ifbp, tmp );
#else
            rc = uil_put_info( tmp );
#endif
        }
    }

    return rc;
}   // dhf_program_pda


/*-----------------------------------------------------------------------------
 *
 * Function: dhf_put_pda_record( hcf_16 code, LTVP rec )
 *
 * Abstract: puts a PDA-record in the (internally stored) PDA
 *
 * Returns:
 *    DHF_SUCCESS  - PDA-record successfully stored in PDA
 *    DHF_FAILURE  - Record with code not found in PDA
 *
 * Description:
 *   Locates the record in the PDA. If the record is not found an error
 *   is returned. Otherwise, the record is copied to the PDA.
 *
 * Notes:
 * - use dhf_program_pda to actually write the complete PDA to the NIC
 * - this function does not check rec in any way
 * - this function does not check the length of the destination record
 *---------------------------------------------------------------------------*/
int dhf_put_pda_record( hcf_16 code, LTVP rec ) {
    int rc = DHF_SUCCESS;
    hcf_16 *pdap;
    
    DSFASSERT( dhf_connected == TRUE )
    DSFASSERT( rec != NULL )
        
#ifdef DSF_ALLOC
    pdap = find_record_in_pda( (hcf_16*)cfg_prod_data + 2, code & CODEMASK );     //;?requires that pri plug records do not overlap PDA plug records
#else
    pdap = find_record_in_pda( (hcf_16*)&cfg_prod_data + 2, code & CODEMASK );    //;?requires that pri plug records do not overlap PDA plug records
#endif

    if ( pdap != NULL ) {
        copy_pda_rec( (LTVP)pdap, rec );
    } else {
        rc = DHF_FAILURE;
    }
    
    return rc;
}   // dhf_put_pda_record


