
#ifndef DHF_H
#define DHF_H


/*-----------------------------------------------------------------------------
 *
 * File: DHF.H
 * $Date: 2003/05/30 11:40:20 $ $Revision: 1.1.1.1 $
 *
 * Contents: structure definitions and function prototypes for unit DHF.
 *
 * Description:
 *   Acronyms: DHF - Download Handling Functions, the routines in this unit
 *             DSF - Download Specific Functions, code written on top of the DHF
 *             WSU - WaveLAN Station Update
 *
 *   DHF is a module that provides a number of routines to download firmware 
 *   images (primary, station, an access point) to volatile or nonvolatile RAM 
 *   in WaveLAN/IEEE NICs. To achieve this DHF makes use of the WaveLAN/IEEE 
 *   UIL- or WCI-module. Use #define DHF_WCI to set which of the two you want
 *   (see below). 
 *
 *   Download to non-volatile RAM is used to update a WaveLAN/IEEE NIC to new
 *   firmware. Normally this will be an upgrade to newer firmware, although
 *   downgrading to older firmware is possible too, in principle.
 *
 *   Download to volatile RAM is used (among other things?) to load a NIC in a
 *   WaveLAN Access Point with firmware at start of day.
 *
 *   DHF is (intended to be) platform-independent. Therefore, the DHF-user
 *   must provide a number of routines to deal with "system calls", the user 
 *   interface and for user-interaction: 
 *     dsf_alloc           - allocate an amount of memory
 *     dsf_callback        - Calls back the application to allow for progress 
 *                           update and/or to request the application to go to
 *                           sleep for the indicated number of milliseconds.
 *     dsf_confirm         - asks for user confirmation for an action
 *     dsf_display_message - displays a message to the user
 *     dsf_free            - free an amount of memory allocated by dsf_alloc
 *
 *   Download recipe:
 *     TODO: ???
 *     
 * Defines:
 * DHF_WCI        - define if WCI-routines must be used to access the WaveLAN-card
 *                  in stead of UIL-routines
 *
 * Notes:   
 * - When using the UIL the NIC is blocked for processes that interfere with 
 *   download, e.g. protocol stacks.
 *
 * Author: John Meertens
 *
 * Changed:
 *
 *
 *-----------------------------------------------------------------------------
 * COPYRIGHT (c) 2000 by Lucent Technologies.    All Rights Reserved.
 *
 * SOFTWARE LICENSE
 *
 * This software is provided subject to the following terms and conditions,
 * which you should read carefully before using the software.  Using this
 * software indicates your acceptance of these terms and conditions.  If you do
 * not agree with these terms and conditions, do not use the software.
 *
 * Copyright  2003 Agere Systems Inc.
 * All rights reserved.
 *
 * Redistribution and use in source or binary forms, with or without
 * modifications, are permitted provided that the following conditions are met:
 *
 * . Redistributions of source code must retain the above copyright notice, this
 *    list of conditions and the following Disclaimer as comments in the code as
 *    well as in the documentation and/or other materials provided with the
 *    distribution.
 * 
 * . Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following Disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 * . Neither the name of Agere Systems Inc. nor the names of the contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * Disclaimer
 *
 * THIS SOFTWARE IS PROVIDED AS IS AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, INFRINGEMENT AND THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  ANY
 * USE, MODIFICATION OR DISTRIBUTION OF THIS SOFTWARE IS SOLELY AT THE USERS OWN
 * RISK. IN NO EVENT SHALL AGERE SYSTEMS INC. OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, INCLUDING, BUT NOT LIMITED TO, CONTRACT, STRICT 
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGE.
 *
 *-----------------------------------------------------------------------------
 *---------------------------------------------------------------------------*/



/*- Includes ----------------------------------------------------------------*/

#if !defined _WIN32_WCE
#include "dhfcfg.h"            // contains user-configurable defines.
#endif


#ifndef STDTYPES
#include "stdtypes.h"
#endif

#ifdef _WIN32_WCE
#include <windef.h>
#endif

#ifdef DHF_WCI
#include "hcf.h"        // includes HCFCFG.H too
#endif

#ifndef HCFCFG_H
//#include "hcfcfg.h"            // needed for hcf_16, etc.
#endif

#ifndef MDD_H
#include "mdd.h"
#endif


/*---- Defines --------------------------------------------------------------*/

#define MAX_VARIANT_RANGES    20         // Specifies the maximum number of
                                         // variants a compatibility range can
                                         // have.
#define INTERFACE_SUPPLIER     0         // Supplier role of component at a
                                         // specific interface.
#define INTERFACE_ACTOR        1         // Actor role of component at a specific
                                         // interface.
#define IEEE2MBPRI            21         // IEEE 1/2 Mbit/s Primary Functions
                                         // firmware.
                                         // 22 - 30: reserved for other Primary
                                         // Functions firmware.
#define IEEE2MBSTA            31         // IEEE 1/2 Mbit/s STA Functions firmware.
#define FW_AP                 32         // AP Functions firmware.
                                         // 33 - 40: reserved for other secondary
                                         // firmware.
#define CFG_AP_IDENTITY 	 331		 // Temporary number?

#define CODEMASK	  0x0000FFFFL        // Codemask for plug records


/*---- Error numbers --------------------------------------------------------*/

#ifdef DHF_WCI

#define DHF_FAILURE 					0xff

#define DHF_SUCCESS                     HCF_SUCCESS
#define DHF_ERR_NO_NIC                  HCF_ERR_NO_NIC
// The following errors cannot occur when using the WCI
#define DHF_ERR_NO_DRIVER               HCF_SUCCESS
#define DHF_ERR_IN_USE                  HCF_SUCCESS
#define DHF_ERR_LEN						HCF_ERR_LEN

#define GET_INFO( pp )  hcf_get_info( dhf_ifbp, (LTVP)pp )
#define PUT_INFO( pp )  hcf_put_info( dhf_ifbp, (LTVP)pp )
#define ACTION( cmd )   hcf_action( dhf_ifbp, (hcf_action_cmd)cmd );

#else

#define DHF_SUCCESS                     UIL_SUCCESS
#define DHF_FAILURE                     UIL_FAILURE
#define DHF_ERR_NO_NIC                  UIL_ERR_NO_NIC
#define DHF_ERR_NO_DRIVER               UIL_ERR_NO_DRV
#define DHF_ERR_IN_USE                  UIL_ERR_IN_USE
#define DHF_ERR_LEN						UIL_ERR_LEN

#define GET_INFO( pp )  uil_get_info( (LTVP)pp )
#define PUT_INFO( pp )  uil_put_info( (LTVP)pp )
#define ACTION( cmd )   uil_action( (int)cmd );

#endif


//---- Result codes -----------------------------------------------------------
// Start at 0x80 to avoid overlay with HCF and UIL error codes
#define DHF_RES_MINIMUM                 0x80
#define DHF_IGNORE                      0x80
#define DHF_DOWNLOAD_COMPLETED          0x81
#define DHF_NOT_NEWER                   0x82
#define DHF_USER_ABORT                  0x83

//---- DHF-specific not-condensed error numbers -------------------------------
#define DHF_ERR_DIAGNOSE                0x90
#define DHF_ERR_GET_NIC_INFO            0x91
#define DHF_ERR_DOWNLOAD                0x92
#define DHF_ERR_NOT_COMPATIBLE          0x93
#define DHF_ERR_SELFTEST_FAILED         0x94

//---- Errors that are condensed by DHF-routines ------------------------------
// More information can be retrieved by function dhf_get_last_error()
#define DHF_ERR_CONDENSE_MINIMUM        0xA0
#define DHF_ERR_CRD_APP_NOT_COMPATIBLE  0xA0
#define DHF_ERR_DRV_APP_NOT_COMPATIBLE  0xA1
#define DHF_ERR_CRD_PRI_NOT_COMPATIBLE  0xA2
#define DHF_ERR_PRI_DRV_NOT_COMPATIBLE  0xA3
#define DHF_ERR_PRI_APP_NOT_COMPATIBLE  0xA4
#define DHF_ERR_CRD_STA_NOT_COMPATIBLE  0xA5
#define DHF_ERR_WEB_RESTRAINT           0xA6
#define DHF_ERR_ALLOC                   0xA7
#define DHF_ERR_CONNECT                 0xA8


//---- Message types ----------------------------------------------------------
// These message types are used by routine dsf_display_message
#define DHF_INFO                        0
#define DHF_ERROR                       1
#define DHF_WARNING                     2


//---- Component types recognized by DHF --------------------------------------
#define DHF_COMP_ID_DRIVER              0
#define DHF_COMP_ID_NIC_STA_FW          1
#define DHF_COMP_ID_NIC_PRI_FW          2
#define DHF_COMP_ID_NIC                 3
#define DHF_COMP_ID_MEMIMAGE            4


//---- Binary file uasage definitions --------------------------------------

#define BIN_FILE_CHECK		0x68774857L
#define BIN_FILE_HERMESI	0x00000001L
#define BIN_FILE_HERMESII	0x00000002L

/*---- Type definitions -----------------------------------------------------*/

/**
 * Type:     plugrecord
 *
 * Abstract: This structure represents a Plug Data Record.
 *
 * Description: This structure definition can be used to overlay the plug records in
 *              the firmware memory image.
 */

typedef struct {
    hcf_32    code;          // Code to plug
    hcf_32    addr;          // Address within the memory image to plug it in
    hcf_32    len;           // The # of bytes which are available to store it
} plugrecord;


/**
 * Type:     segmentrecord
 *
 * Abstract: This structure represents a Segment Data Record.
 *
 * Description: 
 *   This structure definition can be used to overlay the segment records in
 *   the firmware memory image.
 */

typedef struct {
    hcf_32  shift;           // Offset for shifted segment records; 0 for normal segments
    hcf_32  addr;            // Start address of the segment
    hcf_16  len;             // Length of the segment
    hcf_16  CRCadd;          // Is set when a CRC needs to be added to the segment
} segmentrecord;

/**
 * Type: CFG_RANGES_STRCT_xxx
 *
 * Abstract: array of CFG_RANGES_STRCT
 *
 * Description: 
 *   CFG_RANGES_STRCT_xxx stores all CFG_RANGE_SPEC_STRCTs of a component. 
 *   The maximum number of variant/top/bottom triplets is defined by 
 *   MAX_VARIANT_RANGES (currently 20).
 */
//HWi removed to use the RANGE20 from the HCF
//typedef struct CFG_RANGES_STRCT_xxx {
//    hcf_16    len;           // length of RID
//    hcf_16    typ;           // RID identification as defined by Hermes
//    hcf_16    role;
//    hcf_16    id;
//    CFG_RANGE_SPEC_STRCT  variant[MAX_VARIANT_RANGES+1];    //+1 for endsentinel
//} CFG_RANGES_STRCT_xxx;

/**
 * Type: memblock
 *
 * Abstract: A memory image block of a specific length
 *
 * Description: 
 *   memblock contains a list of binary image memory blocks. Each one of these
 *   contains the load address, size and pointer to the memory chunks which make 
 *   up the memory image. The memory block has in principle no limits to its size. 
 */
typedef struct {
    hcf_32    addr;          // load address of this block, 32 bit long
    hcf_16    size;          // size of the block, 16 bit word. Excluding length, type, and in the case of Hermes-II : mode and address bytes.
    hcf_16    flags;         // flags
                             //    CRC:  yes/n
                             //    TYPE: primary/station/tertiarysize
                             //
    hcf_8     *data;         // pointer to the actual F/W code/data to download
} memblock;


/**
 * Type: memimage
 *
 * Abstract: A memory image containing one or more memblocks
 *
 * Description:
 *   This type represents an entire memory image. The image can be built up of
 *   several memory memblocks. These memblocks need not reperesent contiguous 
 *   areas in memory. So the entire image can contain 'holes'. The 'last' field
 *   should ALWAYS be pointing to the final block. The 'execution' field is a 
 *   32-bit address representing the execution address of the firmware within
 *   the memory image. The 'pdaplug' field points to an array of Plug record
 *   structures. The end of the array is indicated by a plug record with length
 *   '0'. The same is valid for the 'priplug' field.
 */
typedef struct {
    memblock            *codep;           // Pointer to array of code records
    hcf_32               execution;        // Execution address of the firmware
#ifndef DSF_HERMESII
    plugrecord           *pdaplug;         // Pointer to array of production data
                                           // plug records
    plugrecord           *priplug;         // Pointer to array of primary plug records
#endif // #ifndef DSF_HERMESII
    CFG_RANGE20_STRCT   *compat;          // Pointer to the compatibility info records
    CFG_IDENTITY_STRCT   *identity;        // Pointer to the identity info records
} memimage;



/*-----------------------------------------------------------------------------
 *
 * DHF function prototypes
 *
 *---------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------
 * Performs a diagnose command on the card currently inserted to check if it is
 * operating properly.
 *
 * Returns:
 *   DHF_SUCCESS      - everthing OK
 *   DHF_ERR_NO_DRV   - no driver present
 *   DHF_ERR_NO_NIC   - no NIC present in slot
 *   DHF_ERR_DIAGNOSE - if something went wrong
 *   
 * Notes:
 * - dhf_diagnose_card() should not be used when downloading to volatile RAM.
 * - Currently, the diagnose command leaves the card in a disabled state.
-----------------------------------------------------------------------------*/
EXTERN_C int dhf_diagnose_card(
    hcf_16 res[2]                           // will contain result of diagnose
                                            // command:
                                            // val[0] - error status
                                            // val[1] - error qualifier
                                            // This information is used for 
                                            // engineering purposes only.
);


/*-----------------------------------------------------------------------------
 * Downloads a complete (primary, station, or access point) firmware image 
 * to the NIC.
 *
 * Returns:
 *   DHF_SUCCESS             - download completed successfully.
 *   DHF_ERR_NO_DRV          - no driver present
 *   DHF_ERR_NO_NIC          - no NIC present in slot
 *   DHF_ERR_NOT_COMPATIBLE  - firmware not compatible
 *   DHF_ERR_DOWNLOAD        - some other problem occurred in download (condensed)
 *
 *   when using the WCI:
 *
 *   when using the UIL:
 *   DHF_ERR_NO_DRV               - no driver present
 *   DHF_ERR_IN_USE               - the UIL-driver connection is already used 
 *                                  by another utility
 * 
 * Description:
 *   After some initialization and possibly blocking the NIC for interfering 
 *   processes, like protocol stacks, the firmware is downloaded in one or 
 *   more steps. At the end the NIC is unblocked if it was blocked at the
 *   start. 
 *
 * Notes:
 *---------------------------------------------------------------------------*/
EXTERN_C int dhf_download_firmware( 
    void                 *ifbp,             // pointer to the ifb; ignored when
                                            // using the UIL
    CFG_RANGE20_STRCT   *app_dui_actor,    // describes DUI-actor range of the 
                                            // download application; ignored when 
                                            // using the WCI
    CFG_RANGE20_STRCT   *app_pri_actor,    // describes primary-actor range of 
                                            // the download application
    memimage *firmware,                     // image to be downloaded
    BOOL to_volatile,                       // TRUE - download to volatile NIC-RAM
                                            // FALSE - download to non-volatile NIC-RAM
    BOOL confirm                            // TRUE=ask user for confirmation
                                            // FALSE=do not ask 
);


/*-----------------------------------------------------------------------------
 * Returns last error encountered by DHF.
 *
 * Returns:
 *   int
 *
 * Description:
 *   Returns the last error encountered by DHF. Resets error to DHF_SUCCESS.
 *
 * Notes:
 * - As this function resets the error to zero (DHF_SUCCESS), it must be
 *   called immediately after an error has been encountered. 
 *---------------------------------------------------------------------------*/
EXTERN_C int dhf_get_last_error( void );


#ifdef DHF_GET_RES_MSG

/*-----------------------------------------------------------------------------

  Gets message pertaining to result code argument

  Returns:
    char* pointing to message
 *---------------------------------------------------------------------------*/
EXTERN_C char *dhf_get_res_msg( 
    int res                                 // result code
);

#endif

/*-----------------------------------------------------------------------------

  Gets version information on a component.

  Returns:
    CFG_IDENTITY_STRCT* containing version information on requested component.
    If comp_id is not recognized by DHF all fields in the structure is set to 0.
 *---------------------------------------------------------------------------*/
EXTERN_C CFG_IDENTITY_STRCT *dhf_get_version_info(
  int comp_id,                              // component identifier, one of
                                            // DHF_COMP_ID_??? constants  
  memimage *firmware                        // firmware image, this parameter
                                            // is ignored if comp_id != 
                                            // DHF_COMP_ID_MEMIMAGE
);



/*-----------------------------------------------------------------------------
 *
 * Functions to be provided by the user of the DHF module. 
 *
 * Note that dsf_callback must allways be implemented, whereas the other 
 * dsf-function implementations need only be provided by UIL-users.
 *
 *---------------------------------------------------------------------------*/

#ifndef DSF_VOLATILE_ONLY
/*-----------------------------------------------------------------------------
  Calls back the application to allow for progress update and/or to request the
  application to go to sleep for the indicated number of milliseconds.
  This function will be called by dhf_download_firmware() when downloading to 
  non-volatile memory and by dhf_diagnose(). If you do not plan to use either 
  of these two options leave this function empty.
-----------------------------------------------------------------------------*/
EXTERN_C void dsf_callback( 
  int progress,                             // Progress in scale 1-1000 of total
                                            // download
  int milli_seconds                         // Number of milliseconds to sleep
);
#endif // DSF_VOLATILE_ONLY

#ifdef DSF_ASSERT

#ifndef  DO_DSFASSERT
#define DO_DSFASSERT	0
#endif

/*-----------------------------------------------------------------------------
  Activate asserts.
-----------------------------------------------------------------------------*/
EXTERN_C void dsf_assert(TCHAR *file_name, unsigned int line_number, TCHAR *szCmt, int nVal);

#endif


#ifdef DSF_ALLOC

/*-----------------------------------------------------------------------------
  Allocate memory of specified size
  
  Returns:
  - pointer to the memory block allocated, or NULL if allocation failed.
-----------------------------------------------------------------------------*/
EXTERN_C void *dsf_alloc(int size) ;

/*-----------------------------------------------------------------------------
  Free a memory block previously allocated by dsf_alloc
-----------------------------------------------------------------------------*/
EXTERN_C void dsf_free(
  void *ptr,                               // pointer to memory block previously
                                           // allocated by dsf_alloc
  int size                                 // size of block (optional)
);

#endif

#ifdef DHF_UIL

#ifdef DSF_CONFIRM

/*-----------------------------------------------------------------------------
  Ask for user confirmation for an action
-----------------------------------------------------------------------------*/
EXTERN_C BOOL dsf_confirm(char *prompt);

#endif // DSF_CONFIRM

#endif  // DHF_UIL

#if defined( DSF_DEBUG_MESSAGE )

/*-----------------------------------------------------------------------------
  Display a debug message
-----------------------------------------------------------------------------*/
EXTERN_C void dsf_debug_message(
  char *msg,                                // Debug message to be displayed
  int msg_type,                             // DHF_ERROR, DHF_WARNING, or DHF_INFO
  int wait                                  // TRUE = Wait for user response; 
                                            // FALSE = Do not wait
);

#endif  // DSF_DEBUG_MESSAGE


#ifdef DSF_BINARY_FILE

#ifndef NDIS_MINIPORT_DRIVER
EXTERN_C DWORD dhf_binary_size(TCHAR *szImageName) ;
#endif

#ifdef NDIS_MINIPORT_DRIVER
memimage *dhf_load_binary(void *pMem, ULONG dwFP) ;
#else
EXTERN_C memimage *dhf_load_binary(TCHAR *szImageName, void *pMem) ;
#endif
#endif // DSF_BINARY_FILE



#endif  // DHF_H
                      
