/*-----------------------------------------------------------------------------
 *
 * Filename : dhf.c
 *
 * Started  : 01-Mar-2000 by John Meertens
 * Rewrite  :
 *   $Date: 2003/06/19 12:12:33 $   $Revision: 1.3 $
 * Purpose  : This file contains generic functions to handle the download of
 *            NIC firmware for Lucent WaveLAN cards
 *
 * Notes:
 * - Names of exported functions are prefixed with 'dhf_'
 * - Interface routines in the DHF-module "condense" their possible results 
 *   into a simplified error/success code.
 *
 * Change history:
 * 10-7-2000: removed check on bottom level of components in function check_comp
 *
 *-----------------------------------------------------------------------------
 * COPYRIGHT (c) 1999 by Lucent Technologies. All Rights Reserved.
 *
 * SOFTWARE LICENSE
 *
 * This software is provided subject to the following terms and conditions,
 * which you should read carefully before using the software.  Using this
 * software indicates your acceptance of these terms and conditions.  If you do
 * not agree with these terms and conditions, do not use the software.
 *
 * Copyright  2003 Agere Systems Inc.
 * All rights reserved.
 *
 * Redistribution and use in source or binary forms, with or without
 * modifications, are permitted provided that the following conditions are met:
 *
 * . Redistributions of source code must retain the above copyright notice, this
 *    list of conditions and the following Disclaimer as comments in the code as
 *    well as in the documentation and/or other materials provided with the
 *    distribution.
 * 
 * . Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following Disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 * . Neither the name of Agere Systems Inc. nor the names of the contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * Disclaimer
 *
 * THIS SOFTWARE IS PROVIDED AS IS AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, INFRINGEMENT AND THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  ANY
 * USE, MODIFICATION OR DISTRIBUTION OF THIS SOFTWARE IS SOLELY AT THE USERS OWN
 * RISK. IN NO EVENT SHALL AGERE SYSTEMS INC. OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, INCLUDING, BUT NOT LIMITED TO, CONTRACT, STRICT 
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGE.
 *
 *-----------------------------------------------------------------------------
 *-----------------------------------------------------------------------------
 *  Log:   V:/dev3dev/wl_dhf/code/dhf.c_v
 *---------------------------------------------------------------------------*/


#define VERSION "2.01"

#ifndef DHF_H
#include "dhf.h"
#endif

/*-----------------------------------------------------------------------------
 * Check configuration #defines
 *---------------------------------------------------------------------------*/

#if !defined( DHF_WCI ) && !defined( DHF_UIL )
#error You must define either DHF_WCI or DHF_UIL
#endif

#if defined( DHF_WCI ) && defined( DHF_UIL )
#error You must define either DHF_WCI or DHF_UIL, but not both
#endif

#if !defined( DHF_BIG_ENDIAN ) && !defined( DHF_LITTLE_ENDIAN )
#error You must define either DHF_BIG_ENDIAN or DHF_LITTLE_ENDIAN
#endif

#if defined( DHF_BIG_ENDIAN ) && defined( DHF_LITTLE_ENDIAN )
#error You must define either DHF_BIG_ENDIAN or DHF_LITTLE_ENDIAN, but not both
#endif

#if defined( DSF_CONFIRM ) && defined( DHF_WCI )
#error DSF_CONFIRM cannot be defined if DHF_WCI is defined
#endif

#if defined( DSF_DEBUG_MESSAGE ) && defined( DHF_WCI )
//#error DSF_DEBUG_MESSAGE cannot be defined if DHF_WCI is defined
#endif


/*-----------------------------------------------------------------------------
 * Other includes
 *---------------------------------------------------------------------------*/

#ifndef DHFPDA_H
#include "dhfpda.h"
#endif

#ifndef _INC_STDLIB
#include <stdlib.h>
#endif

#ifndef _INC_STRING
#include <string.h>
#endif

#if defined( DSF_DEBUG_MESSAGE )
#ifndef _STDIO_H
#include <stdio.h>
#endif
#endif

#ifdef DHF_WCI

#ifndef HCFCFG_H
#include "hcfcfg.h"
#endif

#endif // DHF_WCI


#ifdef DHF_UIL

#ifdef DHF_WIN
#ifndef UIL_WIN_H
#include "uil_win.h"
#endif
#endif

#ifdef DHF_DOS
#ifndef UIL_DOS_H
#include "uil_dos.h"
#endif
#endif

#if defined( DHF_GET_RES_MSG ) && !defined( DHFMSG_H )
#include "dhfmsg.h"
#endif

#endif // DHF_UIL


/*-----------------------------------------------------------------------------
 *
 * Defines, data structures, and global variables
 *
 *---------------------------------------------------------------------------*/

// DSFDBG    - normal debug messages
// DSFDBGEXT - extended debug messages (for engineering purposes only)
#if defined( DSF_DEBUG_MESSAGE ) && !defined( DHF_WCI )

#define DSFDBG(x)       x
#define DSFDBGEXT(x)			// Define DSFDBGEXT if required

#else

// Both DSFDBG and DSFDBGEXT must be empty here
#define DSFDBG(x)
#define DSFDBGEXT(x)

#endif


#if defined DSF_ASSERT && (defined _DEBUG || (defined DBG && DBG != 0))
#define DSFASSERT(x, szCmt, nVal) if(!(x)) dsf_assert( TEXT(__FILE__), (__LINE__), szCmt, nVal)
#else
#define DSFASSERT(x, szCmt, nVal)
#endif



#ifdef DHF_BIG_ENDIAN
#ifndef CNV_LITTLE_TO_INT
#define CNV_LITTLE_TO_INT(w)    ( ((hcf_16)(w) & 0x00ff) << 8 | ((hcf_16)(w) & 0xff00) >> 8 )
#endif
#else
#ifndef CNV_LITTLE_TO_INT
#define CNV_LITTLE_TO_INT(w)    (w)
#endif
#endif

// Timing related defines
#define DIAGNOSE_TIMEOUT    3000         // Diagnose time out
#define SLEEP_MILLI          100         // Sleep period between consecutive get CFG_MB_INFO

// Various firmware type defines
#define FT_UNKNOWN            -1
#define FT_PRIMARY             0
#define FT_STATION             1
#define FT_AP                  2

// Internal error and warning codes
// Note: make sure these codes do not overlap with the exported error codes in DHF.H
#define DHF_ERR_PLUG_PRODUCTION_DATA             0x8A
#define DHF_ERR_PLUG_PRIMARY_DATA                0x8B

#define DHF_WARN_DRV_STA_NOT_COMPATIBLE          0xB0
#define DHF_WARN_DRV_PRI_NOT_COMPATIBLE          0xB1
#define DHF_WARN_SELFTEST_FAILED                 0xB2
#define DHF_WARN_NOT_COMPATIBLE                  0xB3
#define DHF_WARN_DRIVER_NOT_COMPATIBLE           0xB6

//!!!!!!!!temporarily defined here awaiting official code
//#define CFG_AP_IDENTITY 331 -- MJa: moved into DHF.H because FIRMWARE.C also needs it!!
//!!!!!!!!

/*-----------------------------------------------------------------------------
 *
 * The following LTV-records are retrieved from the NIC by function
 * get_nic_info by doconnect and are accessed through LTVP-array dui_info below.
 *
 *---------------------------------------------------------------------------*/

CFG_RANGES_STRCT         nic_pri_sup             = { LOF(CFG_RANGES_STRCT), CFG_PRI_SUP_RANGE };

#ifdef DHF_UIL
CFG_DRV_INFO_STRCT       drv_info                = { LOF(CFG_DRV_INFO_STRCT), CFG_DRV_INFO };
#endif

CFG_IDENTITY_STRCT       drv_identity            = { LOF(CFG_IDENTITY_STRCT), CFG_DRV_IDENTITY };
CFG_IDENTITY_STRCT       pri_identity            = { LOF(CFG_IDENTITY_STRCT), CFG_PRI_IDENTITY };
CFG_IDENTITY_STRCT       nic_identity            = { LOF(CFG_IDENTITY_STRCT), CFG_NIC_IDENTITY };
CFG_IDENTITY_STRCT       sta_identity            = { LOF(CFG_IDENTITY_STRCT), CFG_FW_IDENTITY };

CFG_RANGES_STRCT         drv_sup                 = { LOF(CFG_RANGES_STRCT), CFG_DRV_SUP_RANGE };
CFG_RANGES_STRCT         hsi_sup                 = { LOF(CFG_RANGES_STRCT), CFG_HSI_SUP_RANGE };
CFG_RANGES_STRCT         mfi_sup                 = { LOF(CFG_RANGES_STRCT), CFG_MFI_SUP_RANGE };
CFG_RANGES_STRCT         sta_sup                 = { LOF(CFG_RANGES_STRCT), CFG_FW_SUP_RANGE };
CFG_RANGES_STRCT         cfi_sup                 = { LOF(CFG_RANGES_STRCT), CFG_CFI_SUP_RANGE };

CFG_RANGE20_STRCT       drv_act_range_pri       = { LOF(CFG_RANGE20_STRCT), CFG_DRV_ACT_RANGES_PRI };
CFG_RANGE20_STRCT       drv_act_range_sta       = { LOF(CFG_RANGE20_STRCT), CFG_DRV_ACT_RANGES_STA };
CFG_RANGE20_STRCT       drv_act_range_hsi       = { LOF(CFG_RANGE20_STRCT), CFG_DRV_ACT_RANGES_HSI };
CFG_RANGE20_STRCT       cfi_act_range_pri       = { LOF(CFG_RANGE20_STRCT), CFG_CFI_ACT_RANGES_PRI };

CFG_MAX_LOAD_TIME_STRCT  max_load_time           = { LOF(CFG_MAX_LOAD_TIME_STRCT), CFG_MAX_LOAD_TIME };

CFG_MAC_ADDR_STRCT       mac_addr                = { LOF(CFG_MAC_ADDR_STRCT), CFG_CNF_OWN_MAC_ADDR };

CFG_DL_BUF_STRCT         dl_buf                  = { LOF(CFG_DL_BUF_STRCT), CFG_DL_BUF };

CFG_NIC_SERIAL_NUMBER_STRCT nic_serial_number   = { LOF(CFG_NIC_SERIAL_NUMBER_STRCT), CFG_NIC_SERIAL_NUMBER };

CFG_RANGE20_STRCT     mfi_act_range_sta       = { LOF(CFG_RANGE20_STRCT), CFG_MFI_ACT_RANGES_STA };
CFG_RANGE20_STRCT     cfi_act_range_sta       = { LOF(CFG_RANGE20_STRCT), CFG_CFI_ACT_RANGES_STA };

LTV_STRCT                phy_type                = { LOF(LTV_STRCT), CFG_PHY_TYPE };


#ifndef DSF_HERMESII
#ifdef DSF_ALLOC
// PDA is allocated dynamically to avoid stack problems. If defined cfg_prod_data is allocated in
// doconnect() and freed in dodisconnect(). Make sure dodisconnect() is always called.
hcf_16                  *cfg_prod_data          = NULL;
#else
hcf_16                  cfg_prod_data[PDA_SIZE] = { PDA_SIZE-1, CFG_PROD_DATA };
#endif

#endif // #ifndef DSF_HERMESII

// This struct is needed to save the sizes of the structs, because
// after a GET_INFO() the len field is changed to the real len of the
// RID by the called routine. 
typedef struct {
	LTVP 	ltvp ;
	hcf_16	len ;
} DUI_INFO_STRUCT , *DUI_INFO_STRUCT_PTR ;



// This struct is needed till it is added in the new HCF 4.13 (mmd.h)
//typedef struct CFG_RANGE_SPEC_BYTE_STRCT {
//	hcf_8	number[2];
//	hcf_8	bottom[2];
//	hcf_8	top[2];
//} CFG_RANGE_SPEC_BYTE_STRCT;



/*-----------------------------------------------------------------------------
 * Array dui_info stores NIC information (in the form of LTV-records)
 * needed for download. The information is retrieved from the NIC by function
 * get_nic_info(). A NULL record indicates the end of the array.
 *---------------------------------------------------------------------------*/

DUI_INFO_STRUCT dui_info[] = {
#ifndef DSF_HERMESII
#ifndef DSF_ALLOC
        { (LTVP)&cfg_prod_data[0],  	PDA_SIZE-1 } , 
#endif
#endif // #ifndef DSF_HERMESII
        { (LTVP)&max_load_time,		LOF(CFG_MAX_LOAD_TIME_STRCT) } , 
        { (LTVP)&nic_pri_sup,			LOF(CFG_RANGES_STRCT) } , 
#ifdef DHF_WCI
        // the driver information is stored internally in the IFB
#else
        { (LTVP)&drv_info,			LOF(CFG_DRV_INFO_STRCT) } , 
#endif
//         { (LTVP)&restraint_info,	LOF(CFG_RESTRAINT_INFO_STRCT) } , 
        { (LTVP)&drv_identity,		LOF(CFG_IDENTITY_STRCT) } , 
        { (LTVP)&pri_identity,		LOF(CFG_IDENTITY_STRCT) } , 
        { (LTVP)&nic_identity,		LOF(CFG_IDENTITY_STRCT) } , 
        { (LTVP)&sta_identity,		LOF(CFG_IDENTITY_STRCT) } , 
        { (LTVP)&drv_sup,			LOF(CFG_RANGES_STRCT) } ,
        { (LTVP)&hsi_sup,			LOF(CFG_RANGES_STRCT) } ,
        { (LTVP)&mfi_sup,			LOF(CFG_RANGES_STRCT) } ,
        { (LTVP)&cfi_sup,			LOF(CFG_RANGES_STRCT) } ,
        { (LTVP)&sta_sup,			LOF(CFG_RANGES_STRCT) } ,
        { (LTVP)&drv_act_range_pri,	LOF(CFG_RANGE20_STRCT) } , 
        { (LTVP)&drv_act_range_sta,	LOF(CFG_RANGE20_STRCT) } , 
        { (LTVP)&drv_act_range_hsi,	LOF(CFG_RANGE20_STRCT) } , 
        { (LTVP)&cfi_act_range_pri,	LOF(CFG_RANGE20_STRCT) } , 
        { (LTVP)&mfi_act_range_sta,	LOF(CFG_RANGE20_STRCT) } ,
        { (LTVP)&cfi_act_range_sta,	LOF(CFG_RANGE20_STRCT) } ,
        { (LTVP)&mac_addr,			LOF(CFG_MAC_ADDR_STRCT) } , 
        { (LTVP)&dl_buf,			LOF(CFG_DL_BUF_STRCT) } , 
        { (LTVP)&nic_serial_number,	LOF(CFG_NIC_SERIAL_NUMBER_STRCT) } , 
        { (LTVP)&phy_type,			LOF(LTV_STRCT) } , 
        { (LTVP) NULL, 				0 } 
};


/*-----------------------------------------------------------------------------
 * Global variables needed by functions download_start, download_next, and
 * download_stop
 *---------------------------------------------------------------------------*/

//Storage class static seemed to lead to problems in Win2000-platform/compiler 
// (and others???). Define DHFSTATIC empty if this problem arises, otherwise
// define it static as usual.
// #define DHFSTATIC static
#define DHFSTATIC

#ifdef DHF_WCI
DHFSTATIC IFBP      dhf_ifbp;                    // Pointer to the ifb
#else
DHFSTATIC void      *dhf_ifbp;                   // Pointer to the ifb
DHFSTATIC BOOL      dhf_confirm_switch;          // TRUE = ask user for confirmation where necessary
                                                 // FALSE = do not ask user of action
#endif
DHFSTATIC memimage  *dhf_firmware;               // Pointer to firmware image to be downloaded
DHFSTATIC memblock  *dhf_memblockp;              // Pointer to memblock currently being downloaded
DHFSTATIC hcf_16    dhf_len;                     // remaining length of current memblock
DHFSTATIC hcf_16    dhf_index;                   // index in current memblock from where to download

#ifdef DSF_VOLATILE_ONLY
DHFSTATIC BOOL      dhf_to_volatile = TRUE;      // TRUE if download to volatile NIC-RAM;
#else
DHFSTATIC BOOL      dhf_to_volatile = FALSE;     // FALSE otherwise
DHFSTATIC int       dhf_block_nr;                // number of block being downloaded
DHFSTATIC int       dhf_progress_step;           // stepsize for possible progress bars in a scale
#endif

                                                 // of 1 to 100 (roughly)
DHFSTATIC int       dhf_last_error= DHF_SUCCESS; // last error encountered by DHF
DHFSTATIC BOOL      dhf_always_download_primary; // if TRUE then primary is always 
                                                 // downloaded independent of rules in routine
                                                 // primary_download_needed

#ifdef DSF_DEBUG_MESSAGE

/*-----------------------------------------------------------------------------
 *
 * Debug function prototypes
 *
 *---------------------------------------------------------------------------*/
static void dbg_display_download_info( memimage *firmware );
static void dbg_display_nic_info( void );
static void dbg_display_hex( void *vp, int len );
static void dbg_fdisplay_hex(char *fn, char *mode, void *vp, hcf_16 len );

#endif


/*-----------------------------------------------------------------------------
 *
 * Local service function prototypes
 *
 *---------------------------------------------------------------------------*/

static hcf_16 calc_progress_step( memimage *firmware, BOOL to_volatile );
static int check_comp(CFG_RANGES_STRCT *supplier, CFG_RANGES_STRCT *actor);
static int check_comp_ap( memimage *firmware );
static int check_comp_primary( memimage *firmware );
static int check_comp_station( memimage *firmware );
static int determine_firmware_type( memimage *firmware );
static int doconnect(void *ifbp, CFG_RANGE20_STRCT *app_dui_actor, CFG_RANGE20_STRCT *app_pri_actor, BOOL confirm );
static void dodisconnect( void );
static int download_firmware_image( void );
static int download_next( void );
static int download_start( void );
static int download_stop( void );
static memblock *find_addr_in_image( memimage *firmware, hcf_32 addr );
static int get_nic_info( void );
static BOOL primary_download_needed(  memimage *firmware );

#ifndef DSF_HERMESII
static int check_pda_crc( void );
static int extend_pda_with_pri_records( memimage *firmware );
static int plug_pri_records( memimage *firmware );
static int plug_production_data( memimage *firmware );
#endif


/*-----------------------------------------------------------------------------
 *
 * External function prototypes (defined here, but used in DHFPLUG.C and/or
 * DHFPDA.C)
 *
 *---------------------------------------------------------------------------*/

#ifndef DSF_HERMESII
EXTERN_C hcf_16 calc_crc( 
    hcf_8  *cp,              // pointer to the starting point of the crc calculation.
    hcf_32 count             // number of bytes to calculate over 
                             // (hcf_16 as count should be adequate?)
); 

EXTERN_C hcf_16 *find_record_in_pda(
    hcf_16 *pdap,            // pointer to PDA
    hcf_16 code,             // code to be looked for in the PDA
    hcf_16 *len              // length of the record found:
                             // - set to required length or zero if any length is OK
                             // - set to NULL if not needed
);

EXTERN_C int copy_pda_rec(
    LTVP dest,               // destination record
    LTVP src                 // source record 
);

EXTERN_C int apply_plug_rules(
    hcf_16 *pdap,            // Pointer to production data area.
    hcf_16 **record,         // Pointer to specific record in PDA, may be null.
                             // For certain codes this record may be allocated 
                             // by this routine. In the latter case it is the 
                             // repsonsibility of te caller to deallocate it.
    hcf_16 code              // Code of production data record to be (possibly)
                             // adjusted.
);

#endif // #ifndef DSF_HERMESII

EXTERN_C int condense_error_code(
    int entry_code,          // actual error code
    int condense_code        // condensed error code
);
  
  
/*-----------------------------------------------------------------------------
 *
 * PDA-functions prototypes (see DHFPDA.C)
 *
 *---------------------------------------------------------------------------*/
#ifndef DSF_HERMESII
EXTERN_C int dhf_add_pda_record( LTVP rec );
EXTERN_C LTVP dhf_get_pda_record( hcf_16 code );
EXTERN_C int dhf_load_pda( void );                      // defined in this file
EXTERN_C int dhf_program_pda( void );
EXTERN_C int dhf_put_pda_record( hcf_16 code, LTVP rec );
#endif // #ifndef DSF_HERMESII

#if defined( DSF_DEBUG_MESSAGE )

TCHAR dbgbuf[100];


/*-----------------------------------------------------------------------------
 *-----------------------------------------------------------------------------
 *
 * Debugging routines
 *
 *---------------------------------------------------------------------------*/


/*-----------------------------------------------------------------------------
 *
 * Function:    dbg_get_comp_name
 *
 * Abstract:    Retrieves the name of a WaveLAN component
 *
 * Returns:     name of the component with id
 *
 * Description:
 *---------------------------------------------------------------------------*/
static char *dbg_get_comp_name(int id) 
{
    char* Name;

    // Convert from id to name
    switch ( id ) {
    case 1:  Name = "PC Card Type-II Extended";
             break;
    case 2:  Name = "PC Card Type-II Integrated";
             break;
    case 3:  Name = "PC Card Type-II Extended - 8 dBm";
             break;
    case 4:  Name = "PC Card Type-II Extended - Turbo";
             break;
    case 5:  Name = "Embedded card";
             break;
    case 6:  Name = "PC Card Type-II Extended - Turbo, 8 dBm";
             break;
//Reserved for    case 7:  Name = "HomeCard";
//             break;
    case 8:  Name = "USB Card";
             break;
    case 21: Name = "Primary Functions firmware";
             break;
    case 31: Name = "Station Functions firmware";
             break;
//N/A    case CFG_AP_IDENTITY:Name = "AP Functions firmware"; // TEMPORARY!
//N/A             break;
    case 41: Name = "NDIS 3 Miniport driver";
             break;
    case 42: Name = "Packet driver";
             break;
    case 43: Name = "DOS ODI driver";
             break;
//N/A    case 44: Name = "WaveLAN/IEEE 32-bit ODI driver";
//N/A             break;
    case 45: Name = "PowerBook driver";
             break;
    case 46: Name = "Windows/CE driver";
             break;
    case 47: Name = "Linux driver (light)";
             break;
    case 48: Name = "NDIS 5 Miniport driver";
             break;
    case 49: Name = "Linux driver (library)";
             break;
    case 51: Name = "NDIS 5 Miniport USB driver";
             break;
    case 52: Name = "NDIS 4 Miniport driver";
             break;
    case 61: Name = "WaveMANAGER/CLIENT";
             break;
    case 62: Name = "Wireless Station Update (Windows)";
             break;
    case 63: Name = "Wireless Station Update (DOS)";
             break;
    case 64: Name = "WaveMANAGER/EC";
             break;
    case 65: Name = "WaveMANAGER/AP";
             break;
    case 66: Name = "RG Setup Utility";
             break;
    case 67: Name = "Comms Quality Indicator";
             break;
    case 68: Name = "WaveMANAGER/PRO";
             break;
    case 81: Name = "Access Point Image";
             break;
    case 82: Name = "Point-to-Point Image";
             break;
    case 83: Name = "Ethernet Converter Image";
             break;
//Reserved for    case 84: Name = "StarPOINT Lite Image";
//             break;
//Reserved for    case 85: Name = "StarPOINT PRO Image";
//             break;
    case 86: Name = "RG-1000 Image";
             break;
//87-90: reserved for future images
    case 91: Name = "Ethernet Converter";
             break;
    case 92: Name = "Ethernet Converter (Serial)";
             break;
    case 93: Name = "WavePOINT-II";
             break;
//Reserved for    case 94: Name = "StarPOINT";
//             break;
    case 95: Name = "RG-1000";
             break;
    default: // Check on ranges:
/*
             if (value >= 3  &&  value <= 20) {
                Name = "Network Interface Card";
             } else if (value >= 22  &&  value <= 30) {
                Name = "Primary firmware";
             } else if (value >= 32  &&  value <= 40) {
                Name = "Secondary firmware";
             } else if (value >= 45  &&  value <= 60) {
                Name = "Driver";
             } else if (value >= 63  &&  value <= 80) {
                Name = "Utility";
             } else {
                Name = "-";
             }
*/
             Name = "-";
             break;
    }
    return Name;
}   /* dbg_get_comp_name */


/*-----------------------------------------------------------------------------
 *
 * Function:    dbg_display_download_info
 *
 * Abstract:    Displays download related information
 *
 * Returns:     void
 *
 * Description:
 *---------------------------------------------------------------------------*/
static void dbg_display_download_info( memimage *firmware ) 
{

    CFG_IDENTITY_STRCT  *identityp;      // Pointer to the identity info records

    if ( firmware != NULL ) {

        identityp = firmware->identity;

        _stprintf(dbgbuf, TEXT("The WaveLAN/IEEE card in your system will be updated to :")) ;
        dsf_debug_message( dbgbuf, DHF_INFO, FALSE );
        
        _stprintf( dbgbuf, TEXT("%s, Variant %d, Version %d.%02d"),
                dbg_get_comp_name( identityp->comp_id ), identityp->variant,
                identityp->version_major, identityp->version_minor );
        dsf_debug_message( dbgbuf, DHF_INFO, TRUE );
    } else {
        _stprintf( dbgbuf, TEXT("No firmware image to download.")) ;
        dsf_debug_message( dbgbuf, DHF_INFO, TRUE );
    }

    return;
}   /* dbg_display_download_info */


/*-----------------------------------------------------------------------------
 *
 * Function:    dbg_display_nic_info
 *
 * Abstract:    Displays some key NIC information items
 *
 * Returns:     void
 *
 * Description:
 *---------------------------------------------------------------------------*/
static void dbg_display_nic_info( void ) 
{
    _stprintf(dbgbuf, TEXT("Version information of the Firmware currently in the NIC:"));
    dsf_debug_message( dbgbuf, DHF_INFO, FALSE );

    _stprintf( dbgbuf, TEXT("%s, Variant %d, Version %d.%02d "),
            dbg_get_comp_name( drv_identity.comp_id ), 
            					drv_identity.variant,
            					drv_identity.version_major, 
            					drv_identity.version_minor) ;
    dsf_debug_message( dbgbuf, DHF_INFO, FALSE );

    _stprintf( dbgbuf, TEXT("%s, Variant %d, Version %d.%02d "),
            dbg_get_comp_name( nic_identity.comp_id ), 
            					CNV_LITTLE_TO_INT(nic_identity.variant),
            					CNV_LITTLE_TO_INT(nic_identity.version_major), 
            					CNV_LITTLE_TO_INT(nic_identity.version_minor) );
    dsf_debug_message( dbgbuf, DHF_INFO, FALSE );

    _stprintf( dbgbuf, TEXT("Serial number: %.12s"), nic_serial_number.serial_number );
    dsf_debug_message( dbgbuf, DHF_INFO, FALSE );

    _stprintf( dbgbuf, TEXT("%s, Variant %d, Version %d.%02d "),
            dbg_get_comp_name( pri_identity.comp_id ), 
            					CNV_LITTLE_TO_INT(pri_identity.variant),
            					CNV_LITTLE_TO_INT(pri_identity.version_major), 
            					CNV_LITTLE_TO_INT(pri_identity.version_minor) );
    dsf_debug_message( dbgbuf, DHF_INFO, FALSE );

    _stprintf( dbgbuf, TEXT("%s, Variant %d, Version %d.%02d "),
            dbg_get_comp_name( sta_identity.comp_id ), 
            					CNV_LITTLE_TO_INT(sta_identity.variant),
            					CNV_LITTLE_TO_INT(sta_identity.version_major), 
            					CNV_LITTLE_TO_INT(sta_identity.version_minor) );
    dsf_debug_message( dbgbuf, DHF_INFO, FALSE );

    return;
}   /* dbg_display_nic_info */


/*-----------------------------------------------------------------------------
 *
 * Function:    dbg_display_hex
 *
 * Abstract:    Display area pointed to by vp in hex format over a length of
 *              len hcf_16
 *
 * Returns:     void
 *
 * Description:
 *
 *   void *vp,   Points to area to be displayed
 *   int len     Number of hcf_16 to display
 *---------------------------------------------------------------------------*/
static void dbg_display_hex(void *vp, int len) 
{
    int j = 0;
    hcf_16* p = (hcf_16*)vp;
    char    buf[10];

    strcpy( dbgbuf, "" );
    while ( len-- ) {   
        _stprintf( buf, TEXT(" %04X "), *p++);
        strcat( dbgbuf, buf );
        if ( ++j % 16 == 0 ) {
            dsf_debug_message( dbgbuf, DHF_INFO, TRUE );
            strcpy( dbgbuf, "" );
        }
    }

    return;
}   /* dbg_display_hex */


/*-----------------------------------------------------------------------------
 *
 * Function:    dbg_fdisplay_hex
 *
 * Abstract:    Write an area of hcf_16 to file in hex format
 *
 * Returns:     void
 *
 * Description:
 *
 *   char *fn,      file name
 *   char *mode,    open mode for file
 *   void *vp,      Points to area to be to be written to file
 *   hcf_16 len     Number of hcf_16 to display
 *---------------------------------------------------------------------------*/
static void dbg_fdisplay_hex(char *fn, char *mode, void *vp, hcf_16 len) 
{
    int j = 0;
    hcf_8 *p = (hcf_8*)vp;
    FILE *fp;

    fp = fopen(fn, mode);
    fprintf(fp, "\nnew block:\n\n" );
    while ( len-- ) {
        fprintf(fp, " 0x%02X ", *p++ );
        if ( ++j % 16 == 0 ) fprintf(fp, "\n " );
    }
    fprintf(fp, "\n" );
    fclose(fp);

    _stprintf( dbgbuf, TEXT("\ndump of (partial) image in file: %s"), fn );
    dsf_debug_message( dbgbuf, DHF_INFO, TRUE );

    return;
}   /* dbg_fdisplay_hex */

#endif	// DSF_DEBUG_MESSAGE



/*-----------------------------------------------------------------------------
 *-----------------------------------------------------------------------------
 *
 * Service routines
 *
 *---------------------------------------------------------------------------*/


/*-----------------------------------------------------------------------------
 * Function:    calc_crc
 *
 * Abstract:    Calculate the crc of a given number of bytes.
 *
 * Returns:
 *   hcf_16  - the CRC
 *
 * Description:
 *   This function calculates the CRC over a number of bytes. cp indicates
 *   the starting point of the memory the CRC of which has to be calculated.
 *   count indicates over how many bytes the calculation must be calculated.
 *
 *   hcf_8 *cp,      pointer to the starting point of the crc calculation.
 *   hcf_32 count    number of bytes to calculate over 
 *---------------------------------------------------------------------------*/
hcf_16 calc_crc(hcf_8 *cp, hcf_32 count) 
{ 
    hcf_16  wCarry = 0;     /* Carry-over for the CRC calculation */
    hcf_16  crctmp;         /* temporary storage for the CRC */
    hcf_16  crcvalue = 0;   /* The CRC to return */
    hcf_8   iBit;           /* BYTE mask */

	DSFASSERT( cp != NULL, TEXT("CP == NULL"), 0) ;
	DSFASSERT( count > 0, TEXT("Count > 0"), count ) ;

	// Loop through all bytes of the area over which to do calculation
    while ( count-- ) {         
        crctmp = ( crcvalue ^ *cp++ ) & 0x00ff;
        crcvalue >>= 8;
 
        /* XOR with polynomial for every '1' in the byte */
        for ( iBit = 0; iBit < 8; iBit++ )  {
            /* Now shift all bits right */
            wCarry = crctmp & 0x0001;               /* Remember if there is a carry-over */
            crctmp >>= 1;
            if ( wCarry == 1 ) crctmp ^= 0xa001;    /* XOR with polynomial */
        }
        crcvalue ^= crctmp;
    }
    return crcvalue;
}   /* calc_crc */


#ifndef DSF_VOLATILE_ONLY
/*-----------------------------------------------------------------------------
 * Function:    calc_progress_step
 *
 * Abstract:    Calculate the progress step per memory block to be downloaded
 *              in the memimage.
 *
 * Returns:
 *   hcf_16 - the (rough) stepsize in a scale of 1-1000 for each block to be 
 *            downloaded
 *
 * Description:
 *   When downloading to volatile the  number is equal to the number of blocks
 *   in the memblock* chain in the firmware image.
 *   When downloading to non-volatile memory the number is equal to the sum of
 *   the blocks needed to download the blocks in the memblock* chain in the 
 *   firmware image, possibly taking into account the size of an intermediate
 *   buffer used.
 *   The step size is calculated from the number of blocks.
 *---------------------------------------------------------------------------*/
static hcf_16 calc_progress_step( memimage *firmware, BOOL to_volatile ) 
{
    hcf_16 rc = 0;
    hcf_16 tmplen;
    hcf_16 tmpint = 0;
    int stepsize;
    memblock *mbp;
    
#ifdef DHF_WCI
    stepsize = dl_buf.buf_len; 
#else
    stepsize = min( UIL_DATBUF_SIZE, dl_buf.buf_len ); 
#endif

	DSFASSERT( stepsize > 0, TEXT("stepsize > 0"), stepsize) ;

	if ( firmware != NULL ) {
        for(mbp = firmware->codep; (mbp != NULL) && (mbp->addr != 0) ; mbp++ ) {
            tmplen = mbp->size;
            if ( to_volatile == TRUE ) {
                tmpint++;
            } else {
                tmpint += tmplen / stepsize; 
                if ( ( tmplen % stepsize ) != 0 ) tmpint++;
            }
        }
        
        if ( tmpint > 0 ) {
            rc = 1000 / tmpint;
        }
        
        // We want to be sure we'll always get some progress
        rc = max( rc, 1);
    }
    
    return rc;
}   // calc_progress_step
#endif // !DSF_VOLATILE_ONLY


/*FF***************************************************************************************************************
FUNCT: mmd_check_comp                                                                                2000/02/04 14:15
USERS: hcf_initialize                                   
CALLS: hcf_get_info                                     CNV_LITTLE_TO_INT                                
GLOBL: HCF_SUCCESS                                      p->variant.bottom                                
       p->variant.number                                p->variant.top                                   
PARAM: ifbp                                             p                                                
       p->len                                           type                                             
LOCAL: cnt                                              i                                                
       x                                                x.len                                            
       x.typ                                            x.variant.bottom                                 
       x.variant.number                                 x.variant.top                                    
******************************************************************************************************************/
/******************************************************************************************************************

.MODULE			mmd_check_comp
.DESCRIPTION	Checks compatibility between an actor and a supplier

.ARGUMENTS
  CFG_RANGE_SPEC_STRCT* mmd_check_comp( CFG_RANGES_STRCT *actp, CFG_RANGES_STRCT *supp )
	
.RETURNS
  NULL		incompatible
  <>NULL	pointer to matching CFG_RANGES_STRCT substructure in actor-structure matching the supplier

.NARRATIVE

 Parameters:
  actp	address of the actor specification
  supp	address of the supplier specification

 Description: mmd_check_comp is a support routine to check the compatibility between an actor and a supplier.
 mmd_check_comp is independent of the endianess of the actp and supp structures. This is achieved by checking
 the "bottom" or "role" fields of these structures. Since these fields are restricted to a limited range,
 comparing the contents to a value with a known endian-ess gives a clue to their actual endianess.

.DIAGRAM
 1: The role-field of the actor structure has a known non-zero, not "byte symmetric"  value (namely 
 	COMP_ROLE_ACT or 0x0001), so when the contents of this field matches COMP_ROLE_ACT in Little Endian format,
	the actor structure is Little Endian. Note that the selected macro CNV_LITTLE_TO_INT is a misnomer 
	(especially on a Big Endian platform) but has the intended effect (of doing nothing on a LE platform
	and swapping bytes on a BE platform). By setting i_act appropriately the byte with the "lower" or "higher" 
	address in the number, bottom and top sub-fileds of the CFG_RANGE_SPEC_STRCT field(s) in the actor 
	structure can be selected. Since each of these values is in the range of 1 through 99 inclusive, a byte
	check suffices.
 2:	Since the role-field of the supplier structure is 0x0000, the test as used for the actor does not work.
 	Analoguos to the reasoning why we need only check (either the lower or higher addressed) single byte in
	number, bottom and top, we can conclude that the lower addressed byte of the bottom in a LE can not be 0x00
	and in a BE structure must be 0x00.
 6:	Each of the actor variants is checked against the (single) supplier bottom-top range till either an
 	acceptable match is found or all actor variants are tried. As explained above, due to the limited ranges
	of these values, checking a byte is acceptable and suitable.
 8: depending on whether a match was found or not, the NULL pointer or a pointer to the matching 
 	Number/Top/Bottom record of the Actor structure is returned
 
NOTE: The Endian Detection Algorithm places limitations on future extensions of the fields, i.e. they should 
	stay within the currently defined boundaries of 1 through 99 (although 1 through 255) would work as well
	and there should never be a valid bottom 0 variant of a supplier (this would break the i_sup determination)
NOTE: strategy as used in #2 (i.e. supplier index) can NOT be used for #1 (i.e actor index), because there are 
	actors which contain - in addition to menaingfull CFG_RANGE_SPEC_STRCT structures - zero filled 
	CFG_RANGE_SPEC_STRCT as padding.	
NOTE: the L and T can not be used for the Endian Detection Algorithm, because L and T are always
	Native Endian.
NOTE: the number field of the supllier structure can not be used for the Endian Detection Algorithm, because 
	a variant 0x0000 has been used as Controlled Deployment indication in the past.
 
.ENDOC				END DOCUMENTATION
-----------------------------------------------------------------------------------------------------------*/
CFG_RANGE_SPEC_STRCT* mmd_check_comp( CFG_RANGES_STRCT *actp, CFG_RANGES_STRCT *supp )
{
	CFG_RANGE_SPEC_BYTE_STRCT  *actq = (CFG_RANGE_SPEC_BYTE_STRCT*)actp->variant;
	CFG_RANGE_SPEC_BYTE_STRCT  *supq = (CFG_RANGE_SPEC_BYTE_STRCT*)supp->variant;
	hcf_16	i;
	int		i_act = actp->role == CNV_LITTLE_TO_INT(COMP_ROLE_ACT)  ? 0 : 1;	//actor index				/* 1 */
	int		i_sup = supq->bottom[0] == 0 ? 1 : 0;								//supplier index			/* 2 */

	for ( i = actp->len ; i > 3; actq++, i -= 3 ) {														/* 6 */
            DSFDBG(_stprintf(dbgbuf, TEXT("DHF:MMD_CHECK_COMP: act: var %d top %d bot %d  sup: var %d top %d bot %d\n"),
	                					 actq->number[i_act], actq->top[i_act], actq->bottom[i_act],   
            					    	 supq->number[i_sup],supq->top[i_sup], supq->bottom[i_sup] )) ;
            DSFDBG(dsf_debug_message( dbgbuf, DHF_INFO, FALSE );)
		if ( actq->number[i_act] == supq->number[i_sup] &&
			 actq->bottom[i_act] <= supq->top[i_sup] &&
			 actq->top[i_act]    >= supq->bottom[i_sup]    
		   ) break;
	}
	return i > 3 ? (CFG_RANGE_SPEC_STRCT*)actq : NULL;
}/* mmd_check_comp */


int mmd_comp_versions( CFG_RANGE_SPEC_STRCT *actp, CFG_RANGES_STRCT *supp )
{
	CFG_RANGE_SPEC_BYTE_STRCT  *actq = (CFG_RANGE_SPEC_BYTE_STRCT*) actp;
	CFG_RANGE_SPEC_BYTE_STRCT  *supq = (CFG_RANGE_SPEC_BYTE_STRCT*) supp->variant;
	int	rc = -2;
	int		i_act = actq->bottom[0] == 0 ? 1 : 0;								//actor index				/* 1 */
	int		i_sup = supq->bottom[0] == 0 ? 1 : 0;								//supplier index			/* 2 */

    if ( supq->top[i_sup] == actq->top[i_act] ) {
        rc = 0;
    } else if ( supq->top[i_sup] > actq->top[i_act] ) {
        rc = 1;
    } else if ( supq->top[i_sup] < actq->top[i_act] ) {
        rc = -1;
    }
    DSFDBG(_stprintf(dbgbuf, TEXT("DHF:MMD_COMP_VERSION rc = %d: act: var %d top %d bot %d  sup: var %d top %d bot %d\n"), rc, 
	                					 actq->number[i_act], actq->top[i_act], actq->bottom[i_act],   
            					    	 supq->number[i_sup],supq->top[i_sup], supq->bottom[i_sup] )) ;
    DSFDBG(dsf_debug_message( dbgbuf, DHF_INFO, FALSE );)
    return(rc) ;
}/* mmd_comp_versions */


/*-----------------------------------------------------------------------------
 *
 * Function:    check_comp
 *
 * Abstract:    Checks compatibility of an actor with a supplier
 *
 * Returns:
 *    0  same version
 *    1  compatible but supplier is newer than actor
 *   -1  compatible but supplier is older than actor
 *   -2  not compatible:
 *       + the roles where different;
 *       + the interface ID where not the same;
 *       + the actor did not have a variant mentioned in the supplier range
 *
 * Description:
 *   Checks compatibility of an actor with a supplier. To achieve this, each of
 *   the actor variants is checked against the (single) supplier bottom-top range
 *   till either an acceptable match is found or all actor variants have been
 *   tried. The compatibilities are defined in ....;?
 *   If overlap exists for a variant of the actor, we check whether the actor is
 *   identical, or older, or younger by comparing the value of the tops of the
 *   supplier and the actor. The bottom value is not considered in this check.
 *---------------------------------------------------------------------------*/
static int check_comp(CFG_RANGES_STRCT *supplier, CFG_RANGES_STRCT *actor ) 
{
    CFG_RANGE_SPEC_STRCT *pAct ;
    int rc = -2;
    

    if ( ( supplier != NULL ) && ( actor != NULL ) ) {
    	pAct = mmd_check_comp(actor, supplier) ;
    	if(pAct != NULL) {
    		rc = mmd_comp_versions(pAct, supplier) ;
    	}
	}
	DSFASSERT(rc != -2, TEXT("rc == -2"), rc);
    return(rc) ;
#if 0    	
        for ( cnt_act = 0, j = 4; j < actor->len; cnt_act++, j += 3 ) {
//			OUTPUTDEBUGMSG((TEXT("DHF: cnt_act = %d actor len = %d\n"), cnt_act, actor->len )) ;
//			OUTPUTDEBUGMSG((TEXT("DHF: act: var %d top %d bot %d  sup: var %d top %d bot %d\n"),
//	                					 actor->variant[cnt_act].number,   
//           					    	 	 actor->variant[cnt_act].top,      
//			    			             actor->variant[cnt_act].bottom,   
//            					    	 supplier->variant[0].number,
//            					    	 supplier->variant[0].top,
//            					    	 supplier->variant[0].bottom )) ;
            if ( actor->variant[cnt_act].number == CNV_LITTLE_TO_INT(supplier->variant[0].number) &&
                 actor->variant[cnt_act].bottom <= CNV_LITTLE_TO_INT(supplier->variant[0].top)    &&
                 actor->variant[cnt_act].top    >= CNV_LITTLE_TO_INT(supplier->variant[0].bottom)
               ) break;
        }
    
        if ( j < actor->len ) {
//JMES 10-7-2000: removed check on bottom level of components
            if ( CNV_LITTLE_TO_INT(supplier->variant[0].top) == actor->variant[cnt_act].top ) {
                rc = 0;
            } else if ( CNV_LITTLE_TO_INT(supplier->variant[0].top) > actor->variant[cnt_act].top ) {
                rc = 1;
            } else if ( CNV_LITTLE_TO_INT(supplier->variant[0].top) < actor->variant[cnt_act].top ) {
                rc = -1;
            }
        }
    }
	DSFASSERT(rc != -2, TEXT("rc == -2"), rc);
    return(rc) ;
#endif // if 0    
}   /* check_comp */


/*-----------------------------------------------------------------------------
 *
 * Function:    check_comp_ap( memimage *firmware )
 *
 * Abstract:    Checks compatibility access point firmware
 *
 * Returns:
 *   DHF_SUCCESS                       - firmware OK 
 *   DHF_ERR_CRD_STA_NOT_COMPATIBLE    - compatibility problem
 *
 * Description:
 *   This function uses compatibility and identity information that has been
 *   retrieved from the card which is currently inserted to check whether the
 *   station firmware image to be downloaded is compatible.
 *---------------------------------------------------------------------------*/
static int check_comp_ap(memimage *firmware)
{
    CFG_RANGE20_STRCT  *p;
    int                 rc = -2;

	if(determine_firmware_type( firmware ) == FT_AP) {
        do {    //begin of pseudo goto-less contraption
            p = firmware->compat;
            while ( p->len ) {
                if ( ( p->typ == CFG_MFI_ACT_RANGES_STA ) && ( ( rc = check_comp( &mfi_sup, (CFG_RANGES_STRCT*)p ) ) ) != -2 ) {
                     break;
				}
                p++;
            }
            if ( p->len == 0 ) {
                rc = -2;
                break;
            }

            p = firmware->compat;
            while ( p->len ) {
                if ( ( p->typ == CFG_CFI_ACT_RANGES_STA ) && ( rc = check_comp( &cfi_sup, (CFG_RANGES_STRCT*) p ) ) != -2 ) {
                     	break;
				}
                p++;
            }
            if ( p->len  == 0 ) {
                rc = -2;
                break;
            }
        } while ( 0 );  //end of pseudo goto-less contraption
	} else {
		DSFASSERT(DO_DSFASSERT, TEXT("determine_firmware_type( firmware ) != FT_AP"), determine_firmware_type( firmware )) ;
	}

    switch( rc ) {
    case -2:
        rc = DHF_ERR_CRD_STA_NOT_COMPATIBLE;
        break;
    case -1:
    case  0:
    case  1:
        // Station firmware is same or newer -> OK
        rc = DHF_SUCCESS;
        break;
    default:
		DSFASSERT(DO_DSFASSERT, TEXT("Unknown compatiblity check result in access point firmware"), rc);
        DSFDBG(dsf_debug_message( TEXT("\nUnknown compatiblity check result in access point firmware."), DHF_ERROR, FALSE );)
        rc = DHF_ERR_CRD_STA_NOT_COMPATIBLE;        break;
    }
    
    return rc;
}   /* check_comp_ap */


/*-----------------------------------------------------------------------------
 *

 * Function:    check_comp_primary( memimage *firmware )
 *
 * Abstract:    Checks compatibility primary firmware image
 *
 * Returns:
 *   DHF_SUCCESS                       - firmware OK and newer than firmware in NIC
 *   DHF_ERR_CRD_PRI_NOT_COMPATIBLE    - card not compatible with primary firmware
 *   DHF_WARN_DRV_PRI_NOT_COMPATIBLE   - driver not compatible with primary firmware
 *   DHF_NOT_NEWER                     - firmware OK but not newer than firmware in NIC
 *
 * Description:
 *   This function uses compatibility and identity information that has been
 *   retrieved from the card which is currently inserted to check whether the
 *   primary firmware image to be downloaded is compatible.
 *   Primary firmware must be updated silently, i.e. without consulting the user. 
 *   It is updated only if it is newer than the primary firmware currently present
 *   in the card.
 *---------------------------------------------------------------------------*/
static int check_comp_primary(memimage *firmware) 
{

    CFG_RANGE20_STRCT *p;
    int                  rc = -2;
    int                  tmprc = 0;

	if ( determine_firmware_type( firmware ) == FT_PRIMARY ) {
        do {    //begin of pseudo goto-less contraption

            // check CFI-ranges
            tmprc = DHF_ERR_CRD_PRI_NOT_COMPATIBLE;
            p = firmware->compat;
            while ( p->len ) {
                if ( ( p->typ  == CFG_CFI_ACT_RANGES_STA ) && ( ( rc = check_comp( &cfi_sup, (CFG_RANGES_STRCT*)p ) ) ) != -2 ) break;
                p++;
            }
            if ( p->len == 0 ) { 
                rc = -2; 
                break;
            }

            // check whether driver will be compatible with the new primary 
            // firmware after download.
            tmprc = DHF_WARN_DRV_PRI_NOT_COMPATIBLE;
            p = firmware->compat;
            while ( p->len ) {
                if ( ( p->typ  == CFG_PRI_SUP_RANGE ) && ( ( rc = check_comp( (CFG_RANGES_STRCT*)&drv_act_range_pri, (CFG_RANGES_STRCT*)p ) ) ) != -2 ) {
                    // As the actor now is the "old" part we have to invert the result
                    rc = -rc;
                    break;
                }
                p++;
            }
            if ( p->len == 0 ) { 
            	rc = -2; 
            }
        } while ( 0 );  //end of pseudo goto-less contraption
	} else {
		DSFASSERT(DO_DSFASSERT, TEXT("determine_firmware_type( firmware ) != FT_PRIMARY"), determine_firmware_type( firmware )) ;
    }

    switch( rc ) {
    case -2:
		DSFASSERT(DO_DSFASSERT, TEXT("RC == -2"), rc) ;
        // We will download the primary firmware even if the driver will not be 
        // compatible with the primary firmware AFTER download. We will display
        // a message to the user but download anyway.
        if ( tmprc == DHF_WARN_DRV_PRI_NOT_COMPATIBLE ) {
#if !defined( DHF_WCI ) && defined( DSF_CONFIRM )
            if ( dhf_confirm_switch == TRUE ) {
                dsf_confirm( DHF_CONFIRM_DRV_PRI_NOT_COMPATIBLE, 
                                     DHF_WARNING, TRUE );
            }
#endif
            rc = DHF_SUCCESS;
        } else {
            rc = tmprc;
        }
        break;
    case -1:
        rc = DHF_SUCCESS;
        break;
    case 0:
    case 1:
//!!!TODO: test        rc = DHF_NOT_NEWER;
        rc = DHF_SUCCESS;
        break;
    default:
		DSFASSERT(DO_DSFASSERT, TEXT("Unknown compatiblity check result in primary firmware."), rc) ;
        DSFDBG(dsf_debug_message( TEXT("\nUnknown compatiblity check result in primary firmware."), DHF_ERROR, FALSE );)
        rc = DHF_ERR_CRD_PRI_NOT_COMPATIBLE;
        break;
    }

    return rc;
}   /* check_comp_primary */


/*---------------------------------------------------------------------------- *
 * Function:    check_comp_station( memimage *firmware )
 *
 * Abstract:    Checks compatibility station firmware
 *
 * Returns:
 *   DHF_SUCCESS                       - firmware OK 
 *   DHF_ERR_CRD_STA_NOT_COMPATIBLE    - card not compatible with station firmware
 *   DHF_WARN_DRV_STA_NOT_COMPATIBLE   - driver not compatible with station firmware
 *   DHF_USER_ABORT                    - user decided not to update because firmware is 
 *                                       identical to or older than firmware in NIC
 *
 * Description:
 *   This function uses compatibility and identity information that has been
 *   retrieved from the card which is currently inserted to check whether the
 *   station firmware image to be downloaded is compatible.
 *---------------------------------------------------------------------------*/
static int check_comp_station(memimage *firmware)
{
    CFG_RANGE20_STRCT  *p;
    int                   rc = -2;
    int                   tmprc = 0;

    if ( determine_firmware_type( firmware ) == FT_STATION ) {
        do {    //begin of pseudo goto-less contraption

            /*-----------------------------------------------------------------
             Note that the order in which we check the components involved is
             significant: we only check the first 2 to avoid incompatibility.
             The sta_sup, however, is checked both for compatibility and to see
             whether the firmware in the image to be downloaded happens to be 
             older or the same as the firmware in the card.  
             
             If everything's OK we ask the user for confirmation if 
             dhf_confirm_switch == TRUE
            -----------------------------------------------------------------*/
            
            // check the MFI ranges
            tmprc = DHF_ERR_CRD_STA_NOT_COMPATIBLE;
            p = firmware->compat;
            while( p->len ) {
                if ( ( p->typ  == CFG_MFI_ACT_RANGES_STA ) && ((rc = check_comp( &mfi_sup, (CFG_RANGES_STRCT*)p ) ) ) != -2 ) break;
                p++;
            }
            if ( p->len == 0 ) {
				DSFASSERT(DO_DSFASSERT, TEXT("CFG_MFI_ACT_RANGES_STA NOT COMPATIBLE"), 0);
                rc = -2;
                break;
            }

            // check the CFI ranges
            p = firmware->compat;
            while ( p->len  ) {
                if ( ( p->typ  == CFG_CFI_ACT_RANGES_STA ) && ( rc = check_comp( &cfi_sup, (CFG_RANGES_STRCT*) p ) ) != -2 ) break;
                p++;
            }
            if ( p->len == 0 ) {
				DSFASSERT(DO_DSFASSERT, TEXT("CFG_CFI_ACT_RANGES_STA NOT COMPATIBLE"), 0);
                rc = -2;
                break;
            }

/* HWi removed because not know why this is tested Supplier agains supplier
            // check the Station supplier ranges
            p = firmware->compat;
            while ( p->len  ) {
                if ( ( p->typ  == CFG_FW_SUP_RANGE ) && ( ( rc = check_comp( &sta_sup, (CFG_RANGES_STRCT*)p ) ) ) != -2 ) break;
                p++;
            }
            if ( p->len == 0 ) {
				DSFASSERT(DO_DSFASSERT, TEXT("CFG_FW_SUP_RANGE NOT COMPATIBLE"), 0);
                rc = -2;
                break;
            }
*/            

#ifndef DSF_HERMESII
            // check whether driver will be compatible with the new station 
            // firmware after download.
            tmprc = DHF_WARN_DRV_STA_NOT_COMPATIBLE;
            p = firmware->compat;
            while ( p->len ) {
                if ( ( p->typ  == CFG_FW_SUP_RANGE ) && ( ( rc = check_comp( (CFG_RANGES_STRCT*)p, (CFG_RANGES_STRCT*)&drv_act_range_sta ) ) ) != -2 ) {
                    // As the actor now is the "old" part we have to invert the result
                    rc = -rc;
                    break;
                }
                p++;
            }
            if ( p->len == 0 ) { 
				DSFASSERT(DO_DSFASSERT, TEXT("NEW CFG_FW_SUP_RANGE NOT COMPATIBLE"), 0);
            	rc = -2; 
            }
#endif // #ifndef DSF_HERMESII            
        } while ( 0 );  //end of pseudo goto-less contraption
	} else {
		DSFASSERT(DO_DSFASSERT, TEXT("determine_firmware_type( firmware ) != FT_STATION"), determine_firmware_type( firmware )) ;
    }
    
    DSFASSERT(rc != -2, TEXT("rc == -2"), rc ) ;
    switch( rc ) {
    case -2:
		DSFASSERT(DO_DSFASSERT, TEXT("RC == -2"), rc) ;
        if ( tmprc == DHF_WARN_DRV_STA_NOT_COMPATIBLE ) {
#if !defined( DHF_WCI ) && defined( DSF_CONFIRM )
            // The driver will not be compatible with station firmware after 
            // download. Ask for confirmation.
            if ( ( dhf_confirm_switch == TRUE )       
                 && ( dsf_confirm( DHF_CONFIRM_DRV_STA_NOT_COMPATIBLE ) == FALSE ) ) {
                rc = DHF_USER_ABORT;
            } else {
                rc = DHF_SUCCESS;
            }
#else
//jmes: should we return success here ???
            rc = DHF_SUCCESS;
#endif
        } else {
            rc = tmprc;
        }
        break;
    case -1:
#if !defined( DHF_WCI ) && defined( DSF_CONFIRM )
        // Firmware is newer -> OK
        if ( ( dhf_confirm_switch == TRUE )       
             && ( dsf_confirm( DHF_CONFIRM_UPDATE_CARD ) == FALSE ) ) {
            rc = DHF_USER_ABORT;
        } else {
            rc = DHF_SUCCESS;
        }
#else
//jmes: should we return success here ???
        rc = DHF_SUCCESS;
#endif
        break;
    case  0:
#if !defined( DHF_WCI ) && defined( DSF_CONFIRM )
        if ( ( dhf_confirm_switch == TRUE )
             && ( dsf_confirm(DHF_CONFIRM_SAME_VERSION) == FALSE ) ) {
            rc = DHF_USER_ABORT;
        } else {
            rc = DHF_SUCCESS;
        }
#else
        rc = DHF_SUCCESS;
#endif
        break;
    case 1:
#if !defined( DHF_WCI ) && defined( DSF_CONFIRM )
        if ( ( dhf_confirm_switch == TRUE )
             && ( dsf_confirm(DHF_CONFIRM_OLDER_VERSION) == FALSE ) ) {
            rc = DHF_USER_ABORT;
        } else {
            rc = DHF_SUCCESS;
        }
#else
//jmes: should we return success here ???
        rc = DHF_SUCCESS;
#endif
        break;
    default:
		DSFASSERT( DO_DSFASSERT, TEXT("Unknown compatiblity check result in station firmware."), rc) ;
        DSFDBG(dsf_debug_message( TEXT("\nUnknown compatiblity check result in station firmware."), DHF_ERROR, FALSE );)
        rc = DHF_ERR_CRD_STA_NOT_COMPATIBLE;
        break;
    }
    
    
    DSFASSERT(rc == DHF_SUCCESS, TEXT("rc != DHF_SUCCESS"), rc);
    return rc;
}   /* check_comp_station */


/*-----------------------------------------------------------------------------
 *
 * Function:    check_driver_card( void )
 *
 * Abstract:    check if driver is compatible, card is present, and DSU is
 *              compatible.
 *
 * Returns:
 *   DHF_SUCCESS                       - succesfull validation
 *   DHF_ERR_DRV_PRI_NOT_COMPATIBLE    - various compatibility errors
 *   DHF_ERR_APP_DUI_NOT_COMPATIBLE         
 *   DHF_ERR_APP_PRI_NOT_COMPATIBLE         
 *   DHF_ERR_PRI_CFI_NOT_COMPATIBLE
 *   DHF_ERR_STA_CFI_NOT_COMPATIBLE
 *   DHF_ERR_STA_MFI_NOT_COMPATIBLE
 *   DHF_ERR_STA_SUP_NOT_COMPATIBLE
 *   DHF_ERR_WEB_RESTRAINT   
 *   DHF_ERR_NO_NIC                    - no NIC present in slot
 *
 * Description:
 *   This function checks all compatibility and identity information from the
 *   card which is currently inserted. What information is checked partially 
 *   depends on whether the WCI or the UIL is used. 
 *   The information checked includes:
 *   - the DUI interface (only when using the UIL)
 *     Because the driver is only available when the card is inserted, the
 *     compatibility of the application with the driver is tested first.
 *   - card presence
 *   - CARD_STAT_INCOMP_PRI in the driver information: read directly in the IFB
 *     when using the WCI, read from a CFG_DRV_INFO_STRCT retrieved from the
 *     driver when using the UIL
 *   - whether the driver is setup for WEB whereas the card does not support 
 *     this.
 *    CFG_RANGE20_STRCT *app_dui_actor,    actor range(s) of application for DUI. 
 *                                            If NULL this compatibility is not checked 
 *    CFG_RANGE20_STRCT *app_pri_actor     actor range(s) of application for primary 
 *                                            functions.  If NULL this compatibility is 
 *                                            not checked
 *---------------------------------------------------------------------------*/
static int check_driver_card(CFG_RANGE20_STRCT *app_dui_actor, CFG_RANGE20_STRCT *app_pri_actor) 
{
    int  rc = DHF_ERR_NOT_COMPATIBLE;
    
    do {    //begin of pseudo goto-less contraption

#ifdef DHF_WCI

        // Check whether a card is present
        rc = DHF_ERR_NO_NIC;
/* CARD_STAT_PRESENT not used in HCF         
        if ( (dhf_ifbp->IFB_CardStat & CARD_STAT_PRESENT) == 0 ) { 
			DSFASSERT( DO_DSFASSERT, TEXT("dhf_ifbp->IFB_CardStat & CARD_STAT_PRESENT) == 0"), 0) ;
        	break ;
        }
*/
        // check whether the WCI is compatible with the primary firmware
        rc = DHF_ERR_PRI_DRV_NOT_COMPATIBLE;
        if ( dhf_ifbp->IFB_CardStat & CARD_STAT_INCOMP_PRI ) {
			DSFASSERT( DO_DSFASSERT , TEXT("dhf_ifbp->IFB_CardStat & CARD_STAT_INCOMP_PRI"), dhf_ifbp->IFB_CardStat & CARD_STAT_INCOMP_PRI) ;
        	break ;
        }

#else

        // check whether the application is compatible with the DUI interface
        rc = DHF_ERR_DRV_APP_NOT_COMPATIBLE;
        if((app_dui_actor != NULL ) && check_comp(&drv_sup, (CFG_RANGES_STRCT *)app_dui_actor ) == -2 ) {
			DSFASSERT(DO_DSFASSERT, TEXT("check_comp(&drv_sup, (CFG_RANGES_STRCT *)app_dui_actor ) == -2"), 0) ;
        	break ;
        }

        // Check whether a card is present
        rc = DHF_ERR_NO_NIC;
        if ( (drv_info.card_stat & CARD_STAT_PRESENT) == 0 ) {
            rc = DHF_ERR_NO_NIC;
			DSFASSERT(DO_DSFASSERT , TEXT("drv_info.card_stat & CARD_STAT_PRESENT) == 0"), 0);
            break;
        }

        // check whether the driver is compatible with the primary firmware
        rc = DHF_ERR_PRI_DRV_NOT_COMPATIBLE;
        if ( drv_info.card_stat & CARD_STAT_INCOMP_PRI ) { 
			DSFASSERT(DO_DSFASSERT, TEXT("drv_info.card_stat & CARD_STAT_INCOMP_PRI"), 0 ) ;
        	break;
        }

#endif

        // check whether the application is compatible with the primary firmware
        rc = DHF_ERR_PRI_APP_NOT_COMPATIBLE;
        if((app_pri_actor != NULL ) && check_comp( &nic_pri_sup, (CFG_RANGES_STRCT *)app_pri_actor ) == -2 ) {
			DSFASSERT(DO_DSFASSERT,TEXT("check_comp( &nic_pri_sup, (CFG_RANGES_STRCT *)app_pri_actor ) == -2"), -2) ;
        	break;
        }

        // TODO: Check on NIC identity if and when the specalists conclude such is needed.

        rc = DHF_SUCCESS;

    } while ( 0 );

	DSFASSERT(rc == DHF_SUCCESS, TEXT("rc == DHF_SUCCESS"), rc) ;
    return(rc) ;
}   // check_driver_card


/*-----------------------------------------------------------------------------
 *
 * Function:    check_pda_crc
 *
 * Abstract:    Checks CRC in Production Data Area
 *
 * Returns:
 *   DHF_SUCCESS - CRC correct
 *   DHF_FAILURE - CRC not correct or PDA_END record not found
 *                        
 * Description:
 *   Locates crc value in PDA. Calculates what CRC should be. Compares the 
 *   calculated value with the value found and returns result of comparison.
 *---------------------------------------------------------------------------*/
#ifndef DSF_HERMESII
static int check_pda_crc( void )
{
    hcf_16  crc;
    hcf_16  *pp;
    LTVP    p = (LTVP) &cfg_prod_data[0];
    int     rc = DHF_FAILURE;         //;?define better error code
    
#ifdef DSF_ALLOC
    pp = find_record_in_pda( (hcf_16*)cfg_prod_data + 2, PDA_END, NULL );
#else
    pp = find_record_in_pda(&cfg_prod_data[2], PDA_END, NULL );
#endif

    if ( pp != NULL ) {
        pp = ((LTVP)pp)->val;
        crc = calc_crc( (hcf_8*)p->val, (pp - p->val) * sizeof(hcf_16) );
        if ( crc == CNV_LITTLE_TO_INT(*pp) ) {
            rc = DHF_SUCCESS;
        } else DSFASSERT( DO_DSFASSERT, TEXT("PDA CRC Error"), CNV_LITTLE_TO_INT(*pp) ) ;
	} else {
		DSFASSERT( DO_DSFASSERT, TEXT("DSF_DOASSERT: PDA_END not Found!"), 0) ;
    }

    return(rc) ;
}   /* check_pda_crc */
#endif // #ifndef DSF_HERMESII


/*-----------------------------------------------------------------------------
 *
 * Function:    condense_error_code
 *
 * Abstract:    Condense the entry error code to the specified one, excluding a
 *              number of error codes (see notes below)
 *
 * Returns:
 *   The condensed error code
 *
 * Description:
 *   We condense all error codes < DHF_RES_MINIMUM and >= DHF_CONDENSE_MINIMUM
 *   to the specified error code except for a number of errors we want to pass 
 *   on to the DHF-programmer as they may help in taking decisions on how to 
 *   continue:
 *   DHF_SUCCESS
 *   DHF_ERR_NO_NIC - this error cannot be returned in Windows as the driver
 *                    will be unloaded if the NIC is removed. Hence, the
 *                    code DHF_ERR_NO_DRV will be returned.
 *   DHF_ERR_IN_USE - user can be asked to close all other WaveLAN utilities
 *   DHF_ERR_NO_NIC - display error message that no NIC is available
 *   DHF_ERR_NO_DRV - display error message that no NIC is available
 *
 *   The actual error code is stored n global variable dhf_last_error. This error
 *   code is returned by dhf_get_last_error.
 *
 * Notes:
 * - The codes that are returned conform to the errors returned by existing WSUs.
 * - This routine must only be used by routines on the DHF-interface. Internal 
 *   DHF routines must return uncondensed error codes.
 * - We do not return all possible HCF/UIL errors for various reasons:
 *   HCF/UIL_FAILURE      - HCF_FAILURE is no longer used anyway? Furthermore,
 *                          as failure is a vague message anyway we might as 
 *                          well return the condense_code.
 *   HCF/UIL_ERR_TIME_OUT - specification of this error does not help the 
 *                          programmer in deciding what to do -> condense
 *
 *   int entry_code,        actual error code
 *   int condense_code      condensed error code
 *
 *---------------------------------------------------------------------------*/
int condense_error_code(int entry_code, int condense_code) 
{
    int rc = entry_code;
    
    // store error for dhf_get_last_error
    dhf_last_error = rc;    

    if ( ( ( entry_code < DHF_RES_MINIMUM ) || 
           ( entry_code >= DHF_ERR_CONDENSE_MINIMUM ) ) &&
         ( entry_code != DHF_SUCCESS    )               && 
         ( entry_code != DHF_ERR_IN_USE )               && 
         ( entry_code != DHF_ERR_NO_NIC )               && 
         ( entry_code != DHF_ERR_NO_DRIVER ) ) {
        rc = condense_code;
    }

	DSFASSERT(rc == DHF_SUCCESS, TEXT("condense_error_code: rc != DHF_SUCCESS"), rc) ;
    return rc;
}   // condense_error_code


/*-----------------------------------------------------------------------------
 *
 * Function:    doconnect
 *
 * Abstract:    Doconnect DHF for download
 *
 * Returns:
 *   DHF_SUCCESS
 *   DHF_ERR_ALLOC           - memory allocation failed (only if DSF_ALLOC is defined)
 *   DHF_ERR_NO_NIC          - no NIC present in slot
 *   DHF_ERR_NOT_COMPATIBLE  - the driver is incompatible
 *   DHF_ERR_IN_USE          - the UIL-driver connection is already used by another 
 *                             application
 *
 * Description:
 *   Sets global variables for DHF.
 *   If successful, retrieves NIC information from card and checks compatibility
 *   of various components.
 *
 * Notes:
 * - The function assumes the application has already been connected by a
 *   call to either hcf_connect or uil_connect.
 * - the values of parameter confirm is stored in global variable 
 *   dhf_confirm_switch.
 *
 *   void              *ifbp,             pointer to the ifb
 *   CFG_RANGE20_STRCT *app_dui_actor,    describes DUI-actor range of the 
 *                                        download application
 *   CFG_RANGE20_STRCT *app_pri_actor,    describes primary-actor range of 
 *                                        the download application
 *   BOOL confirm                         TRUE=ask user for confirmation
 *                                        FALSE=do not ask 
 *
 *---------------------------------------------------------------------------*/
static int doconnect(void *ifbp, CFG_RANGE20_STRCT *app_dui_actor, CFG_RANGE20_STRCT *app_pri_actor, BOOL confirm) 
{
    int rc = DHF_SUCCESS;

#ifdef DHF_WCI
    dhf_ifbp = (IFBP)ifbp;
#else
    dhf_confirm_switch = confirm;
    dhf_ifbp = NULL;
#endif

#ifdef DSF_ALLOC
#ifndef DSF_HERMESII
    // Allocate and initialize block to contain PDA information
    cfg_prod_data = (hcf_16*)dsf_alloc( PDA_SIZE * sizeof( hcf_16 ));
    if ( cfg_prod_data != NULL) {
        cfg_prod_data[0] = PDA_SIZE - 1;
        cfg_prod_data[1] = CFG_PROD_DATA;
    } else {
        rc = DHF_ERR_ALLOC;
    }
#endif // #ifndef DSF_HERMESII    
#endif

#ifndef DSF_HERMESII
//! not sure if this should be included for Hermes-II, since specified that PRI image does not support ACCESS cmd (rlav)
    if ( rc == DHF_SUCCESS ) {
        rc = get_nic_info();
    }

    if ( rc == DHF_SUCCESS ) {
        rc = check_driver_card( app_dui_actor, app_pri_actor );
    }
#endif

    return rc;
}   // doconnect                                              


/*-----------------------------------------------------------------------------
 *
 * Function:    copy_pda_rec
 *
 * Abstract:    Copies a PDA-record from a source record to a destination record
 *
 * Returns:
 *   DHF_SUCCESS - record copied successfully
 *   DHF_FAILURE - some error occurred, possibly one of the parameters not correct
 *
 * Description:
 *   Copies source record, including len and typ fields, to destination. The
 *   write is not done if the destination record would lie beyond PDA_END
 *
 *   LTVP dest,          destination record
 *   LTVP src            source record 
 *
 *---------------------------------------------------------------------------*/
#ifndef DSF_HERMESII
#ifndef DSF_VOLATILE_ONLY
int copy_pda_rec(LTVP dest, LTVP src) 
{
    //NATIE (see note 2) hcf_16 *ptr;
	hcf_16 *pdest;
	hcf_16 *psrc;
    int i;
    int rc = DHF_SUCCESS;
    
    if ( ( src != NULL ) && ( dest != NULL ) 
#ifdef DSF_ALLOC
//HWi Error in pointers???
         && ( (cfg_prod_data + PDA_SIZE - dest ) > src->len + 1 ) ) {
#else
         && ( (hcf_16)(&cfg_prod_data[PDA_SIZE] - (hcf_16*) dest ) > src->len + 1 ) ) {
/* NOTE: Distance to end of PDA structure should be greater than the length
 * of the new source record. I made the change for DSF_ALLOC although I never 
 * tested it (this specific check was implemented correctly in dhf_add_pda_record())
 */
#endif
		i = src->len + 1;
		pdest = (hcf_16*)dest;
		psrc = (hcf_16*)src;
        while ( i > 0 ) {
            *pdest = *psrc;
            psrc++; pdest++; i--; 
        }
	   /* NOTE 2: reimplementation with correct pointer arithmetic (the old version
		*  caused the CRC PDA to become corrupt).
		*/
    } else {
        rc = DHF_FAILURE;
    }
    
    return rc;
}   // copy_pda_rec
#endif // DSF_VOLATILE_ONLY
#endif // #ifndef DSF_HERMESII


/*-----------------------------------------------------------------------------
 *
 * Function:    determine_firmware_type
 *
 * Abstract:    Determines the type of a firmware memory image.
 *
 * Returns: The type of firmware:
 *            FT_UNKNOWN - Unknown or invalid type of firmware
 *            FT_PRIMARY - Primary
 *            FT_STATION - Station
 *            FT_AP      - Access Point
 *
 * Description:
 *   The firmware type can be derived from the CFG_IDENTITY_STRCT in memimage.
 *---------------------------------------------------------------------------*/
static int determine_firmware_type( memimage *firmware )
{
    int rc = FT_UNKNOWN;

    if ( firmware != NULL ) {
        switch( firmware->identity->typ ) {
			//TODO: adjust next DEFINE until one is assigned!!!
            case CFG_AP_IDENTITY:
                rc = FT_AP;
                break;
            case CFG_PRI_IDENTITY:
                rc = FT_PRIMARY;
                break;
            case CFG_FW_IDENTITY:
                rc = FT_STATION;
				break;
			default:
				DSFASSERT( DO_DSFASSERT, TEXT("DSF_DOASSERT"), 0) ;
				break;
        }
    }
    return rc;
}   // determine_firmware_type


/*-----------------------------------------------------------------------------
 *
 * Function:    dodisconnect
 *
 * Abstract:    Dodisconnects DHF after use:
 *              performs any necessary cleanup
 *
 * Return:
 *   None
 *
 * Description:
 *   Dodisconnects DHF: Frees any allocated resources.
 *---------------------------------------------------------------------------*/
static void dodisconnect( void ) {

    dhf_ifbp = NULL;

#ifdef DSF_ALLOC
#ifndef DSF_HERMESII
    if ( cfg_prod_data != NULL ) {
        dsf_free( cfg_prod_data, PDA_SIZE * sizeof(hcf_16) );
    }
#endif // #ifndef DSF_HERMESII
#endif // DSF_ALLOC

}  // dodisconnect


/*-----------------------------------------------------------------------------
 * Downloads a firmware image to the NIC.
 *
 * Returns:
 *   DHF_SUCCESS             - download completed successfully.
 *   DHF_ERR_NO_NIC          - no NIC present in slot
 *   DHF_ERR_NO_DRV          - no driver present
 *   DHF_ERR_DOWNLOAD        - problem occurred in download
 *   + error codes returned by uil_action when using UIL
 * 
 * Description:
 *   The firmware is downloaded in one or more steps. When using the UIL the
 *   NIC is blocked for processes that might interfere with download.
 *
 * Notes:
 * - Assumes the firmware image has been checked for validity previously.
 * - If more granularity is required (e.g. to show progress to user) then use 
 *   download_start, download_next, and download_stop separately.
 *---------------------------------------------------------------------------*/
static int download_firmware_image( void ) 
{
    int   rc;
#ifdef DHF_UIL
    BOOL  blocked = FALSE;
#endif	// DHF_UIL

    rc = download_start();
	DSFASSERT(rc == DHF_SUCCESS, TEXT("download_start rc != DHF_SUCCESS"), rc) ;

#ifdef DHF_UIL
//TODO: block must be done at different location!!! before any access to card???
    // When using the UIL block for interfering processes, so download will
    // not be disturbed.
    if ( rc == DHF_SUCCESS ) {
        rc = uil_action( UIL_ACT_BLOCK );
        if ( rc == DHF_SUCCESS ) {
            blocked = TRUE;
        }
    }
	DSFASSERT(rc == DHF_SUCCESS, TEXT("uil_action UIL_ACT_BLOCK rc != DHF_SUCCESS"), rc) ;
#endif	// DHF_UIL

    while ( rc == DHF_SUCCESS ) {
        rc = download_next();
    }
	DSFASSERT((rc == DHF_SUCCESS || rc == DHF_DOWNLOAD_COMPLETED), TEXT("download_next rc != DHF_SUCCESS"), rc) ;

    if ( rc == DHF_DOWNLOAD_COMPLETED ) {
        rc = download_stop();
    }
	DSFASSERT(rc == DHF_SUCCESS, TEXT("download_stop rc != DHF_SUCCESS"), rc) ;

#ifdef DHF_UIL
    if ( blocked == TRUE ) {
        // We're not interested in the result of UNBLOCKING the NIC
        (void)uil_action( UIL_ACT_UNBLOCK );
    }
#endif	// DHF_UIL

	DSFASSERT(rc == DHF_SUCCESS, TEXT("rc != DHF_SUCCESS"), rc) ;
    return(rc) ;
}   // download_firmware_image


/*-----------------------------------------------------------------------------
 *
 * Function:    download_next
 *
 * Abstract:    Downloads next chunk of a block of firmware to (non-)volatile NIC-memory
 *
 * Returns:
 *   DHF_SUCCESS             - one step in download completed successfully.
 *   DHF_DOWNLOAD_COMPLETED  - last step of download completed
 *   DHF_ERR_NO_NIC          - no NIC present in slot
 *   DHF_ERR_NO_DRV          - no driver present
 *   DHF_ERR_DOWNLOAD        - problem occurred in download
 *
 * Description:
 *   This routine is used after successful preparation by download_start() to
 *   consecutively download blocks of information to NIC nonvolatile RAM.
 *   This function must be called repeatedly until either it returns 
 *   DHF_DOWNLOAD_COMPLETED, or a value other than DHF_SUCCESS.
 *
 * Notes:
 * - the first call to download_next must be preceded by a call to 
 *   download_start, and followed by a call to download_stop on successful 
 *   completion.
 *---------------------------------------------------------------------------*/
static int download_next( void ) {
#ifndef DSF_HERMESII
    CFG_DL_START_STRCT      ltv_dl = { LOF(CFG_DL_START_STRCT) /*!!!, CFG_DLNV_START*/ };
#endif

    hcf_16                  ltv[20];             // used for MailBox and CFG_DL_STOP,
                                                 // keep a decent size such that ASSERT message
                                                 // as MBIB are interpretable
    char                    safe[10];             // safekeep part of image temporarily overwritten with L and T

#ifdef DSF_HERMESII
    CFG_PROG_STRCT          *p;
#else
    LTVP                    p;                   // mostly points to ltv, shortly it points into image
#endif
    int                     rc = DHF_SUCCESS;
    hcf_16                  crc, tlen;
    hcf_32                  addr;

#ifndef DSF_HERMESII
    if ( dhf_to_volatile == TRUE ) {
        ltv_dl.typ = CFG_DLV_ADDR;
    } else {
        ltv_dl.typ = CFG_DLNV_START;
    }
#endif
    if ( dhf_len == 0 ) {
    
        // End of segment reached or at very start of download process.
        // Set pointer to next (or first) segment.
        // For the last segment addr is NULL.
        if ( dhf_memblockp == NULL ) {
            dhf_memblockp = dhf_firmware->codep;
        } else {
            dhf_memblockp++;
        }

        if ( dhf_memblockp != NULL ) {
            addr = dhf_memblockp->addr;
            if(addr == 0 ) {
                dhf_len = 0;
                rc = DHF_DOWNLOAD_COMPLETED;
            } else {
                dhf_len = dhf_memblockp->size; // size is blocksize excluding preceding length, type ... bytes.
                dhf_index = 0;
#ifndef DSF_HERMESII
                // Plug the CRC if needed
                if ( dhf_memblockp->flags /*& CRC*/ ) {
                    crc = calc_crc( dhf_memblockp->data + 4, dhf_len - 2 );
                    DSFDBG(_stprintf(dbgbuf, TEXT("plug_crc addr: %lp, index:%04X, CRC:%04X "), dhf_memblockp->data, dhf_len, crc);)
                    DSFDBG(dsf_debug_message( dbgbuf, DHF_INFO, FALSE );)
                    *(hcf_16*)(dhf_memblockp->data + dhf_len + 2) = crc;
                }
				DSFDBGEXT(dbg_fdisplay_hex(TEXT("c:\\temp\\plugged.dat"), TEXT("a"), memblockp->data, len);)

                // Start the download of the new segment (supply NV-RAM address to NIC)
                ltv_dl.low_addr  = (hcf_16)addr;
                ltv_dl.high_addr = (hcf_16)(addr>>16);

                rc = PUT_INFO( &ltv_dl );
#else
                rc = DHF_SUCCESS;
#endif
            }
        } else {
            rc = DHF_ERR_DOWNLOAD;
        }
    }

    // Copy next chunk to the NIC
    // Wait for completion of programming a single chunk
	if ( rc == DHF_SUCCESS ) {
		//HWi 3/13/2001 4:16PM Removed the len for volatile	because USB can NOT do large buffer size (now len usb is 2000 decimal)
		// When downloading to volatile memory the size of the download buffer does not play a role.
//		if ( dhf_to_volatile == FALSE ) {

//! Hermes-II volatile download for non-USB does not have a maximum buffer size (rlav)
//! TODO : Fix this code also for Hermes-II USB (rlav)
#ifdef DSF_HERMESII
        /************************************************************************************************
        * Hermes-II does not use an intermediate buffer in NIC-RAM for volatile download. Instead, the  *
        * host directly writes the memory block at the desired NIC-RAM address via the aux port.        *
        ************************************************************************************************/
		// get minimum of "what fits in intermediate NIC RAM buffer" and "left to do"
        if ( dhf_to_volatile == TRUE ) {
		   tlen = min(dhf_len, 2 * HCF_MAX_LTV - sizeof(CFG_PROG_STRCT) + 2 * 2);
	    }
	    else {
		   tlen = min(dhf_len, dl_buf.buf_len );
		}
#else
		tlen = min(dhf_len, dl_buf.buf_len );
#endif

//		} else {
//			tlen = dhf_len;
//	}

#ifdef DHF_UIL
         tlen = min(tlen, UIL_DATBUF_SIZE);                 // also take into account the size of
                                                            // the UIL's data buffer
#endif

		if ( dhf_index >= dhf_memblockp->size ) {
			rc = DHF_ERR_DOWNLOAD;
		} else {
#ifdef DSF_HERMESII
	        p = (CFG_PROG_STRCT*)(dhf_memblockp->data + dhf_index);        // convert chunck to LTV by overwrite words preceding chunk
	        memcpy( safe, p, sizeof(CFG_PROG_STRCT) - 2);      // safekeep overwritten words to restore after use as LTV
#else
	        p = (LTVP)(dhf_memblockp->data + dhf_index);        // convert chunck to LTV by overwrite words preceding chunk
	        memcpy( safe, p, 4 );                              // safekeep overwritten words to restore after use as LTV
#endif
//	        DSFDBG(dbg_display_hex( dhf_memblockp->data + dhf_index, 32 );)

//tlen is the actual 'payload' of the chunk, excluding the 'length', 'type' etc fields.
#if defined DSF_HERMESII
			DSFASSERT( (tlen & 0x01) == 0, TEXT("Odd chunk size"), tlen) ;
	        p->len = tlen/2 + sizeof(CFG_PROG_STRCT)/2 - 2;
	        p->typ = CFG_PROG;
	        if ( dhf_to_volatile == TRUE ) {
	            p->mode = CFG_PROG_VOLATILE;
	        } else {
	            p->mode = CFG_PROG_FLASH;
	        }
	        p->low_addr  = (hcf_16) ( (dhf_memblockp->addr + dhf_index) & 0xFFFF);
	        p->high_addr = (hcf_16) ( ((dhf_memblockp->addr + dhf_index)>>16) & 0xFFFF);
#else
	        p->len = tlen/2 + 1;                                // +1 for typ field
	        if ( dhf_to_volatile == TRUE ) {
	            p->typ = CFG_DLV_DATA;
	        } else {
	            p->typ = CFG_DLNV_DATA;
	        }
#endif

//Use following line when testing flow without wanting real access to the card
//*JMES*/    p->typ = CFG_DLNV_DATA ^ 0x1000;                    // MAKE IT INVALID SO NOTHING CAN HAPPEN
	
	        rc = PUT_INFO( p );
	
// 	        memmove( p, safe, 4);                               // restore overwritten chunk
#ifdef DSF_HERMESII
	        memcpy( p, safe,  sizeof(CFG_PROG_STRCT) - 2);      // safekeep overwritten words to restore after use as LTV
#else
	        memcpy( p, safe, 4 );                               // safekeep overwritten words to restore after use as LTV
#endif

	        dhf_len -= tlen;                                    // adjust remaining length of segment
	        dhf_index += tlen;

//! rlav : I don't see the use of the following statement (p = (LTVP)ltv; ) :
//!	        p = (LTVP)ltv;


#if !defined DSF_VOLATILE_ONLY && !defined DSF_HERMESII
	        dhf_block_nr++;
#endif
		}
    }   //end of single segment

    return rc;
}   /* download_next */


/*-----------------------------------------------------------------------------
 *
 * Function:    download_start
 *
 * Abstract:    Prepares download to firmware by initializing some global variables
 *
 * Returns:
 *   DHF_SUCCESS
 *   DHF_ERR_NO_NIC     - no NIC present in slot
 *   DHF_ERR_NO_DRV     - no driver present
 *   DHF_ERR_DOWNLOAD   - if plugging failed or if firmware image is not valid
 *
 * Description:
 *   Prepares the image for download by plugging primary and production data 
 *   plug records stored in the firmware image. If preparation and blocking is 
 *   successful initializes global variables dhf_memblockp, dhf_len, and 
 *   dhf_index.
 *---------------------------------------------------------------------------*/
static int download_start( void ) 
{
#ifdef 	DSF_HERMESII
    return DHF_SUCCESS;
#else
	int  rc = DHF_SUCCESS ;
    CFG_DL_START_STRCT ltv_addr = { LOF(CFG_DL_START_STRCT), CFG_DLV_START };
    DSFDBG(dbg_display_download_info( dhf_firmware );)

	do {
        if((rc = extend_pda_with_pri_records( dhf_firmware ) ) != DHF_SUCCESS ) {
        	break;
        }
        if((rc = plug_production_data( dhf_firmware ) ) != DHF_SUCCESS ) {
        	break;
        }
        if((rc = plug_pri_records( dhf_firmware ) ) != DHF_SUCCESS ) {
        	break;
        }

        if ( dhf_to_volatile == TRUE ) {
            ltv_addr.low_addr  = (hcf_16)dhf_firmware->execution;
            ltv_addr.high_addr = (hcf_16)(dhf_firmware->execution >> 16);
            rc = PUT_INFO( &ltv_addr );
        }                             
        if(rc != DHF_SUCCESS ) {
        	break ;
        }

#ifndef DSF_VOLATILE_ONLY
        dhf_progress_step = calc_progress_step( dhf_firmware, dhf_to_volatile );
#endif
    } while ( 0 );


    DSFDBG(dbg_display_download_info( dhf_firmware );)

    return rc;
#endif // #ifndef DSF_HERMESII
}   // download_start


/*-----------------------------------------------------------------------------
 *
 * Function:    download_stop
 *
 * Abstract:    Completes download of firmware to (non-)volatile NIC-memory
 *
 * Returns:
 *   DHF_SUCCESS        - succesful completion of download
 *   DHF_ERR_NO_NIC     - no NIC present in slot
 *   DHF_ERR_NO_DRV     - no driver present
 *   DHF_ERR_DOWNLOAD   - in case of some error in download (condensed)
 *
 * Description:
 *   Ends download process by:
 *   1. Putting CFG_DL_STOP record to the NIC.
 *   2. In case of download of primary firmware retrieves NIC information
 *      from the NIC as this information may have changed by the download
 *      process.
 *---------------------------------------------------------------------------*/
static int download_stop( void )  {

    int      rc = DHF_SUCCESS;

	LTVP     p;

#if defined DSF_HERMESII
	CFG_PROG_STRCT ltv;
	ltv.len = sizeof(CFG_PROG_STRCT) / 2 - 2;
	ltv.typ = CFG_PROG;
	ltv.mode = CFG_PROG_STOP;
	ltv.low_addr = (hcf_16)(dhf_firmware->execution & 0xFFFF);
	ltv.high_addr = (hcf_16)((dhf_firmware->execution>>16) & 0xFFFF);
	p = (LTVP)&ltv;
#else
    hcf_16   ltv[20];                      /* used for CFG_DL_STOP,
                                            * keep a decent size such that ASSERT message
                                            * as MBIB are interpretable */
    // Stop the download (re-activate normal NIC behaviour)
    p = (LTVP)ltv;
    p->len = 1;
    p->typ = CFG_DL_STOP;
#endif
    rc = PUT_INFO( p );
    if ( ( rc == HCF_ERR_INCOMP_STA ) && ( dhf_firmware->identity->typ == CFG_PRI_IDENTITY ) ) {
        rc = DHF_SUCCESS;
    }

    // DSFDBG(_stprintf(dbgbuf, TEXT("Result of download to NIC-RAM: %d"), rc);)
    // DSFDBG(dsf_debug_message( dbgbuf, DHF_INFO, FALSE );)
	OUTPUTDEBUGMSG((TEXT("Result of download to NIC-RAM: %d"), rc)) ;

    return rc;
}   // download_stop


/*-----------------------------------------------------------------------------
 *
 * Function:    extend_pda_with_pri_records
 *
 * Abstract:    extends production data area with primary records
 *
 * Returns:
 *   DHF_SUCCESS
 *   DHF_ERR_NO_NIC     - no NIC present in slot
 *   DHF_FAILURE        - problem occurred (condensed)
 *
 * Description:
 *   Locate end of information in (temporary) production data area (PDA)
 *     retrieved from the NIC
 *   For all primary plug records in the image to be downloaded
 *     If enough room is left in the (temporary) PDA then
 *       Retrieve the information from the NIC (GET_INFO)
 *       and store that information in the (temporary) PDA
 *   If enough room is left to add the PDA_END record to the (temporary) PDA then
 *     Add that record (len=0, typ=0)
 *   Else
 *     Return error-code
 *
 * Notes:
 * - Information on the primary records needed is stored in the memimage. Later on 
 *   they are used by function plug_pri_records to plug the memimage.
 * - (Station) firmware that executes in RAM has no access to primary firmware
 *   in Flash-ROM. However, station firmware needs some information stored in
 *   primary firmware in order to operate properly. Therefore, the PDA is
 *   extended with these primary (plug) records.
 * - The extended PDA is not written back to the NIC. The records are only added
 *   here so that other routines in the DHF have access to them.
 *
 *   memimage *firmware      // image to be extended
 *---------------------------------------------------------------------------*/
#ifndef DSF_HERMESII
static int extend_pda_with_pri_records(memimage *firmware) 
{
    plugrecord  *plugrecordp = firmware->priplug;
    hcf_16      len;
    int         rc = DHF_SUCCESS;
    hcf_16      *pp;
    hcf_32      code;

    DSFASSERT(plugrecordp != NULL, TEXT("plugrecordp == NULL"), 0) ;
    
#ifdef DSF_ALLOC
    pp = find_record_in_pda( (hcf_16*)cfg_prod_data + 2, PDA_END, NULL );  //locate PDAEND record
#else
    pp = find_record_in_pda(&cfg_prod_data[2], PDA_END, NULL ); //locate PDAEND record
#endif

	if(pp != NULL) {
		while ( ( rc == DHF_SUCCESS ) && ( code = plugrecordp->code ) != 0 ) {                 //scan through all plugrecords in firmware image
			if ( (code & CODEMASK) <= 0x0000FFF0 ) {                     //act on primary plug records only
	            rc = DHF_FAILURE;
#ifdef DSF_ALLOC
	            len = (cfg_prod_data + PDA_SIZE - pp) ; 				//get max L for LTV to fit in cfg_prod_data
#else
	            len = (hcf_16)(&cfg_prod_data[PDA_SIZE] - pp) ; 		//get max L for LTV to fit in cfg_prod_data
#endif
	            len = min( len, PDA_SIZE );
	            if ( len > 2 ) {
	            	// Don't change endianess of len/type yet, because HCF needs native endianess!
	                pp[0] = len - 1 ;
	                pp[1] = (hcf_16)(code & CODEMASK) ;
	                rc = GET_INFO( pp );
	            	// We MUST use the len returned by GET_INFO()!
	            	len = pp[0] ;
	            	// Change endianess of len/type to LITTLE, because PDA Data is all in little endianess
	                pp[0] = CNV_INT_TO_LITTLE(len) ;
	                pp[1] = (hcf_16)CNV_INT_TO_LITTLE((code & CODEMASK));
	                pp += len + 1;                    //get pointer to next free location
	            }
	        }
	        plugrecordp++;
	    }
	    if ( rc == DHF_SUCCESS ) {
#ifdef DSF_ALLOC
	        if ( (cfg_prod_data + PDA_SIZE - pp) < 2 ) {
#else
	        if ( (&cfg_prod_data[PDA_SIZE] - pp) < 2 ) {
#endif
	            rc = DHF_FAILURE;
	        } else {
	            pp[0] = pp[1] = 0;      //;?this seems intended to restore (sort of) PDA_END record
	                                    //;? which seemed overwritten with the first primary LTV to use in PriPlugging
	                                    //;? also it does not handle numerical matches between LPDA and Pri records
	        }
	    }
		if ( rc != DHF_SUCCESS && rc != DHF_ERR_NO_NIC ) {
		    rc = DHF_FAILURE;  // condense return value
		}
	}
	else {
		rc = DHF_FAILURE ;
	}	
 
    DSFDBG(_stprintf(dbgbuf, TEXT("Result of extend_pda_with_pri_records: %d"), rc);)
    DSFDBG(dsf_debug_message( dbgbuf, DHF_INFO, FALSE );)
    return(rc) ;
}   /* extend_pda_with_pri_records */
#endif // #ifndef DSF_HERMESII


/*-----------------------------------------------------------------------------
 *
 * Function:    find_addr_in_image
 *
 * Abstract:    Looks for a memblock in the firmware image that includes the
 *              address passed in as parameter
 *
 * Returns:
 *   A pointer to the memblock that includes the address, or NULL if the address
 *   is not found.
 *
 * Description:
 *   Follow the list of memory blocks in the firmware image until the addr 
 *   searched for lies within the block.
 *---------------------------------------------------------------------------*/
static memblock *find_addr_in_image( 
    memimage *firmware,     // firmware image containing address
    hcf_32 addr             // address to be located
                            //;?does addr need to be an hcf_32
) {
    memblock    *memblockp = firmware->codep;

	DSFASSERT( memblockp != NULL, TEXT("memblockp == NULL"), 0) ;

	while ( memblockp->addr ) {
        if ( memblockp->addr <= addr && addr <= memblockp->addr + memblockp->size ) {
//TODO: what is index to be???
            DSFDBGEXT(_stprintf( dbgbuf, TEXT("addr:%08lX offset:%04X"), addr, index );)
            DSFDBGEXT(dsf_debug_message( dbgbuf, DHF_INFO, TRUE );)
            break;
        }
        memblockp++;
    }
    if ( memblockp->addr == 0 ) {
        memblockp = NULL;
    }

//    DSFDBG(_stprintf(dbgbuf,TEXT("find_addr_in_image: %lp"), addr);)
//    DSFDBG(dsf_debug_message( dbgbuf, DHF_INFO, FALSE );)

    return memblockp;
}   /* find_addr_in_image */


/*-----------------------------------------------------------------------------
 *
 * Function:    find_record_in_pda
 *
 * Abstract:    Locates a production data record with specified code in the
 *              production data area.
 *
 * Returns:
 *   Pointer to record if code found, else NULL
 *
 * Description:
 *   Follow the list of PDA (of type LTV) records until the typ-field of the
 *   LTV-record is equal to the code parameter, or until the end of the PDA is
 *   reached (The end of the PDA is indicated by a record with code 0).
 *   If the len parameter is not NULL then there are two possibilities:
 *   1. if len == 0 the length of the record found is returned in len
 *   2. if len != 0 the length of the record found is compared with len; if not
 *      equal then the record found is not correct and NULL is returned
 *
 *   hcf_16 *pdap,   // pointer to production data area
 *   hcf_16 code,    // code of production data record to be located
 *   hcf_16 *len     // length of the record found:
 *                   // - set to required length or zero if any length is OK
 *                   // - set to NULL if not needed
 *---------------------------------------------------------------------------*/
#ifndef DSF_HERMESII
hcf_16 *find_record_in_pda(hcf_16 *pdap, hcf_16 code, hcf_16 *len ) 
{

    hcf_16	pdac = PDA_END ;
    hcf_16*	pdaheadp = pdap;

	DSFASSERT( pdap != NULL, TEXT("pdap == NULL"), 0 ) ;

    //+1 in the following line as we look at the typ-field of the LTV-record    
    //NATIE (see note)
    while((pdap < (pdaheadp + PDA_SIZE - 2)) && (( pdac = CNV_LITTLE_TO_INT(*(pdap+1)) ) != PDA_END) ) { 
        if(pdac == code) {
        	break;
        }
        DSFDBGEXT(_stprintf( dbgbuf, TEXT("%lp %04X "), pdap, CNV_LITTLE_TO_INT(*pdap) );)
        DSFDBGEXT(dsf_debug_message( dbgbuf, DHF_INFO, FALSE );)
        pdap += CNV_LITTLE_TO_INT(*pdap) + 1;
    }
/* NOTE: If a corrup PDA is read this while loop might pass the end of the 
 * PDA structure and causes crashes - blue screen on NT. Since *pdap is 
 * passed as a pointer to a hcf_16 we dont now the length of the remaining 
 * PDA but assume that it is PDA_SIZE - 2. This logic is also follow in
 * other places of this module.
 */

    if ( pdac == code ) {
        if ( len != NULL ) {
            if ( *len == 0 ) {
                *len = CNV_LITTLE_TO_INT(*pdap);
            } else {
                if ( CNV_LITTLE_TO_INT(*pdap) != *len ) {
                    pdap = NULL;
                }
            }
        }
    } else {
        pdap = NULL;
    }
    return pdap;
}   /* find_record_in_pda */
#endif // #ifndef DSF_HERMESII


/*-----------------------------------------------------------------------------
 *
 * Function:    get_nic_info
 *
 * Abstract:    Retrieves all information needed for download from the NIC.
 *
 * Returns:
 *   DHF_SUCCESS
 *   DHF_FAILURE          - error in CRC of PDA
 *   DHF_ERR_NO_DRV       - no driver present
 *   DHF_ERR_NO_NIC       - no NIC present in slot
 *
 * Description:
 *   The routine successively retrieves the LTV-records from the card and 
 *   stores them in array dui_info. If retrieval of all these records was 
 *   successful, the CRC of the PDA is checked.
 *---------------------------------------------------------------------------*/
static int get_nic_info( void ) {

    int  rc = DHF_SUCCESS;
    DUI_INFO_STRUCT_PTR pp = dui_info;
    LTVP p;                                                                  

#ifndef DSF_HERMESII
    rc = dhf_load_pda();
#endif // #ifndef DSF_HERMESII   

    while ( ( rc == DHF_SUCCESS ) && ( ( p = pp->ltvp) != NULL ) ) {
    	// Set len to origional len. This len is changed to real len by GET_INFO()
    	p->len = pp->len ;
        rc = GET_INFO( p );
		DSFASSERT(rc == DHF_SUCCESS, TEXT("Error: get_nic_info"), rc) ;
		DSFASSERT(rc == DHF_SUCCESS, TEXT("Error: get_nic_info Type "), p->typ) ;
		DSFASSERT(rc == DHF_SUCCESS, TEXT("Error: get_nic_info Len"), p->len) ;

		/*---------------------------------------------------------------------
		 * DHF_ERR_LEN can be returned in two cases:
		 * 1. the buffer was really not large enough to store the 
		 *    requested information
		 * 2. the RID requested was unknown. In this case the len-field
		 *    of the LTV-record is 0.
		 * In the latter case we can just ignore the error and continue.
		 *-------------------------------------------------------------------*/
		if ( ( rc == DHF_ERR_LEN ) && ( (hcf_16)(p->len) == 0 ) ){
			rc = DHF_SUCCESS;
		}
		pp++ ;
    }

#ifndef DSF_HERMESII
    if ( rc == DHF_SUCCESS ) { 
    	rc = check_pda_crc(); 
    }
#endif    
    if ( rc == DHF_SUCCESS ) {
#ifdef DHF_WCI
        DSFDBG(_stprintf(dbgbuf, TEXT("card stat: 0x%04x"), dhf_ifbp->IFB_CardStat);)
#else
        DSFDBG(_stprintf(dbgbuf, TEXT("card stat: 0x%04x"), drv_info.card_stat);)
#endif
        DSFDBG(dsf_debug_message( dbgbuf, DHF_INFO, FALSE );)
        DSFDBG(dbg_display_nic_info();)

    } else if ( rc != DHF_ERR_NO_NIC ) {

        DSFDBG(_stprintf(dbgbuf, TEXT("get_nic_info failed: %d"), rc );)
        DSFDBG(dsf_debug_message( dbgbuf, DHF_INFO, FALSE );)
    }

    return rc;
}   /* get_nic_info */


/*-----------------------------------------------------------------------------
 *
 * Function:    plug_pri_records
 *
 * Abstract:    Plug firmware image with primary records
 *
 * Returns:
 *   DHF_SUCCESS
 *   DHF_ERR_PLUG_PRI_DATA - problem occurred (condensed)
 *
 * Description:
 *   For all primary records to be plugged
 *     Locate position of plug record\
 *       If enough room is available to fit in the plug record then
 *         Locate the position of the plug record in the image to be downloaded
 *         Copy the contents of NIC-record to the image
 *       Else
 *         Return DHF_ERR_PLUG_PRIMARY_DATA
 *
 *   memimage *firmware the image to be plugged 
 *
 *---------------------------------------------------------------------------*/
#ifndef DSF_HERMESII
static int plug_pri_records(memimage *firmware) 
{
    plugrecord  *plugrecordp = firmware->priplug;
    hcf_8       *p;                   // pointer to area to be plugged
    hcf_32      code;                 // Code to plug
    hcf_16      *pdap;                // pointer to matching code found in pda
    int         rc = DHF_SUCCESS;
    memblock    *memblockp;

    //Scan through all plugrecords in firmware image
    while ( ( code = plugrecordp->code ) != 0 ) {
        if ( (code & CODEMASK) <= 0x0000FFF0 ) {			//act on primary plug records only
#ifdef DSF_ALLOC
            pdap = find_record_in_pda( (hcf_16*)cfg_prod_data + 2, (hcf_16)(code & CODEMASK), NULL );     //;?requires that pri plug records do not overlap PDA plug records
#else
            pdap = find_record_in_pda(&cfg_prod_data[2], (hcf_16)(code & CODEMASK), NULL );     //;?requires that pri plug records do not overlap PDA plug records
#endif
            if ( pdap == NULL ) {
//JMES??? what happens if we end up here? can we just continue?
                DSFDBG(_stprintf(dbgbuf, TEXT("code %08lX not found in Primary Data Area"), code);)
                DSFDBG(dsf_debug_message( dbgbuf, DHF_INFO, FALSE );)
                //;? rc = DHF_FAILURE;  or do we need plugging rules
            } else if((hcf_32) ((CNV_LITTLE_TO_INT(*pdap) + 1 ) * 2) > plugrecordp->len ) {

                //!! Be aware of two differences with production data plug records:
                //!! 1. primary plug records may be smaller that the room available
                //!!    for them, whereas production data plug records must fit
                //!!    exactly.
                //!! 2. *pdap+1 this time in stead of -1 for plug_production_data,
                //!!    because we are interested in the primary record which is
                //!!    embedded in the record pointed to by pdap and thus the
                //!!    len field of which lies at *pdap+1.
                DSFDBG(_stprintf(dbgbuf, TEXT("size mismatch in Primary Data Area, code:%08lX PDA (L)size: %04X, image (byte)size: %04X"),
                       code, CNV_LITTLE_TO_INT(*pdap), plugrecordp->len );)
                DSFDBG(dsf_debug_message( dbgbuf, DHF_INFO, FALSE );)
                rc = DHF_ERR_PLUG_PRIMARY_DATA;
            } else {

	            DSFDBGEXT(_stprintf(dbgbuf TEXT("Plugrecord -> code: %08x  Address: %08x  Length: %08x"), plugrecordp->code, plugrecordp->address, plugrecordp->length );)
                DSFDBG(dsf_debug_message( dbgbuf, DHF_INFO, FALSE );)
                DSFDBGEXT(_stprintf(dbgbuf,  TEXT("1) PDA, 2) Image before plugging 3) image after plugging" ));)
                DSFDBG(dsf_debug_message( dbgbuf, DHF_INFO, FALSE );)
                DSFDBGEXT(dbg_display_hex( pdap, CNV_LITTLE_TO_INT(*pdap) + 1 );)     //display L,T and V of PDA_LTV

                if ( ( memblockp = find_addr_in_image( firmware, plugrecordp->addr ) ) != NULL ) {

                    // Calculate the offset of the data to be plugged in the image.
                    // We add 4 because of extra L- and T-field at start of data
                    // field in memblock
                    p = &memblockp->data[plugrecordp->addr - memblockp->addr + 4];
					DSFDBGEXT(dbg_display_hex( p, plugrecordp->len / 2 );)
                    memcpy( p, pdap, (CNV_LITTLE_TO_INT(*pdap) + 1) * 2);
	                DSFDBGEXT(dbg_display_hex( p, plugrecordp->len / 2 );)
                } else {
                    rc = DHF_SUCCESS;
                }
            }
        }
        plugrecordp++;
    }
    return rc;
}   /* plug_pri_records */
#endif // #ifndef DSF_HERMESII


/*-----------------------------------------------------------------------------
 *
 * Function:    plug_production_data
 *
 * Abstract:    Plug firmware image with production data records
 *
 * Returns:
 *   DHF_SUCCESS
 *   DHF_ERR_PLUG_PRODUCTION_DATA
 *
 * Description:
 *   For all records to be plugged
 *     Locate position of plug record
 *     Apply special plugging rules if necessary
 *     If enough room is available to fit in the plug record then
 *       Locate the position of the plug record in the image to be downloaded
 *       Copy the contents of NIC-record to the image
 *     Else
 *       Return DHF_ERR_PLUG_PRODUCTION_DATA
 *---------------------------------------------------------------------------*/
#ifndef DSF_HERMESII
static int plug_production_data(memimage *firmware) 
{
    plugrecord  *plugrecordp = firmware->pdaplug;
    int         rc = DHF_SUCCESS, plugrc = DHF_SUCCESS;
    memblock    *memblockp;
    hcf_32      code;                 // Code to plug
    hcf_16      *pdap;                // pointer to matching code found in pda
    hcf_8       *p;                   // pointer to area to be plugged

	DSFASSERT( plugrecordp != NULL, TEXT("plugrecordp == NULL"), 0) ;

    //Scan through all plugrecords in firmware image
//TODO: what to do if apply_plug_rules returns DHF_FAILURE???
    while((rc != DHF_ERR_PLUG_PRODUCTION_DATA ) && ( code = plugrecordp->code ) != 0 ) {
#ifdef DSF_ALLOC
        if((plugrc = apply_plug_rules( (hcf_16*)cfg_prod_data + 2, (hcf_16 **)&pdap, code & CODEMASK ) ) == DHF_SUCCESS ) {
#else
        if((plugrc = apply_plug_rules(&cfg_prod_data[2], (hcf_16 **)&pdap, (hcf_16)(code & CODEMASK) ) ) == DHF_SUCCESS ) {
#endif
            if ( pdap == NULL ) {
//JMES??? what happens if we end up here? can we just continue?
                DSFDBG(_stprintf(dbgbuf, TEXT("code %08lX not found in Plug Data Area"), code);)
                DSFDBG(dsf_debug_message( dbgbuf, DHF_INFO, FALSE );)
                //;? rc = DHF_FAILURE;  or do we need plugging rules
            } else if((hcf_32) (( CNV_LITTLE_TO_INT(*pdap) - 1 ) * 2) != plugrecordp->len ) {
                //!! Be aware of two differences with primary plug records:
                //!! 1. as opposed to plug_pri_records '!=' rather than '>' as production data
                //!!    plug records must fit exactly at their location
                //!! 2. *pdap-1 in stead of +1. pdap points to the typ-field
                //!!    of the record we're interested in; hence the len-field
                //!!    is positioned at *pdap-1.
                DSFDBG(_stprintf(dbgbuf, TEXT("size mismatch Plug Data Area, code:%08lX PDA (L)size: %04X, image (byte)size: %04X"),
                       code, CNV_LITTLE_TO_INT(*pdap), plugrecordp->len);)
                DSFDBG(dsf_debug_message( dbgbuf, DHF_INFO, FALSE );)

                rc = DHF_ERR_PLUG_PRODUCTION_DATA;
            } else {
                DSFDBGEXT(printf( TEXT("Plugrecord -> code: %08x  Address: %08x  Length: %08x"), plugrecordp->code, plugrecordp->address, plugrecordp->len );)
                DSFDBGEXT(printf( "1) Image before plugging 2) image after plugging" );)

                if ( ( memblockp = find_addr_in_image( firmware, plugrecordp->addr ) ) != NULL ) {

                    //Calculate the offset of the data to be plugged in the image.
                    //We add 4 because of extra L- and T-field at start of data field in memblock
                    p = &memblockp->data[plugrecordp->addr - memblockp->addr + 4];
					DSFDBGEXT(dbg_display_hex( p, plugrecordp->len / 2 );)
                    memcpy( p, pdap+2, plugrecordp->len );
					DSFDBGEXT(dbg_display_hex( p, plugrecordp->len / 2 );)
                }
            }
        } else if ( plugrc == DHF_FAILURE ) {
			DSFASSERT(DO_DSFASSERT, TEXT("Error in plugging"), (hcf_16) code) ;
            rc = DHF_ERR_PLUG_PRODUCTION_DATA;
        }
        plugrecordp++;
    }
    return rc;
}   /* plug_production_data */
#endif // #ifndef DSF_HERMESII


/*-----------------------------------------------------------------------------
 *
 * Function:    primary_download_needed
 *
 * Abstract:    Checks if download of primary firmware is needed
 *
 * Returns:
 *   TRUE - download of primary needed
 *   FALSE - no download needed
 *
 * Description:
 *   The routines applies all rules needed, comparing the firmware in the NIC
 *   and in the image provided, to determine whether downloading the primary
 *   firmware is warranted.
 *
 * Notes:
 * - The rules for determining whether primary download is needed can be found
 *   in ???.
 * - Primary firmware is downloaded silently, i.e. without asking the user
 *   for confirmation.
 *
 *---------------------------------------------------------------------------*/
static BOOL primary_download_needed(memimage *primary) 
{

    BOOL rc = FALSE;

//TODO: provide some mechanism to set global variable dhf_always_download_primary
    DSFDBGEXT(dhf_always_download_primary = TRUE;)
    
    // In Major version 4 of the primary firmware functions of Hermes and Shark
    // were combined. Prior to that two seperate versions existed. We only have
    // to download primary firmware if major version of primary firmware in the
    // NIC < 4.
    if ( ( dhf_always_download_primary == TRUE ) || 
         ( pri_identity.version_major < 4 ) ) {
        rc = TRUE;
    }

    // This is a debug (and not a release) display as primary firmware is supposed
    // to be updated silently, i.e. without letting the user know.
    DSFDBG(if (rc == TRUE ) { dsf_debug_message(TEXT("Update of primary firmware needed."), DHF_INFO, FALSE); })

    return rc;
}   /* primary_download_needed */                                 


/*-----------------------------------------------------------------------------
 *
 * Exported functions
 *
 *---------------------------------------------------------------------------*/


/*-----------------------------------------------------------------------------
 * 
 * Function: dhf_download_firmware
 *
 * Abstract: Downloads a complete (primary, station, or access point) firmware 
 *           image to the NIC.
 *
 * Returns:
 *   DHF_SUCCESS             - download completed successfully.
 *   DHF_ERR_NO_NIC          - no NIC present in slot
 *   DHF_ERR_NOT_COMPATIBLE  - firmware not compatible
 *   DHF_NOT_NEWER           - firmware OK but not newer than firmware in NIC
 *   DHF_ERR_DOWNLOAD        - some other problem occurred in download (condensed)
 *
 *   when using the WCI:
 *
 *   when using the UIL:
 *   DHF_ERR_NO_DRV               - no driver present
 *   DHF_ERR_IN_USE               - the UIL-driver connection is already used by 
 *                                  another utility
 * 
 * Description:
 *   Initialize global variables
 *   Connect to the DHF
 *   Check the compatibility of the image (For primary firmware images it is checked first 
 *     whether download is necessary.)
 *   If everything's download the firmware. 
 *   Disconnect from the DHF.
 *
 *   void *ifbp,                        pointer to the ifb; ignored when using the UIL
 *   CFG_RANGE20_STRCT *app_dui_actor   describes DUI-actor range of the download application;
 *                                      ignored when using the WCI
 *   CFG_RANGE20_STRCT *app_pri_actor   describes primary-actor range of the download application
 *   memimage *firmware                 station image to be downloaded
 *   BOOL to_volatile                   TRUE - download to volatile NIC-RAM
 *                                      FALSE - download to non-volatile NIC-RAM
 *   					    			implied TRUE if DSF_VOLATILE_ONLY is defined
 *   BOOL confirm                       TRUE=ask user for confirmation
 *                                      FALSE=do not ask 
 *---------------------------------------------------------------------------*/
int dhf_download_firmware(void *ifbp, CFG_RANGE20_STRCT *app_dui_actor, CFG_RANGE20_STRCT *app_pri_actor, memimage *firmware, BOOL to_volatile, BOOL confirm) 
{
    int rc = DHF_SUCCESS;
    BOOL download = TRUE;
    int ft;
    dhf_last_error = DHF_SUCCESS;

#ifdef DHF_WCI
	DSFASSERT( ifbp != NULL, TEXT("ifbp == NULL"), 0 ) ;
#endif
#ifdef DHF_UIL
	DSFASSERT( app_dui_actor != NULL, TEXT("app_dui_actor == NULL") , 0 ) ;
#endif
	DSFASSERT( app_pri_actor != NULL, TEXT("app_pri_actor == NULL") , 0 ) ;
	DSFASSERT( firmware != NULL, TEXT("firmware == NULL"), 0) ;

    if ( firmware != NULL ) {
        // Initialize global variables
        dhf_firmware = firmware;
#ifndef DSF_VOLATILE_ONLY
        dhf_to_volatile = to_volatile;
        dhf_block_nr = 0;
        dhf_progress_step = 0;
#endif
        dhf_memblockp = NULL;
        dhf_len = 0;
        dhf_index = 0;

// At this point, the Primary firmware in the Hermes-II may be active. The primary FW image does not support
// hcf_get_info().
        rc = doconnect( ifbp, app_dui_actor, app_pri_actor, confirm );
		DSFASSERT(rc == DHF_SUCCESS, TEXT("rc != DHF_SUCCESS"), rc) ;
    
        if(rc == DHF_SUCCESS ) {
            ft = determine_firmware_type( dhf_firmware );
            switch( ft ) {
                case FT_UNKNOWN:
					DSFASSERT(DO_DSFASSERT, TEXT("determine_firmware_type( dhf_firmware ) == FT_UNKNOWN"), determine_firmware_type( dhf_firmware )) ;
                    rc = DHF_ERR_NOT_COMPATIBLE;
                    break;
                case FT_PRIMARY:
                    // Download primary firmware if necessary and allowed. This is
                    // done silently (without telling the user) and only if the firmware
                    // in the download image is newer than the firmware in the card.

                    download = primary_download_needed( dhf_firmware );
                    if ( download == TRUE ) {
                        // Check if the primary firmware is compatible with the NIC.
                        // We do not check whether the primary firmware, if downloaded, 
                        // will be compatible with the new station firmware. It is the  
                        // responsibility of the WSU-developer to make sure the 2 images
                         // are compatible.
#ifndef DSF_HERMESII
                        rc = check_comp_primary( firmware );
#endif //DSF_HERMESII
						DSFASSERT(rc == DHF_SUCCESS, TEXT("check_comp_primary(firmware) = "), rc) ;
                    }
                    break;
                case FT_STATION:
#ifndef DSF_HERMESII
                    rc = check_comp_station( firmware );
#endif //DSF_HERMESII
					DSFASSERT(rc == DHF_SUCCESS, TEXT("check_comp_station(firmware) = "), rc) ;
                    break;
                case FT_AP:
#ifndef DSF_HERMESII
                    rc = check_comp_ap( firmware );
#endif //DSF_HERMESII
					DSFASSERT(rc == DHF_SUCCESS, TEXT("check_comp_ap(firmware) = "), rc) ;
                    break;
            }
        }
        
        // Download firmware if it is compatible with the NIC.
        if ( ( download == TRUE) && (rc == DHF_SUCCESS ) ) {
            rc = download_firmware_image(); 
        }
            
        // Always dodisconnect even in case of download failure, to make sure any
        // resources claimed are freed.
        dodisconnect();
    }
    
	DSFASSERT(rc == DHF_SUCCESS, TEXT("rc != DHF_SUCCESS"), rc) ;

#ifdef _WIN32_WCE
	if (rc == 0x100) return 0;
#endif
    return condense_error_code( rc, DHF_ERR_DOWNLOAD );
}   // dhf_download_firmware


#ifdef DHF_GET_RES_MSG

/*-----------------------------------------------------------------------------
 *
 * Function:    dhf_get_res_msg( int res )
 *
 * Abstract:    Get result message pertaining to result code argument
 *
 * Return:
 *   char* pointing to message
 *
 * Notes
 * - The result code includes both WCI/UIL and DHF-specific error/message codes.
 * - The result code includes both error codes and 'normal' message codes.
 * - DHF_ERR_PLUG_PRODUCTION_DATA and DHF_ERR_PLUG_PRIMARY_DATA are internal 
 *   error codes only. So no message is needed.
 *
 * Description:
 *---------------------------------------------------------------------------*/
char *dhf_get_res_msg(int res) 
{
    char *p;

    switch( res )  {
    // Errors
#ifndef DHF_WCI
    case UIL_ERR_IN_USE:
        p = DHF_ERROR_DUI_IN_USE;
        break;
    case UIL_ERR_NO_DRV:
        p = DHF_ERROR_NO_DRIVER;
        break;
#endif
    case DHF_ERR_DIAGNOSE:
        p = DHF_ERROR_DIAGNOSE;
        break;
    case UIL_ERR_NO_NIC:
        p = DHF_ERROR_NO_CARD;
        break;
    case DHF_ERR_DOWNLOAD:
        p = DHF_ERROR_DOWNLOAD;
        break;
    case DHF_ERR_NOT_COMPATIBLE:
        p = DHF_ERROR_NOT_COMPATIBLE;
        break;
    case DHF_ERR_CRD_APP_NOT_COMPATIBLE:
        p = DHF_ERROR_CRD_APP_NOT_COMPATIBLE;
        break;
    case DHF_ERR_DRV_APP_NOT_COMPATIBLE:
        p = DHF_ERROR_DRV_APP_NOT_COMPATIBLE;
        break;
    case DHF_ERR_CRD_PRI_NOT_COMPATIBLE:
        p = DHF_ERROR_CRD_PRI_NOT_COMPATIBLE;
        break;
    case DHF_ERR_PRI_DRV_NOT_COMPATIBLE:
        p = DHF_ERROR_PRI_DRV_NOT_COMPATIBLE;
        break;
    case DHF_ERR_PRI_APP_NOT_COMPATIBLE:
        p = DHF_ERROR_PRI_APP_NOT_COMPATIBLE;
        break;
    case DHF_ERR_CRD_STA_NOT_COMPATIBLE:
        p = DHF_ERROR_CRD_STA_NOT_COMPATIBLE;
        break;

    // Result messages
    case DHF_DOWNLOAD_COMPLETED:
        p = DHF_MSG_DOWNLOAD_COMPLETED;
        break;
    case DHF_USER_ABORT:
        p = DHF_MSG_ABORTED;
        break;
        
    // Codes without messages
    case DHF_SUCCESS:
    case DHF_NOT_NEWER:
    case DHF_IGNORE: 
        p = DHF_MSG_NULL;
        break;

    // General message for all other problems        
//!!!    case HCF_ERR_BUSY:
//!!!    case HCF_ERR_SEQ_BUG:
//!!!    case UIL_ERR_BUSY:
//!!!    case UIL_ERR_SEQ_BUG:
//!!!    case UIL_ERR_LEN:
//!!!    case UIL_ERR_PIF_CONFLICT:
//!!!    case UIL_ERR_TIME_OUT:
//!!!    case UIL_ERR_DOS_CALL:
//!!!    case UIL_ERR_NSTL:
//!!!    case UIL_ERR_WRONG_IFB:
//!!!    case DHF_FAILURE:
//!!!    case DHF_ERR_CONNECT:
//!!!    case DHF_ERR_GET_NIC_INFO:
//!!!    case DHF_ERR_ALLOC:
//!!!    case DHF_ERR_PLUG_PRODUCTION_DATA:
//!!!    case DHF_ERR_PLUG_PRIMARY_DATA:
    default:
		DSFASSERT( DO_DSFASSERT, TEXT("DSF_DOASSERT"), 0) ;
        p = DHF_ERROR_UPDATE;
        break;
    }
    return p;
} // dhf_get_res_msg

#endif


/*-----------------------------------------------------------------------------
 *
 * Function:    dhf_get_last_error
 *
 * Abstract:    Returns last error encountered by DHF.
 *
 * Returns:
 *   int
 *
 * Description:
 *   Returns the last error encountered by DHF.
 *   Sets error to zero.
 *
 * Notes:
 * - As this function resets the error to zero (DHF_SUCCESS), it function must 
 *   be called immediately after an error has been encountered. 
 *---------------------------------------------------------------------------*/
int dhf_get_last_error( void ) 
{
    int rc = dhf_last_error; 
    dhf_last_error = DHF_SUCCESS;
    return rc;
}   // dhf_get_last_error


/*-----------------------------------------------------------------------------
 *
 * Function: dhf_get_version_info
 *
 * Abstract: Gets version information of a component.
 *
 * Returns:
 *   CFG_IDENTITY_STRCT * containing version information on requested component.
 *   If comp_id is not recognized by DHF all fields in the structure is set to 0.
 * int comp_id,       component identifier, one of DHF_COMP_ID_??? constants  
 * memimage *firmware firmware image, this parameter is ignored if comp_id != DHF_COMP_ID_MEMIMAGE
 *---------------------------------------------------------------------------*/
EXTERN_C CFG_IDENTITY_STRCT *dhf_get_version_info(int comp_id, memimage *firmware) 
{
    static CFG_IDENTITY_STRCT  version_info = { 0, 0, 0, 0, 0, 0 };
    CFG_IDENTITY_STRCT *id = NULL;
    
	DSFASSERT( comp_id >= 0 && comp_id <= 99, TEXT("comp_id < 0 && comp_id > 99"), comp_id) ;
	DSFASSERT( firmware != NULL, TEXT("firmware == NULL"), 0 ) ;

	dhf_last_error = DHF_SUCCESS;

    switch( comp_id ) {    
    case DHF_COMP_ID_DRIVER:
        id = &drv_identity;
        break;
    case DHF_COMP_ID_NIC_STA_FW:
        id = &sta_identity;
        break;
    case DHF_COMP_ID_NIC_PRI_FW:
        id = &pri_identity;
        break;
    case DHF_COMP_ID_NIC:
        id = &nic_identity;
        break;
    case DHF_COMP_ID_MEMIMAGE:
        if ( firmware != NULL ) { id = firmware->identity; }
        break;
    }
    
    if ( id == NULL ) {
        memset( &version_info, 0, sizeof( version_info ) );
    } else {
//         memmove( &version_info, id, sizeof( version_info ) );
        memcpy( &version_info, id, sizeof( version_info ) );
    }

    return &version_info;
}   // dhf_get_version_info


/*-----------------------------------------------------------------------------
 *
 * Function:    dhf_load_pda
 *
 * Abstract:    Retrieves PDA from NIC. The information is stored in an internal
 *              buffer.
 *
 * Returns:
 *   DHF_SUCCESS        - information successfully plugged
 *   DHF_ERR_NO_NIC     - no NIC present in slot
 *   DHF_ERR_NO_DRV     - no driver present
 *
 * Description:
 *   Reads the PDA from the NIC. The information is stored internally in
 *   array cfg_prod_data, which is not accessible to the DHF-user. Indirect 
 *   access to cfg_prod_data is provided by routines dhf_read_pda_record and
 *   dhf_write_pda_record (both defined in unit DHFPDA*.C.) which can be used 
 *   to change the contents of cfg_prod_data. 
 *
 * Notes:
 * - Ths routine is part of a restricted interface dealing with the PDA. It is
 *   not available to the general public, i.e. it is not mentioned in file
 *   DHF.H.
 * - For more routines handling the PDA see file DHFPDA.C/H.
 *---------------------------------------------------------------------------*/
#ifndef DSF_HERMESII
int dhf_load_pda( void ) 
{
    int rc = DHF_SUCCESS;
    LTVP pda_ltv;

    dhf_last_error = DHF_SUCCESS;

#ifdef DSF_ALLOC
    pda_ltv = (LTVP)cfg_prod_data;
#else
    pda_ltv = (LTVP) &cfg_prod_data[0];
#endif

	/*NATIE (help with corrupted PDA)*/
	memset (pda_ltv->val, 0, (PDA_SIZE-2)*sizeof(hcf_16));

    rc = GET_INFO( pda_ltv );
    
	DSFASSERT(rc == DHF_SUCCESS, TEXT("Error: dhf_load_pda"), rc) ;
    return condense_error_code( rc, DHF_ERR_GET_NIC_INFO );
}   // dhf_load_pda

#endif // #ifndef DSF_HERMESII


#ifdef DSF_BINARY_FILE

/*-----------------------------------------------------------------------------
 *
 * Function: dhf_binary_size
 *
 * Abstract: Gets the size of the binary version of a firmware image.
 *
 * Returns: The size of the image if no error else 0xFFFFFFFF
 *   
 *---------------------------------------------------------------------------*/
#ifndef NDIS_MINIPORT_DRIVER
ULONG dhf_binary_size(TCHAR *szImageName)
{
	HANDLE	hFile ;
	ULONG	dwFP ;

    hFile = CreateFile(szImageName, GENERIC_READ, 0, (LPSECURITY_ATTRIBUTES) NULL, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, (HANDLE) NULL); 	

	if(hFile == INVALID_HANDLE_VALUE) {
		OUTPUTDEBUGMSG((TEXT("\nError in Opening file %s : %ld\n"), szImageName, hFile)) ;
		return(0xFFFFFFFF) ;
	}

	dwFP = GetFileSize(hFile, NULL) ;
	CloseHandle(hFile) ;

	return(dwFP) ;
}
#endif


/*-----------------------------------------------------------------------------
 *
 * Function: dhf_load_binary
 *
 * Abstract: Loads the external file with the binary Firmware.
 *
 * Returns: memimage * containing a pointer to the image to download
 *          if there is a error this function returns a NULL
 *---------------------------------------------------------------------------*/
#ifdef NDIS_MINIPORT_DRIVER
memimage *dhf_load_binary(void *pMem, ULONG dwFP)
#else
memimage *dhf_load_binary(TCHAR *szImageName, void *pMem)
#endif
{
#ifndef NDIS_MINIPORT_DRIVER
	HANDLE		hFile ;
	ULONG		dwFP ;
	ULONG		dwRead ;
#endif	
	int			i ;
	ULONG		*dwCheck ;
	memimage	*pImage ;

	OUTPUTDEBUGMSG((TEXT("dhf_load_binary()\n"))) ;
#ifndef NDIS_MINIPORT_DRIVER
    hFile = CreateFile(szImageName, GENERIC_READ, 0, (LPSECURITY_ATTRIBUTES) NULL, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, (HANDLE) NULL); 	

	if(hFile == INVALID_HANDLE_VALUE) {
		OUTPUTDEBUGMSG((TEXT("Error in Opening file %s : 0x%x\n"), szImageName, GetLastError())) ;
		return(NULL) ;
	}

	dwFP = GetFileSize(hFile, NULL) ;

	if(NULL == pMem) {
		pMem = malloc(dwFP) ;
	}

	if(pMem == NULL) {
		printf("\nError in Alloccation Memory!\n") ;
		OUTPUTDEBUGMSG((TEXT("\nError in Alloccation Memory : 0x%x\n"), GetLastError())) ;
		CloseHandle(hFile) ;
		return(NULL) ;
	}

    if(FALSE == ReadFile(hFile, pMem, dwFP, &dwRead, (LPOVERLAPPED) NULL)) {  
		OUTPUTDEBUGMSG((TEXT("\nError in ReadFile %s : 0x%x\n"), szImageName, GetLastError())) ;
		CloseHandle(hFile) ;
		free(pMem) ;
		return(NULL) ;
	}
	CloseHandle(hFile) ;

#endif // #ifndef NDIS_MINIPORT_DRIVER

	
	// Recalc image pointers to real memory pointers
	pImage = (memimage *) ((ULONG) pMem + dwFP - sizeof(memimage)) ;

	// Check if biniry file is a FW file
	dwCheck = (ULONG *) pImage ;
	dwCheck -= 2 ;
	if(*dwCheck != BIN_FILE_CHECK) {
		OUTPUTDEBUGMSG((TEXT("\nError in BINFILE check: 0x%08lx\n"), *dwCheck)) ;
		return(NULL) ;
	}
	dwCheck++ ;
#ifdef DSF_HERMESII
	if(*dwCheck != BIN_FILE_HERMESII) {
#else
	if(*dwCheck != BIN_FILE_HERMESI) {
#endif
		OUTPUTDEBUGMSG((TEXT("\nError in BINFILE Type check: 0x%08lx\n"), *dwCheck)) ;
		return(NULL) ;
	}

	pImage->codep = (memblock *) ((ULONG) pMem + (ULONG) pImage->codep);
	pImage->identity = (CFG_IDENTITY_STRCT *)((ULONG) pMem + (ULONG) pImage->identity);
#ifndef DSF_HERMESII
	pImage->pdaplug  = (plugrecord *)        ((ULONG) pMem + (ULONG) pImage->pdaplug );
	pImage->priplug  = (plugrecord *)        ((ULONG) pMem + (ULONG) pImage->priplug) ;
#endif
	pImage->compat   = (CFG_RANGE20_STRCT *)  ((ULONG) pMem + (ULONG) pImage->compat) ;
	for(i = 0; pImage->codep[i].size; i++) {
		pImage->codep[i].data = (hcf_8 *)    ((ULONG) pMem + (ULONG) pImage->codep[i].data) ;
		// OUTPUTDEBUGMSG((TEXT("pImage->codep[%d].addr %08lX size %04X flags %04X"), i, pImage->codep[i].addr, pImage->codep[i].size, pImage->codep[i].flags)) ;
		// OUTPUTDEBUGMSG((TEXT("pImage->codep[%d].data = [4]%02X [5]%02X [6]%02X [7]%02X"), i, pImage->codep[i].data[4], pImage->codep[i].data[5], pImage->codep[i].data[6], pImage->codep[i].data[7])) ;
}
	OUTPUTDEBUGMSG((TEXT("BinImage Component:%d Variant %d Version %d.%02d"), pImage->identity->comp_id, pImage->identity->variant, pImage->identity->version_major, pImage->identity->version_minor)) ;
	



	return(pImage) ;
}

#endif // DSF_BINARY_FILE
